package org.lsst.ccs.localdb.statusdb;

import java.util.Properties;
import java.util.concurrent.TimeUnit;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.hibernate.SessionFactory;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusAlert;
import org.lsst.ccs.bus.messages.StatusData;
import org.lsst.ccs.localdb.statusdb.utils.CacheStatistics;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.messaging.BusMessageFilterFactory;

/**
 * Subsystem wrapper around a StatusPersister object
 */
public class StatusPersisterSubsystem extends Subsystem {

    private final StatusPersister dataStatusBatchPersister ;
    
    private final AlertPersister alertBatchPersister;
    
    private static SessionFactory fac;
    
    
    public static synchronized void init(Properties p) {
        fac = StatusdbUtils.getSessionFactory(p);
    }
    
    public StatusPersisterSubsystem(Properties p) {
        super("localdb", AgentInfo.AgentType.SERVICE);

        if (p == null) {
            p = new Properties();
        }
        init(p);
        
        dataStatusBatchPersister = new StatusPersister(fac);
        
        alertBatchPersister = new AlertPersister(fac);
    }
    
    @Override
    public void doStart() {
        getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(alertBatchPersister);
        getMessagingAccess().addStatusMessageListener(dataStatusBatchPersister,BusMessageFilterFactory.messageClass(StatusData.class).and(BusMessageFilterFactory.messageOrigin(null)));
        getMessagingAccess().addStatusMessageListener(alertBatchPersister,BusMessageFilterFactory.messageClass(StatusAlert.class).and(BusMessageFilterFactory.messageOrigin(null)));
        
        getScheduler().scheduleWithFixedDelay(dataStatusBatchPersister, 0, 1, TimeUnit.SECONDS);
        getScheduler().scheduleWithFixedDelay(alertBatchPersister, 0, 1, TimeUnit.SECONDS);
        getScheduler().scheduleAtFixedRate(
                () -> publishSubsystemDataOnStatusBus(new KeyValueData("statistics", new CacheStatistics(fac)))
                , 1, 5, TimeUnit.MINUTES);
    }

    public static void main(String[] args) {
        Properties p = new Properties();
        parseArgs(args, p);
        StatusPersisterSubsystem system = new StatusPersisterSubsystem(p);
        system.start() ;
    }
    
    
    private static void parseArgs(String[] args, Properties p) {
        // define the command line options
        Options commandLineOptions = new Options();
        // The help option
        commandLineOptions.addOption("h", "help", false, "Print the help message");
        // The name of the description file.
        commandLineOptions.addOption("u", "update", false,
                "Start the Trending Ingest Module in update mode.");

        CommandLineParser parser = new BasicParser();
        try {
            CommandLine line = parser.parse(commandLineOptions, args, false);
            
            if (line.hasOption("help")) {
                usage(commandLineOptions);
            } else {
                if ( line.hasOption("update") ) {
                    p.setProperty("hibernate.hbm2ddl.auto", "update");
                }
            }

        } catch (Exception e) {
            usage(commandLineOptions);
        }

    }

    static void usage(Options options) {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp(100, "TrendingIngestModule", "", options, "", true);
        System.exit(0);
    }
    
    
}
