package org.lsst.ccs.localdb.statusdb;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.hibernate.LockMode;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.localdb.dao.LocaldbFacade;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.DataPath;
import org.lsst.ccs.localdb.statusdb.model.MetaDataData;
import org.lsst.ccs.localdb.statusdb.model.PlotData;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * StatusPersister object. Implements StatusListener and updates the db.
 * <p/>
 * Can be run inside a StatusPersisterSubsystem wrapper.
 * <p/>
 * Deprecated use: inside a Message Driver Bean (tied to JMS implementation)
 *
 * @author aubourg
 */
public class StatusPersister extends BatchPersister implements StatusMessageListener {

    static Logger log = Logger.getLogger("org.lsst.ccs.localdb");
    private static final DataDesc NULL_DESCRIPTION = new DataDesc();
    private static final long[] DEFAULT_TIME_BIN_WIDTH = new long[]{300000, 1800000}; //5 and 30 minutes

    private final Map<DataPath, DataDesc> map = new ConcurrentHashMap<>();

    // TODO : force a re-read of the database or re-read the database periodically?
    // TODO : clean code and change method names. Too much history there...
    /**
     * The constructor initializes entries in StatDesc with regards of existing
     * entries in DataDesc. It also creates an internal map to analize data
     * description integrity with future entries.
     */
    @SuppressWarnings("unchecked")
    public StatusPersister(SessionFactory fac) {
        super(fac);
        log.fine("Starting StatusPersister");
        Session sess = fac.openSession();
        Transaction tx = sess.beginTransaction();
        List<DataDesc> l = sess.createQuery("from DataDesc").list();
        for (DataDesc dd : l) {
            map.put(dd.getDataPath(), dd);
            log.fine("storing " + dd.getDataPath().getFullKey());
            //Create default statistical binning for existing entries that don't have any.
            addDefaultStatDescsFor(dd, sess);
        }
        tx.commit();
        sess.close();

        //Check the integrity of the existing trending channel names.
        analizeDataMapIntegrity();

    }

    @Override
    public void onStatusMessage(StatusMessage s) {
        KeyValueDataList encodedData = (KeyValueDataList) s.getEncodedData();
        if (encodedData != null) {
            // Status data are persisted as key value pairs
            String source = s.getOriginAgentInfo().getName();
            for (KeyValueData d : encodedData) {
                KeyValueData.KeyValueDataType type = d.getType();
                if (type == KeyValueData.KeyValueDataType.KeyValueTrendingData) {
                    queueImmediateScalar(d.getTimestamp(), new DataPath(source, d.getKey()), d.getValue());
                } else if (type == KeyValueData.KeyValueDataType.KeyValueMetaData) {
                    String metaPath = d.getKey();
                    int lastIndex = metaPath.lastIndexOf('/');
                    String metaname = metaPath.substring(lastIndex + 1);
                    String key = metaPath.replace("/" + metaname, "");
                    queueImmediateMetaData(d.getTimestamp(), new DataPath(source, key), metaname, (String) d.getValue());
                } else if (type == KeyValueData.KeyValueDataType.KeyValuePlotData) {
                    PlotData plotData = new PlotData();
                    plotData.setTime(d.getTimestamp());
                    plotData.setPlotData((String) d.getValue());
                    addToQueue(new Object[]{plotData, new DataPath(source, d.getKey())});
                }
            }
        }
    }

    public void queueImmediateMetaData(long tStamp, DataPath name, String metadata, String value) {
        MetaDataData md = new MetaDataData();
        md.setName(metadata);
        md.setValue(value);
        md.setStartTime(tStamp);
        addToQueue(new Object[]{md, name});
    }

    public void queueImmediateScalar(long tStamp, DataPath name, Object d) {

        log.debug("got update " + name);

        RawData data = new RawData();
        data.setTime(tStamp);
        if (d instanceof Double) {
            data.setDoubleData((Double) d);
        } else if (d instanceof Float) {
            data.setDoubleData(((Float) d).doubleValue());
        } else if (d instanceof Integer) {
            data.setDoubleData(((Integer) d).doubleValue());
        } else if (d instanceof Short) {
            data.setDoubleData(((Short) d).doubleValue());
        } else if (d instanceof Long) {
            data.setDoubleData(((Long) d).doubleValue());
        } else {
            //TODO: if array do an Array.toString
            data.setStringData(String.valueOf(d));
        }
        addToQueue(new Object[]{data, name});
    }

    private void persistData(RawData data, DataPath name, Session sess) {

        DataDesc dd = getDataDescription(name, "trending", sess);
        if (dd == null || dd == NULL_DESCRIPTION) {
            return;
        }

        data.setDataDesc(dd);
        sess.lock(dd, LockMode.NONE);
        sess.persist(data);

        List<StatDesc> stats = dd.getDerived();

        // TODO use named queries
        Query q = sess.createQuery("from StatData s where s.statDesc = :d "
                + "and s.statTimeInterval.binWidth=:binWidth and s.statTimeInterval.startTime<=:t2 and s.statTimeInterval.startTime>:t1");
        q.setLockMode("s", LockMode.UPGRADE);

        for (StatDesc stat : stats) {
            if (data.getDoubleData() == null) {
                continue;
            }

            //Statistical bin width
            long binWidth = stat.getTimeBinWidth();
            long dataTime = data.getTime();

            q.setEntity("d", stat);
            q.setLong("binWidth", binWidth);
            q.setLong("t2", dataTime);
            q.setLong("t1", dataTime-binWidth);
            StatData sd = (StatData) q.uniqueResult();

            if (sd == null) {
                sd = new StatData(stat, data, LocaldbFacade.getStatTimeInterval(binWidth, dataTime, sess));
                sess.persist(sd);
            } else {
                sd.accumulate(data);
            }
        }
    }

    private void persistMetadata(MetaDataData md, DataPath name, Session sess) {

        DataDesc dd = getDataDescription(name, "trending", sess);
        if (dd == null || dd == NULL_DESCRIPTION) {
            return;
        }

        sess.lock(dd, LockMode.NONE);
        md.setDataDesc(dd);

        // First update the tstop of a previously existing record for this
        // metadata
        Query q = sess.createQuery("from MetaDataData md "
                + "where dataDesc.id = :id "
                + "and name = :n and endTime <= 0");
        q.setParameter("id", dd.getId());
        q.setParameter("n", md.getName());
        MetaDataData oldMetaData = (MetaDataData) q.uniqueResult();
        if (oldMetaData != null) {
            oldMetaData.setEndTime(md.getStartTime());
            sess.update(oldMetaData);
        }
        // Now commit the new value of the metadata
        sess.persist(md);
    }

    private void persistPlotData(PlotData pd, DataPath name, Session sess) {
        DataDesc dd = getDataDescription(name, "plot", sess);
        if (dd == null || dd == NULL_DESCRIPTION) {
            return;
        }

        sess.lock(dd, LockMode.NONE);
        pd.setDataDesc(dd);
        sess.persist(pd);
    }

    private DataDesc getDataDescription(DataPath key, String type, Session sess) {
        Query q = sess.getNamedQuery("findDataDesc")
                .setString("agentName", key.getAgentName())
                .setString("dataName", key.getDataName());
        DataDesc dd = (DataDesc) q.uniqueResult();

        // TODO we should not do that in production.
        // this piece of code will make ALL data passing on the bus be
        // persisted by creating some default data description for it.
        // in preliminary version, datadesc should be populated by hand
        // TODO create GUI to allow user to select what should be persisted
        if (dd == null) {
            // in the map, we only use lower case, to be case independent.
            //Check if there is no conflict with existing data https://jira.slac.stanford.edu/browse/LSSTCCS-732
            for (DataPath existingKey : map.keySet()) {
                if (!areDataNamesConsistent(existingKey.getFullKey(), key.getFullKey())) {
                    if (map.get(existingKey) == NULL_DESCRIPTION) {
                        continue;
                    }
                    log.warning("Cannot add data description for " + key + " as it's in conflict with " + existingKey);
                    map.put(key, NULL_DESCRIPTION);
                    return NULL_DESCRIPTION;
                }
            }
            dd = new DataDesc();
            dd.setDataPath(key);
            dd.setDataType(type);
            sess.persist(dd);
            log.debug("Adding default Data Description for " + key + ": " + dd.getId());
            map.put(key, dd);
            addDefaultStatDescsFor(dd, sess);
        }
        return dd;
    }

    private void addDefaultStatDescsFor(DataDesc dd, Session sess) {
        List<StatDesc> stats = dd.getDerived();
        if (stats.isEmpty()) {
            for (long statInterval : DEFAULT_TIME_BIN_WIDTH) {
                StatDesc defaultSD = new StatDesc();
                defaultSD.setDataDesc(dd);
                defaultSD.setTimeBinWidth(statInterval);
                sess.persist(defaultSD);
                stats.add(defaultSD);
                log.fine("Adding default statistical binning of " + statInterval + " milliseconds to " + dd.getDataPath());
            }
        }
    }

    /**
     * Check that the current data, loaded from the database is internally consistent
     * according to the specifications provided at
     * https://jira.slac.stanford.edu/browse/LSSTCCS-732.
     * If it isn't it will print out a warning message for the offending channels.
     * The data will not be touched.
     *
     */
    private void analizeDataMapIntegrity() {
        for (DataPath existingKey1 : map.keySet()) {
            for (DataPath existingKey2 : map.keySet()) {
                if (existingKey1.equals(existingKey2)) {
                    continue;
                }
                if (!areDataNamesConsistent(existingKey1.getFullKey(), existingKey2.getFullKey())) {
                    log.warning("Conflict for existing keys: " + existingKey1 + " and " + existingKey2);
                }
            }
        }
    }

    /**
     * Check if two data names are consistent according to
     * https://jira.slac.stanford.edu/browse/LSSTCCS-732.
     * We check that neither names starts with the other and that they have
     * different levels of directories. This last bit is so that we accept
     * the situation where the leaf name differs by only the ending:
     * a.b.c and a.b.ca are consistent.
     * a.b and a.bc.d are consistent.
     * a.b and a.b.c are not.
     *
     * @param name1 the first data name
     * @param name2 the second data name
     * @return true/false
     */
    static boolean areDataNamesConsistent(String name1, String name2) {
        if (name1.equals(name2)) {
            return true;
        }

        String[] array1 = name1.split("/");
        String[] array2 = name2.split("/");

        for (int i = 0; i < Math.min(array1.length, array2.length); i++) {
            if (!array1[i].equals(array2[i])) {
                return true;
            }
        }
        return array1.length == array2.length;
    }

    @Override
    public void persist(Object[] obj, Session sess) {
        Object toPersist = obj[0];
        DataPath name = (DataPath) obj[1];
        if (toPersist instanceof RawData) {
            persistData((RawData) toPersist, name, sess);
        } else if (toPersist instanceof MetaDataData) {
            persistMetadata((MetaDataData) toPersist, name, sess);
        } else if (toPersist instanceof PlotData) {
            persistPlotData((PlotData) toPersist, name, sess);
        } else {
            if (toPersist != null) {
                sess.persist(toPersist);
            }
        }
    }

}
