package org.lsst.ccs.localdb.statusdb;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.messages.StatusAlert;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.messages.StatusRuntimeInfo;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.localdb.dao.LocaldbFacade;
import static org.lsst.ccs.localdb.statusdb.StatusPersister.log;
import org.lsst.ccs.localdb.statusdb.model.AgentDesc;
import org.lsst.ccs.localdb.statusdb.model.AlertData;
import org.lsst.ccs.localdb.statusdb.model.AlertDesc;
import org.lsst.ccs.localdb.statusdb.model.ClearedAlertData;
import org.lsst.ccs.localdb.statusdb.model.RaisedAlertData;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;

/**
 *
 * @author LSST CCS Team
 */
public class AlertPersister extends BatchPersister implements StatusMessageListener, AgentPresenceListener {
    
    private final SessionFactory fac;
    private final List<String> agentsToCheck = new CopyOnWriteArrayList<>();
    
    public AlertPersister(SessionFactory fac) {
        super(fac);
        
        log.fine("Starting AlertPersister");
        
        this.fac = fac;
        
        Session sess = this.fac.openSession();
        
        sess.close();
    }
    
    @Override
    public void onStatusMessage(StatusMessage msg) {

        if ( agentsToCheck.size() > 0 && msg instanceof StatusRuntimeInfo  && 
                agentsToCheck.contains(msg.getOriginAgentInfo().getName())) {
            pushActiveAlerts(msg.getOriginAgentInfo().getName(), System.currentTimeMillis()-((StatusRuntimeInfo)msg).getRuntimeInfo().getUptime()*1000);
        }
        // Safe cast
        StatusAlert statusAlert = (StatusAlert)msg;
        RaisedAlertSummary alertSummary = statusAlert.getRaisedAlertSummary();
        // Case 1 : received a StatusRaisedAlert
        if (statusAlert instanceof StatusRaisedAlert) {
            String alertId = ((Alert)statusAlert.getEncodedData()).getAlertId();
            String alertDescription = ((StatusRaisedAlert)statusAlert).getCause();
            AlertState currentSeverity = alertSummary.getRaisedAlert(alertId).getLatestAlertState();
            RaisedAlertData arad = new RaisedAlertData(currentSeverity, alertDescription, msg.getTimeStamp());
            addToQueue(new Object[]{arad, statusAlert, alertId});
        } else if (statusAlert instanceof StatusClearedAlert) {
            // Case 2 : received a StatusClearedAlert
            StatusClearedAlert sca = (StatusClearedAlert)statusAlert;
            String[] clearedAlerts = sca.getClearAlertIds();
            if (clearedAlerts != null) {
                for (String clearedId : clearedAlerts) {
                    ClearedAlertData cad = new ClearedAlertData();
                    cad.setTime(msg.getTimeStamp());
                    addToQueue(new Object[]{cad, statusAlert,clearedId});
                }
            }
        }
    }
    
    
    
    /** AgentPresenceListener callback methods.
     * When a agent connects, its alert entries in ActiveRaisedAlertData are removed.
     * @param agent
     */
    @Override
    public void connecting(AgentInfo agent) {
        if ( agent.getType().compareTo(AgentType.WORKER) >= 0 ) {
            agentsToCheck.add(agent.getName());
        }
    }
    
    /**
     * The active alerts for the disconnecting agent are pushed to history.
     * @param agent 
     */
    @Override
    public void disconnecting(AgentInfo agent) {
        if ( agent.getType().compareTo(AgentType.WORKER) >= 0 ) {
            pushActiveAlerts(agent.getName(), System.currentTimeMillis());
        }
    }
    
    private void pushActiveAlerts(String agentName, long agentStartTime ) {
        agentsToCheck.remove(agentName);
        Session sess = fac.openSession();
        Transaction tx = sess.beginTransaction();
        log.info("updating active alerts for agent : " + agentName);
        Query q = sess.createQuery("from RaisedAlertData rad where rad.agentDesc.agentName=:name and rad.active=true and rad.time<=:t");
        q.setString("name", agentName);
        q.setLong("t", agentStartTime);
        
        List<RaisedAlertData> res = q.list();
        for (RaisedAlertData rad : res) {
            rad.setActive(false);
            sess.update(rad);
        }
        tx.commit();
        sess.close();
    }
    
    
    @Override
    public void persist(Object[] obj, Session sess) {
        // The object to be persisted
        AlertData alData = (AlertData)obj[0];

        // The original status
        StatusAlert statusAlert = (StatusAlert)obj[1];
        
        // The alertId associated to
        String alertId = (String)obj[2];
        
        AgentInfo ai = statusAlert.getOriginAgentInfo();
        StateBundle state = statusAlert.getState();
        
        // Fetching agent description from local cache or create it and persist it.
        AgentDesc agDesc = LocaldbFacade.getAgentDesc(ai, sess);
        alData.setAgentDesc(agDesc);
        
        // State reconstruction.
        alData.setAgentState(LocaldbFacade.getAgentState(ai, state, sess));
        
        // Fetching alert description from local cache or create and persist it.
        if (statusAlert instanceof StatusRaisedAlert) {
            Alert al = ((StatusRaisedAlert)statusAlert).getRaisedAlert();
            AlertDesc ad = LocaldbFacade.getAlertDescOrPersist(agDesc, al, sess);
            alData.setAlertDesc(ad);
            sess.persist(alData);

        } else if (statusAlert instanceof StatusClearedAlert) {
            AlertDesc ad = LocaldbFacade.getAlertDesc(agDesc, alertId, sess); 
            if (ad != null) { 
                alData.setAlertDesc(ad); 
                sess.persist(alData); 
            } else { 
                log.warn("could not find description for alert with id " + alertId); 
            }
            // Cleared flag is updated on the instances of the active cleared alert.
            Query q = sess.createQuery("from RaisedAlertData rad where"
                    + " rad.agentDesc.agentName=:name and rad.alertDesc.alertId =:id and rad.active=true")
                    .setString("id", alertId).setString("name", agDesc.getAgentName());
            List<RaisedAlertData> res = q.list();
            for (RaisedAlertData arad : res) {
                arad.setClearingAlert((ClearedAlertData)alData);
                sess.update(arad);
            }
            log.fine("cleared " + res.size() + " instances of " + agDesc.getAgentName()+"/"+alertId);
        }
    }
    
    
    
}
