package org.lsst.ccs.localdb.statusdb;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.hibernate.LockMode;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.CommandMessage;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.localdb.statusdb.model.CommandMessageData;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.DataMetaData;
import org.lsst.ccs.localdb.statusdb.model.LogMessageData;
import org.lsst.ccs.localdb.statusdb.model.PlotData;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.messaging.CommandMessageListener;
import org.lsst.ccs.messaging.LogMessageListener;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * StatusPersister object. Implements StatusListener and updates the db.
 * <p/>
 * Can be run inside a StatusPersisterSubsystem wrapper.
 * <p/>
 * Deprecated use: inside a Message Driver Bean (tied to JMS implementation)
 *
 * @author aubourg
 */
public class StatusPersister implements StatusMessageListener, LogMessageListener, CommandMessageListener {

    static Logger log = Logger.getLogger("org.lsst.ccs.localdb");
    private static SessionFactory fac;

    public static synchronized void init(Properties p) {
        fac = StatusdbUtils.getSessionFactory(p);
    }

    Map<String, DataDesc> map = new ConcurrentHashMap<>();
    protected DataWriter writer = new DataWriter();

    // TODO : force a re-read of the database
    // or re-read the database periodically?
    // TODO : clean code and change method names. Too much history there...
    @SuppressWarnings("unchecked")
    public StatusPersister(Properties p) {
        if (p == null) {
            p = new Properties();
        }
        init(p);
        log.fine("Starting StatusPersister");
        Session sess = fac.openSession();
        List<DataDesc> l = sess.createQuery("from DataDesc").list();
        for (DataDesc dd : l) {
            String key = dd.getSrcSubsystem() + "/" + dd.getSrcName();
            map.put(key.toLowerCase(), dd);
            log.fine("storing " + key);
        }
        sess.close();

        // TODO  use executor or scheduler?
        Thread writerThread = new Thread(writer);
        writerThread.setDaemon(true);
        writerThread.start();

        // TODO stop cleanly the writer thread?
    }

    // TODO handle overflows, use blocking queue?
    protected Queue<Object[]> rq = new ConcurrentLinkedQueue<>();

    public class DataWriter implements Runnable {

        @Override
        public void run() {
            ArrayList<Object[]> workingList = new ArrayList<>(1000);
            long sleepTime = 1000L;

            while (true) {

                // get at most 1000 objets from the queue
                for (int i = 0; i < 1000; i++) {
                    Object[] d = rq.poll();
                    if (d == null) {
                        break;
                    }
                    workingList.add(d);
                }

                // see if queue is empty, time since last run, and sleep
                // accordingly
                // else persist
                if (workingList.isEmpty()) {
                    try {
                        Thread.sleep(sleepTime);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                } else {
                    Session sess = fac.openSession();
                    Transaction tx = sess.beginTransaction();
                    try {
                        for (Object[] o : workingList) {
                            Object toPersist = o[0];
                            String name = (String) o[1];
                            if (toPersist instanceof RawData) {
                                persistData((RawData) toPersist, name, sess);
                            } else if (toPersist instanceof DataMetaData) {
                                persistMetadata((DataMetaData) toPersist, name, sess);
                            } else if (toPersist instanceof PlotData) {
                                persistPlotData((PlotData)toPersist, name, sess);
                            } else {
                                sess.persist(toPersist);
                            }
                        }
                    } catch (Exception e) {
                        log.error(e.toString());
                    }
                    tx.commit();
                    sess.close();
                    workingList.clear();
                }
            }

        }
    }

    @Override
    public void onStatusMessage(StatusMessage s) {

        KeyValueDataList encodedData = (KeyValueDataList) s.getEncodedData();
        String source = s.getOriginAgentInfo().getName();
        if (encodedData != null) {
            for (KeyValueData d : encodedData) {
                KeyValueData.KeyValueDataType type = d.getType();
                String fullKey = source+"/"+d.getKey();

                if (type == KeyValueData.KeyValueDataType.KeyValueTrendingData) {
                    queueImmediateScalar(d.getTimestamp(), fullKey, d.getValue());
                } else if (type == KeyValueData.KeyValueDataType.KeyValueMetaData) {
                    int lastIndex = fullKey.lastIndexOf("/");
                    String metaname = fullKey.substring(lastIndex + 1);
                    String name = fullKey.replace("/" + metaname, "");
                    queueImmediateMetaData(d.getTimestamp(), name, metaname, (String) d.getValue());
                } else if (type == KeyValueData.KeyValueDataType.KeyValuePlotData) {
                    PlotData plotData = new PlotData();
                    plotData.setTstamp(d.getTimestamp());
                    plotData.setPlotData((String)d.getValue());
                    addToQueue(plotData,fullKey);
                }
            }
        }
    }

    @Override
    public void onCommandMessage(CommandMessage msg) {
        /*
        CommandMessageData cmd = new CommandMessageData(msg);
        addToQueue(cmd, null);
        */
    }

    @Override
    public void onLogMessage(LogMessage msg) {
        /*
        LogMessageData lmd = new LogMessageData(msg);
        addToQueue(lmd,null);
                */
    }

    

    public void queueImmediateMetaData(long tStamp, String name, String metadata, String value) {
        DataMetaData md = new DataMetaData();
        md.setName(metadata);
        md.setValue(value);
        md.setTstart(tStamp);
        addToQueue(md, name);
    }

    public void queueImmediateScalar(long tStamp, String name, Object d) {

        log.debug("got update " + name);

        RawData data = new RawData();
        data.setTstamp(tStamp);
        if (d instanceof Double) {
            data.setDoubleData((Double) d);
        } else if (d instanceof Float) {
            data.setDoubleData(((Float) d).doubleValue());
        } else if (d instanceof Integer) {
            data.setDoubleData(((Integer) d).doubleValue());
        } else if (d instanceof Short) {
            data.setDoubleData(((Short) d).doubleValue());
        } else if (d instanceof Long) {
            data.setDoubleData(((Long) d).doubleValue());
        } else {
            //TODO: if array do an Array.toString
            data.setStringData(String.valueOf(d));
        }
        addToQueue(data, name);
    }

    public void addToQueue(Object data, String name) {
        rq.add(new Object[]{data, name});
    }

    private void persistData(RawData data, String name, Session sess) {

        DataDesc dd = getDataDescription(name, "trending", sess);
        if (dd == null) {
            return;
        }

        data.setDescr(dd);
        sess.lock(dd, LockMode.NONE);
        sess.persist(data);

        List<StatDesc> stats = dd.getDerived();
        // TODO use named queries
        Query q = sess.createQuery("from StatData s where s.descr = :d "
                + "order by s.tstampFirst desc");
        q.setLockMode("s", LockMode.UPGRADE);
        for (StatDesc stat : stats) {
            q.setEntity("d", stat);
            q.setMaxResults(1);
            StatData sd = (StatData) q.uniqueResult();
            if (sd == null) {
                sd = new StatData(stat, data);
                sess.persist(sd);
            } else if (data.getTstamp() > sd.getTstampFirst()
                    + stat.getTimeBinWidth()) {
                sd = new StatData(stat, data);
                sess.persist(sd);
            } else {
                sd.accumulate(data);
            }
        }
    }

    private void persistMetadata(DataMetaData md, String name, Session sess) {

        DataDesc dd = getDataDescription(name, "trending", sess);
        if (dd == null) {
            return;
        }

        sess.lock(dd, LockMode.NONE);
        md.setRawDescr(dd);

        // First update the tstop of a previously existing record for this
        // metadata
        Query q = sess.createQuery("from DataMetaData md "
                + "where rawDescr_id = :id "
                + "and name = :n and tstopmillis <= 0");
        q.setParameter("id", dd.getId());
        q.setParameter("n", md.getName());
        DataMetaData oldMetaData = (DataMetaData) q.uniqueResult();
        if (oldMetaData != null) {
            oldMetaData.setTstop(md.getTstart());
            sess.update(oldMetaData);
        }
        // Now commit the new value of the metadata
        sess.persist(md);
    }

    private void persistPlotData(PlotData pd, String name, Session sess) {
        DataDesc dd = getDataDescription(name, "plot", sess);
        if (dd == null) {
            return;
        }

        sess.lock(dd, LockMode.NONE);
        pd.setDescr(dd);
        sess.persist(pd);
    }

    private DataDesc getDataDescription(String key, String type, Session sess) {

        log.debug("Looking for data description in map " + map + " " + key);
        // in the map, we only use lower case, to be case independent.
        DataDesc dd = map.get(key.toLowerCase());

        // TODO we should not do that in production.
        // this piece of code will make ALL data passing on the bus be
        // persisted by creating some default data description for it.
        // in preliminary version, datadesc should be populated by hand
        // TODO create GUI to allow user to select what should be persisted
        if (dd == null) {
            log.debug("Adding default Data Description for " + key);
            dd = new DataDesc();
            dd.setDataType(type);
            dd.setName(key);
            int ind = key.indexOf("/");
            dd.setSrcName(key.substring(ind + 1));
            dd.setSrcSubsystem(key.substring(0, ind));
            sess.persist(dd);
            map.put(key.toLowerCase(), dd);
        }

        return dd;
    }

}
