package org.lsst.ccs.localdb.statusdb;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.hibernate.LockMode;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.lsst.ccs.bus.messages.EncodedDataStatus;
import org.lsst.ccs.bus.messages.KVList;
import org.lsst.ccs.bus.messages.KeyData;
import org.lsst.ccs.bus.messages.MetadataStatus;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.DataMetaData;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.messaging.KeyValueStatusListener;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * StatusPersister object. Implements StatusListener and updates the db.
 * <p/>
 * Can be run inside a StatusPersisterSubsystem wrapper.
 * <p/>
 * Deprecated use: inside a Message Driver Bean (tied to JMS implementation)
 * 
 * @author aubourg
 */
public class StatusPersister implements StatusMessageListener, KeyValueStatusListener {

    /*
     * public static AnnotationConfiguration ac = new AnnotationConfiguration();
     * public static Configuration cfg = ac.configure("hibernate-tm.cfg.xml");
     * public static SessionFactory fac = cfg.buildSessionFactory();
     */

    static {
//        Logger.configure();
    }
    static Logger log = Logger.getLogger("org.lsst.ccs.localdb");
    private static SessionFactory fac;

    public static synchronized void init(Properties p) {
        fac = StatusdbUtils.getSessionFactory(p);
    }

    Map<String, DataDesc> map = new ConcurrentHashMap<>();
    protected DataWriter writer = new DataWriter();

    // TODO : force a re-read of the database
    // or re-read the database periodically?
    
    // TODO : clean code and change method names. Too much history there...
    
    @SuppressWarnings("unchecked")
    public StatusPersister(Properties p) {
        if (p == null) {
            p = new Properties();
        }
        init(p);
        log.info("Starting StatusPersister");
        Session sess = fac.openSession();
        List<DataDesc> l = sess.createQuery("from DataDesc").list();
        for (DataDesc dd : l) {
            String key = dd.getSrcSubsystem() + "/" + dd.getSrcName();
            map.put(key.toLowerCase(), dd);
            log.info("storing " + key);
        }
        sess.close();

        // TODO  use executor or scheduler?
        
        Thread writerThread = new Thread(writer);
        writerThread.setDaemon(true);
        writerThread.start();

        // TODO stop cleanly the writer thread?
    }


    // TODO handle overflows, use blocking queue?
    protected Queue<Object[]> rq = new ConcurrentLinkedQueue<>();

    public class DataWriter implements Runnable {

        @Override
        public void run() {
            ArrayList<Object[]> workingList = new ArrayList<>(1000);
            long sleepTime = 1000L;
            long lastSleep = -1L; // last time we slept after doing real work

            while (true) {

                // get at most 1000 objets from the queue
                for (int i = 0; i < 1000; i++) {
                    Object[] d = rq.poll();
                    if (d == null) break;
                    workingList.add(d);
                }

                // see if queue is empty, time since last run, and sleep
                // accordingly
                // else persist
                if (workingList.isEmpty()) {
                    log.info("empty list, sleeping");
                    // TODO adjust sleep time dynamically
                    try {
                        Thread.sleep(sleepTime);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                } else {
                    log.info("batch persisting " + workingList.size()
                            + " objects");
                    Session sess = fac.openSession();
                    Transaction tx = sess.beginTransaction();
                    try {
                        for (Object[] o : workingList) {
                            if (o[0] instanceof RawData) {
                                persistData((RawData)o[0], (String)o[1], sess);
                            } else {
                                persistMetadataStatus((MetadataStatus)o[0],
                                                      sess);
                            }
                        }
                    } catch (Exception e) {
                        log.error(e.toString());
                    }
                    tx.commit();
                    sess.close();
                    workingList.clear();
                }
            }

        }
    }


    @Override
    public void onStatusMessage(StatusMessage s) {
        if (s instanceof  EncodedDataStatus) {
            queueEncodedData((EncodedDataStatus)s);
        } else if (s instanceof MetadataStatus) {
            queueMetadataStatus((MetadataStatus)s);
        } else {
            log.info("can't persist message " + s + " of class " + s.getClass());
        }
    }


    @Override
    public void onKeyValueStatusDecomposition(String source, long timeStamp,
                                              String key, Object value,
                                              int commonID) {
        //TODO: will this work with "well-known" types that are not precisely
        //  scalars?  arrays for instance
        queueImmediateScalar(timeStamp, source+'/'+key, value);
    }


    /**
     * persists A status with Encoded Data (list of keyValue pairs)
     * @param encodedDataStatus
     */
    @Deprecated //KeyValueStatusListener takes care of that see method below
    public void queueEncodedData(EncodedDataStatus encodedDataStatus) {
        //todo: get rid of this code
        for (EncodedDataStatus dataStatus : encodedDataStatus) {
            KVList list = dataStatus.getContent();
            for (KeyData keyData : list) {
                long timeStamp =  dataStatus.getDataTimestamp() ;
                List<KeyData> detailsList = keyData.getContentAsList();
                for(KeyData detaileddata : detailsList) {
                    String key = detaileddata.getKey() ;
                    Optional<Object> optional = detaileddata.getValue() ;
                    if(optional.isPresent()) {
                        queueImmediateScalar(timeStamp, key, optional.get());
                    }
                }
            }
        }
    }


    public void queueImmediateScalar(long tStamp, String name, Object d) {
 
        log.debug("got update " + name);

        RawData data = new RawData();
        data.setTstamp(tStamp);
        if (d instanceof Double) {
            data.setDoubleData((Double) d);
        } else if (d instanceof Float) {
            data.setDoubleData((Double) (double) (Float) d);
        } else if (d instanceof Integer) {
            data.setDoubleData((Double) (double) (Integer) d);
        } else if (d instanceof Short) {
            data.setDoubleData((Double) (double) (Short) d);
        } else {
		//TODO: if array do an Array.toString
            data.setStringData(String.valueOf(d));
        }
        addToQueue(data, name);
    }


    public void queueMetadataStatus(MetadataStatus mst) {
        addToQueue(mst, null);
    }


    public void addToQueue(Object data, String name) {
        rq.add(new Object[]{data, name});
    }


    private void persistData(RawData data, String name, Session sess) {

        DataDesc dd = getDataDescription(name, sess);
        if (dd == null) return;

        data.setDescr(dd);
        sess.lock(dd, LockMode.NONE);
        sess.persist(data);

        List<StatDesc> stats = dd.getDerived();
        // TODO use named queries
        Query q = sess.createQuery("from StatData s where s.descr = :d "
                                   + "order by s.tstampFirst desc");
        q.setLockMode("s", LockMode.UPGRADE);
        for (StatDesc stat : stats) {
            q.setEntity("d", stat);
            q.setMaxResults(1);
            StatData sd = (StatData) q.uniqueResult();
            if (sd == null) {
                sd = new StatData(stat, data);
                sess.persist(sd);
            } else if (data.getTstamp() > sd.getTstampFirst()
                    + stat.getTimeBinWidth()) {
                sd = new StatData(stat, data);
                sess.persist(sd);
            } else {
                sd.accumulate(data);
            }
        }
    }


    private void persistMetadataStatus(MetadataStatus mst, Session sess) {

        String dataName = mst.getOriginAgentInfo().getName()+ "/" + mst.getDataName();
        DataDesc dd = getDataDescription(dataName, sess);
        if (dd == null) return;

        sess.lock(dd, LockMode.NONE);

        DataMetaData metadata = new DataMetaData();
        metadata.setName(mst.getMetadataName());
        metadata.setRawDescr(dd);
        metadata.setValue(mst.getMetadataValue());
        metadata.setTstart(mst.getTimeStamp());

        // First update the tstop of a previously existing record for this
        // metadata
        Query q = sess.createQuery("from DataMetaData md "
                                   + "where rawDescr_id = :id "
                                   + "and name = :n and tstopmillis <= 0");
        q.setParameter("id", dd.getId());
        q.setParameter("n", metadata.getName());
        DataMetaData oldMetaData = (DataMetaData) q.uniqueResult();
        if (oldMetaData != null) {
            oldMetaData.setTstop(metadata.getTstart());
            sess.update(oldMetaData);
        }
        // Now commit the new value of the metadata
        sess.persist(metadata);
    }


    private DataDesc getDataDescription(String key, Session sess) {

        log.debug("Looking for data description in map " + map + " " + key);
        // in the map, we only use lower case, to be case independent.
        DataDesc dd = map.get(key.toLowerCase());

        // TODO we should not do that in production.
        // this piece of code will make ALL data passing on the bus be
        // persisted by creating some default data description for it.
        // in preliminary version, datadesc should be populated by hand
        // TODO create GUI to allow user to select what should be persisted

        if (dd == null) {
            log.debug("Adding default Data Description for " + key);
            dd = new DataDesc();
            dd.setDataType("a");
            dd.setName(key);
            int ind = key.indexOf("/");
            dd.setSrcName(key.substring(ind + 1));
            dd.setSrcSubsystem(key.substring(0, ind));
            sess.persist(dd);
            map.put(key.toLowerCase(), dd);
        }

        return dd;
    }

}
