package org.lsst.ccs.localdb.statusdb;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.hibernate.LockMode;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.lsst.ccs.bus.BusMessage;
import org.lsst.ccs.bus.DataValueNotification;
import org.lsst.ccs.bus.EncodedDataStatus;
import org.lsst.ccs.bus.KVList;
import org.lsst.ccs.bus.KeyData;
import org.lsst.ccs.bus.KeyValueStatusListener;
import org.lsst.ccs.bus.MetadataStatus;
import org.lsst.ccs.bus.ObjectNType;
import org.lsst.ccs.bus.StatusListener;
import org.lsst.ccs.bus.TrendingStatus;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.bus.trending.TrendingData;
import org.lsst.ccs.localdb.Trending;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.DataMetaData;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * StatusPersister object. Implements StatusListener and updates the db.
 * <p/>
 * Can be run inside a StatusPersisterSubsystem wrapper.
 * <p/>
 * Deprecated use: inside a Message Driver Bean (tied to JMS implementation)
 * 
 * @author aubourg
 */
public class StatusPersister implements StatusListener, KeyValueStatusListener {

    /*
     * public static AnnotationConfiguration ac = new AnnotationConfiguration();
     * public static Configuration cfg = ac.configure("hibernate-tm.cfg.xml");
     * public static SessionFactory fac = cfg.buildSessionFactory();
     */

    static {
//        Logger.configure();
    }
    static Logger log = Logger.getLogger("org.lsst.ccs.localdb");
    private static SessionFactory fac;

    public static synchronized void init(Properties p) {
        fac = StatusdbUtils.getSessionFactory(p);
    }

    Map<String, DataDesc> map = new ConcurrentHashMap<>();
    protected DataWriter writer = new DataWriter();

    // TODO : force a re-read of the database
    // or re-read the database periodically?
    
    // TODO : clean code and change method names. Too much history there...
    
    @SuppressWarnings("unchecked")
    public StatusPersister(Properties p) {
        if (p == null) {
            p = new Properties();
        }
        init(p);
        log.info("Starting StatusPersister");
        Session sess = fac.openSession();
        List<DataDesc> l = sess.createQuery("from DataDesc").list();
        for (DataDesc dd : l) {
            String key = dd.getSrcSubsystem() + "/" + dd.getSrcName();
            map.put(key, dd);
            log.info("storing " + key);
        }
        sess.close();

        // TODO  use executor or scheduler?
        
        Thread writerThread = new Thread(writer);
        writerThread.setDaemon(true);
        writerThread.start();

        // TODO stop cleanly the writer thread?
    }


    // TODO handle overflows, use blocking queue?
    protected Queue<Object[]> rq = new ConcurrentLinkedQueue<>();

    public class DataWriter implements Runnable {

        @Override
        public void run() {
            ArrayList<Object[]> workingList = new ArrayList<>(1000);
            long sleepTime = 1000L;
            long lastSleep = -1L; // last time we slept after doing real work

            while (true) {

                // get at most 1000 objets from the queue
                for (int i = 0; i < 1000; i++) {
                    Object[] d = rq.poll();
                    if (d == null) break;
                    workingList.add(d);
                }

                // see if queue is empty, time since last run, and sleep
                // accordingly
                // else persist
                if (workingList.isEmpty()) {
                    log.info("empty list, sleeping");
                    // TODO adjust sleep time dynamically
                    try {
                        Thread.sleep(sleepTime);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                } else {
                    log.info("batch persisting " + workingList.size()
                            + " objects");
                    Session sess = fac.openSession();
                    Transaction tx = sess.beginTransaction();
                    try {
                        for (Object[] o : workingList) {
                            if (o[0] instanceof RawData) {
                                persistData((RawData)o[0], (String)o[1], sess);
                            } else {
                                persistMetadataStatus((MetadataStatus)o[0],
                                                      sess);
                            }
                        }
                    } catch (Exception e) {
                        log.error(e.toString());
                    }
                    tx.commit();
                    sess.close();
                    workingList.clear();
                }
            }

        }
    }


    @Override
    public void onStatus(BusMessage s) {
        if (s instanceof  EncodedDataStatus) {
            queueEncodedData((EncodedDataStatus)s);
        } else if (s instanceof TrendingStatus) {
            queueTrendingStatus((TrendingStatus)s);
        } else if (s instanceof MetadataStatus) {
            queueMetadataStatus((MetadataStatus)s);
        } else {
            log.info("can't persist message " + s + " of class " + s.getClass());
        }
    }


    @Override
    public void onKeyValueStatusDecomposition(String source, long timeStamp,
                                              String key, Object value,
                                              int commonID) {
        //TODO: will this work with "well-known" types that are not precisely
        //  scalars?  arrays for instance
        queueImmediateScalar(timeStamp, source+'/'+key, value);
    }


    public void queueTrendingStatus(TrendingStatus pv) {
        queueData(pv.getOrigin(), pv.getTimeStamp(), pv.getValue());
    }


    /**
     * persists A status with Encoded Data (list of keyValue pairs)
     * @param encodedDataStatus
     */
    @Deprecated //KeyValueStatusListener takes care of that see method below
    public void queueEncodedData(EncodedDataStatus encodedDataStatus) {
        //todo: get rid of this code
        for (EncodedDataStatus dataStatus : encodedDataStatus) {
            KVList list = dataStatus.getContent();
            for (KeyData keyData : list) {
                long timeStamp =  dataStatus.getDataTimestamp() ;
                List<KeyData> detailsList = keyData.getContentAsList();
                for(KeyData detaileddata : detailsList) {
                    String key = detaileddata.getKey() ;
                    Optional<Object> optional = detaileddata.getValue() ;
                    if(optional.isPresent()) {
                        queueImmediateScalar(timeStamp, key, optional.get());
                    }
                }
            }
        }
    }


    // Trending data can be published in three ways
    // - a single ValueNotification object, i.e. name, Object, timestamp
    // - a list of ValueNotification objects for publishing a whole bunch of
    // data with different timestamps (like in the case of the shutter movement)
    // - to publish a lot of values with the same timestamp a structure is
    // published. All the member variables annotated with @Trending will be
    // added to the
    // trending database with the name of the variable.
    public void queueData(String origin, long ts, Object p) {
        if (p instanceof DataValueNotification[]) {
            DataValueNotification[] vv = (DataValueNotification[]) p;
            for (DataValueNotification v : vv) {
                queueData(origin, v.gettStamp(), v);
            }
        } else if (p instanceof List<?> && !((List<?>) p).isEmpty()
                && ((List<?>) p).get(0) instanceof DataValueNotification) {
            @SuppressWarnings("unchecked")
            List<DataValueNotification> vv = (List<DataValueNotification>) p;
            for (DataValueNotification v : vv) {
                queueData(origin, v.gettStamp(), v);
            }

        } else if (p instanceof DataValueNotification) {
            DataValueNotification dvNotification = (DataValueNotification) p;
            queueDataValueNotification(origin, dvNotification);
        } else if (p instanceof ValueNotification[]) {
            ValueNotification[] vv = (ValueNotification[]) p;
            for (ValueNotification v : vv) {
                queueData(origin, v.gettStamp(), v);
            }
        } else if (p instanceof List<?> && !((List<?>) p).isEmpty()
                && ((List<?>) p).get(0) instanceof ValueNotification) {
            @SuppressWarnings("unchecked")
            List<ValueNotification> vv = (List<ValueNotification>) p;
            for (ValueNotification v : vv) {
                queueData(origin, v.gettStamp(), v);
            }
        } else if (p instanceof ValueNotification) {
            ValueNotification dt = (ValueNotification) p;
            Object data = dt.getData();
            ts = dt.gettStamp();
            if (data instanceof Number || data instanceof String) {
                String key = origin + "/" + dt.getName();
                queueImmediateScalar(ts, key, data);
            } else {
                Class<? extends Object> dataClass = data.getClass();
                log.debug("will try to persist class " + dataClass);
                boolean saveAll = dataClass.isAnnotationPresent(Trending.class);
                Field[] fields = dataClass.getDeclaredFields();
                for (Field f : fields) {
                    if (saveAll || f.isAnnotationPresent(Trending.class)) {
                        try {
                            log.debug("persisting field " + f.getName());
                            Object d = f.get(data);
                            String key = origin + "/" + dt.getName() + "/"
                                    + f.getName();
                            queueImmediateScalar(ts, key, d);
                        } catch (IllegalArgumentException
                                  | IllegalAccessException ex) {
                            log.error("pb reading field", ex);
                        }
                    }

                }
            }
        } else {
            if (p instanceof List) {
                Object x = ((List) p).get(0);
                log.info("can't persist a list of " + x + " class " + x != null ? p
                        .getClass() : "null");
            } else {
                log.info("can't persist " + p + " class " + p.getClass());
            }
            // this is not worth an exception. Or is it?
            // or should we try to persist stuff not encapsulated in
            // ValueNotification?
        }
    }


    public void queueDataValueNotification(String origin,
                                           DataValueNotification dv) {
        if (dv.isOfWellKnownType()) {
            String name = origin + "/" + dv.getName();
            queueDescribedScalar(dv.gettStamp(), name, dv.getObjectNType());
        } else {
            Map<String, ObjectNType> mapDesc = dv.getTrendingMap();
            if (mapDesc != null) {
                for (Map.Entry<String, ObjectNType> entry : mapDesc.entrySet()) {
                    String name = origin + "/" + entry.getKey();
                    queueDescribedScalar(dv.gettStamp(), name, entry.getValue());
                }
            } else {
                Object o = dv.getData();
                queueObject(origin + "/" + dv.getName(), dv.gettStamp(), o);
            }
        }
    }


    // This saves an object, either all fields, or fields with @Trending
    // annotation.
    // the new @Trendable annotation should be handled via maps in crystallized
    // objects.
    // TODO we might want to save some objects as BLOBs ?
    private void queueObject(String name, long ts, Object data) {
        Class<? extends Object> dataClass = data.getClass();
        log.debug("will try to persist class " + dataClass);
        boolean saveAll = dataClass.isAnnotationPresent(Trending.class);
        Field[] fields = dataClass.getDeclaredFields();
        for (Field f : fields) {
            if (saveAll || f.isAnnotationPresent(Trending.class)
                    || f.isAnnotationPresent(TrendingData.class)) {
                try {
                    log.debug("persisting field " + f.getName());
                    Object d = f.get(data);
                    String key = name + "/" + f.getName();
                    queueImmediateScalar(ts, key, d);
                } catch (IllegalArgumentException | IllegalAccessException ex) {
                    log.error("pb reading field", ex);
                }
            }
        }
    }


    // TODO: this is copied from queueImmediateScalar!
    public void queueDescribedScalar(long tStamp, String name,
                                     ObjectNType descriptionAndValue) {
        log.debug("got update " + name);

        // BEGIN COPIED
        RawData data = new RawData();
        data.setTstamp(tStamp);
        // END COPIED
        String className = descriptionAndValue.getClassName();
        boolean useString = false;
        Object value = null;
        try {
            value = descriptionAndValue.getData();
        } catch (ClassNotFoundException e) {
            // should not happen!
            log.error("impossible deserialization", e);
            return;
        }
        if (descriptionAndValue.isOfPrimitiveType()) {
            if ("char".equals(className)) {
                useString = true;
            }
        } else {
            Class dataClass = null;
            try {
                dataClass = Class.forName(className);
            } catch (ClassNotFoundException e) {
                log.error("impossible deserialization", e);
                return;
            }
            if (Number.class.isAssignableFrom(dataClass)) {
                useString = false;
            } else {
                useString = true;
            }
        }
        if (useString) {
            data.setStringData(String.valueOf(value));
        } else {
            data.setDoubleData(((Number) value).doubleValue());
        }
        addToQueue(data, name);
    }


    public void queueImmediateScalar(long tStamp, String name, Object d) {
 
        log.debug("got update " + name);

        RawData data = new RawData();
        data.setTstamp(tStamp);
        if (d instanceof Double) {
            data.setDoubleData((Double) d);
        } else if (d instanceof Float) {
            data.setDoubleData((Double) (double) (Float) d);
        } else if (d instanceof Integer) {
            data.setDoubleData((Double) (double) (Integer) d);
        } else if (d instanceof Short) {
            data.setDoubleData((Double) (double) (Short) d);
        } else {
		//TODO: if array do an Array.toString
            data.setStringData(String.valueOf(d));
        }
        addToQueue(data, name);
    }


    public void queueMetadataStatus(MetadataStatus mst) {
        addToQueue(mst, null);
    }


    public void addToQueue(Object data, String name) {
        rq.add(new Object[]{data, name});
    }


    private void persistData(RawData data, String name, Session sess) {

        DataDesc dd = getDataDescription(name, sess);
        if (dd == null) return;

        data.setDescr(dd);
        sess.lock(dd, LockMode.NONE);
        sess.persist(data);

        List<StatDesc> stats = dd.getDerived();
        // TODO use named queries
        Query q = sess.createQuery("from StatData s where s.descr = :d "
                                   + "order by s.tstampFirst desc");
        q.setLockMode("s", LockMode.UPGRADE);
        for (StatDesc stat : stats) {
            q.setEntity("d", stat);
            q.setMaxResults(1);
            StatData sd = (StatData) q.uniqueResult();
            if (sd == null) {
                sd = new StatData(stat, data);
                sess.persist(sd);
            } else if (data.getTstamp() > sd.getTstampFirst()
                    + stat.getTimeBinWidth()) {
                sd = new StatData(stat, data);
                sess.persist(sd);
            } else {
                sd.accumulate(data);
            }
        }
    }


    private void persistMetadataStatus(MetadataStatus mst, Session sess) {

        String dataName = mst.getOrigin() + "/" + mst.getDataName();
        DataDesc dd = getDataDescription(dataName, sess);
        if (dd == null) return;

        sess.lock(dd, LockMode.NONE);

        DataMetaData metadata = new DataMetaData();
        metadata.setName(mst.getMetadataName());
        metadata.setRawDescr(dd);
        metadata.setValue(mst.getMetadataValue());
        metadata.setTstart(mst.getTimeStamp());

        // First update the tstop of a previously existing record for this
        // metadata
        Query q = sess.createQuery("from DataMetaData md "
                                   + "where rawDescr_id = :id "
                                   + "and name = :n and tstopmillis <= 0");
        q.setParameter("id", dd.getId());
        q.setParameter("n", metadata.getName());
        DataMetaData oldMetaData = (DataMetaData) q.uniqueResult();
        if (oldMetaData != null) {
            oldMetaData.setTstop(metadata.getTstart());
            sess.update(oldMetaData);
        }
        // Now commit the new value of the metadata
        sess.persist(metadata);
    }


    private DataDesc getDataDescription(String key, Session sess) {

        log.debug("Looking for data description in map " + map + " " + key);
        DataDesc dd = map.get(key);

        // TODO we should not do that in production.
        // this piece of code will make ALL data passing on the bus be
        // persisted by creating some default data description for it.
        // in preliminary version, datadesc should be populated by hand
        // TODO create GUI to allow user to select what should be persisted

        if (dd == null) {
            log.debug("Adding default Data Description for " + key);
            dd = new DataDesc();
            dd.setDataType("a");
            dd.setName(key);
            int ind = key.indexOf("/");
            dd.setSrcName(key.substring(ind + 1));
            dd.setSrcSubsystem(key.substring(0, ind));
            sess.persist(dd);
            map.put(key, dd);
        }

        return dd;
    }

}
