package org.lsst.ccs.efd;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.influxdb.dto.Query;
import org.lsst.ccs.camera.sal.xml.Mapping;

public class EfdQueryBuilder  {

    private final Map<String, EfdPathWrapper> pathToWrapperMap = new ConcurrentHashMap<>();
    
    private final String namespace;
    private final Mapping map = Mapping.getMapping("avro.map");

    private EfdPathWrapper efdPathWrapper;
    private long startTime = -1;
    private long endTime = -1;
    private int nSamples = -1;
    
    public EfdQueryBuilder(String namespace) {
        this.namespace = namespace;
    }
    
    public EfdQueryBuilder forPath(String path) {        
        efdPathWrapper = pathToWrapperMap.computeIfAbsent(path, (p) -> new EfdPathWrapper(path) );
        return this;
    }
    
    
    public EfdPathWrapper getEfdPathWrapper() {
        return efdPathWrapper;
    }
    
    public EfdQueryBuilder withTimeRange(long startTime, long endTime) {
        this.startTime = startTime;
        this.endTime = endTime;
        return this;
    }
    
    public EfdQueryBuilder withSamples(int nSamples) {
        this.nSamples = nSamples;
        return this;
    }
    
    public boolean isRawData() {
        return nSamples < 1;
    }
    
    public long getTimeWidth() {
        return (endTime - startTime)/nSamples;         
    }
    
    public Query buildQuery() {
        if ( startTime == -1 || endTime == -1 ) {
            throw new RuntimeException("Start and End times must be specified.");
        }
        
        boolean groupBy = nSamples > 0;
        
        String queryStr = "select ";
        String field = efdPathWrapper.getField();
        //What to select
        if ( groupBy ) {
            queryStr += "max("+field+") as maxValue, min("+field+") as minValue, "
                    + "mean("+field+") as meanValue, stddev("+field+") as stddevValue"; 
        } else {
            queryStr += field;
        }
        
        
        
        queryStr += " from \""+efdPathWrapper.getTopicName()+"\" where "+
                "time >  "+startTime+"ms  and time < "+endTime+"ms ";
        for ( String tag : efdPathWrapper.getTags() ) {
            queryStr += "and "+tag+" = '"+efdPathWrapper.getTagValue(tag)+"' ";
        }
        
        if ( groupBy ) {
            queryStr += "group by time("+getTimeWidth()+"ms) fill(none)";
        }
        return new Query(queryStr, efdPathWrapper.getNamespace());
    }
    
    
    
    
    public class EfdPathWrapper {
        
        private final Map<String,String> tags = new ConcurrentHashMap<>();

        private final String topicName;
        private final String field;
        
        
        EfdPathWrapper(String path) {

            int firstSlash = path.indexOf("/");
            String agentName = path.substring(0, firstSlash);
            String pathWithoutAgent = path.substring(firstSlash+1);
            
            Mapping.Match match = map.match(pathWithoutAgent);
            if ( match == null ) {
                throw new RuntimeException("Could not find a match for path "+pathWithoutAgent);
            }

            topicName = namespace + "." + match.getPatternName();
            
            for (String tag : match.getTagNames()) {
                tags.put(tag, match.getTagValue(tag));
            }
            
            tags.put("Agent",agentName);

            field = match.getPathAfterMatch();            
        }
        
        public String getNamespace() {
            return namespace;
        }
        
        public String getTopicName() {
            return topicName;
        }
        
        public String getField() {
            return field;
        }
        
        public Set<String> getTags() {
            return tags.keySet();
        }
        
        public String getTagValue(String tag) {
            return tags.get(tag);
        }
        
    }
    

}
