package org.lsst.ccs.localdb.statusdb.server;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import javax.inject.Singleton;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import org.hibernate.FlushMode;
import org.hibernate.LockMode;
import org.hibernate.query.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.lsst.ccs.localdb.statusdb.model.AgentSimpleState;
import org.lsst.ccs.localdb.statusdb.model.AlertDesc;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.DataPath;
import org.lsst.ccs.localdb.statusdb.model.InnerStateDesc;
import org.lsst.ccs.localdb.statusdb.model.MetaDataData;
import org.lsst.ccs.localdb.statusdb.model.RaisedAlertData;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.model.StatTimeInterval;
import org.lsst.ccs.localdb.statusdb.model.StateChangeNotificationData;
import org.lsst.ccs.localdb.statusdb.server.TrendingData.AxisValue;
import org.lsst.ccs.localdb.statusdb.server.TrendingData.DataValue;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;

/**
 * The DataServer receives requests for trending data analysis. It delivers the
 * requested data as the result of requests to the database.
 */
@SuppressWarnings("restriction")
@Path("/dataserver")
@Singleton
public class DataServer {

    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb");

    private final SessionFactory fac;

    public DataServer() {
        fac = StatusdbUtils.getSessionFactory();
        log.log(Level.INFO, "Starting Data Server");
    }

    // our session context is a thread context
    // we should request the current session, and not open a new one
    // if there is no current session if will create a new one
    // inner functions will reuse the session opened in the outer function.

    // REST entry points can use try-with-resource to close the session upon exit
    // inner methods just get the session

    // some framework have fool-proof @Transitional annotations that do enter/exit counting
    // we could add this to ease the use.

    private Session getSession() {
        Session s = fac.getCurrentSession();
        Transaction tx = s.getTransaction();
        if (!tx.isActive())
            tx.begin();
        return s;
    }

    private List<DataChannel> getListOfChannels(long dtSeconds, Session sess) {

        long start = System.currentTimeMillis();

        ArrayList<DataChannel> channels = new ArrayList<>();
        // select dataDescId, max(time) from ccs_rawData group by dataDescId
        //
        // select distinct dataDescId from (select dataDescId, max(time) m
        // from
        // ccs_rawData group by dataDescId) as xx where m>15249072119
        //
        // select distinct dataDescId from (select dataDescId from
        // ccs_rawData group by
        // dataDescId having max(time)>15249072119) as xx;

        // We don't want a join to happen in the SQL subquery. Using ids
        // prevents
        // the generated SQL to have a join with DataDesc
        Query dataQuery = sess.createQuery("from DataDesc as dd where dd.id in " //
                + " (select d.dataId from DataDescLastUpdate d where d.lastUpdate >:t)");

        if (dtSeconds > 0) {
            dataQuery.setParameter("t", System.currentTimeMillis() - dtSeconds * 1000L);
        } else {
            dataQuery.setParameter("t", 0L);
        }
        dataQuery.setReadOnly(true);
        @SuppressWarnings("unchecked")
        List<DataDesc> l = dataQuery.list();

        for (DataDesc d : l) {
            DataChannel dc = new DataChannel(d);
            dc.addMetadata("subsystem", d.getDataPath().getAgentName());
            dc.addMetadata("type", d.getDataType());
            channels.add(dc);
        }
        double delta = (System.currentTimeMillis() - start);
        log.log(Level.FINE, "Fetched list of channels ({0})  in {1} ms", new Object[] { l.size(), delta });

        return channels;
    }

    private long getDataIdForPath(String path, Session sess) {
        if (path.startsWith("/")) {
            path = path.substring(1);
        }
        int index = path.indexOf("/");
        String agentName = path.substring(0, index);
        String dataName = path.substring(index + 1);
        long dataId;
        Query q = sess.createQuery("select dd.id from DataDesc dd where dd.dataPath = :dataPath");
        q.setReadOnly(true);
        q.setParameter("dataPath", new DataPath(agentName, dataName));
        Object res = q.uniqueResult();
        if ( res == null ) {
            throw new RuntimeException("Could not find data id for path \""+path+"\"");
        }
        dataId = (long) q.uniqueResult();
        return dataId;
    }

    private String getPathForId(long id, Session sess) {
        String path = null;
        Query q = sess.createQuery("select dd.dataPath from DataDesc dd where dd.id = :id");
        q.setReadOnly(true);
        q.setParameter("id", id);
        path = ((DataPath) q.uniqueResult()).getFullKey();
        return path;
    }

    @GET
    @Path("/data/latest/")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public TrendingData getLatestDataByPath(@QueryParam("path") String path) {
        TrendingData result = null;
        try (Session sess = getSession()) {
            long dataId = getDataIdForPath(path, sess);            
            Query q = sess.createQuery("select r from RawData r inner join DataDescLastUpdate ddlu on ddlu.dataId = r.dataDesc.id where r.dataDesc.id = :id and r.time >= ddlu.lastUpdate order by r.time DESC");
            q.setReadOnly(true);
            q.setMaxResults(1);
            q.setParameter("id", dataId);
            RawData data = (RawData) q.uniqueResult();
            if ( data == null ) {
                data = new RawData();
                data.setTime(System.currentTimeMillis());
                data.setDoubleData(0.0);                
            }
            long tStamp = data.getTime();
            result = new TrendingData();
            AxisValue axisValue = new AxisValue("time", tStamp);
            result.setAxisValue(axisValue);
            DataValue[] dataValue = new DataValue[1];
            Double dd = data.getDoubleData();
            dataValue[0] = dd == null ? new DataValue("value", data.getStringData()) : new DataValue("value", dd);
            result.setDataValue(dataValue);
        }

        return result;
    }
        
    @GET
    @Path("/data/search/")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public List<Data> getDataByPath(@QueryParam("path") List<String> paths,
            @QueryParam("t1") @DefaultValue("-1") long t1, @QueryParam("t2") @DefaultValue("-1") long t2,
            @QueryParam("flavor") String flavor, @QueryParam("n") @DefaultValue("30") int nbins) {

        try (Session sess = getSession()) {
            List<Data> result = new ArrayList<>();
            for (String path : paths) {
                long dataId = getDataIdForPath(path, sess);
                Data d = getInnerData(path, dataId, t1, t2, flavor, nbins, sess);
                result.add(d);
            }
            return result;
        }
    }

    @GET
    @Path("/data/")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public List<Data> getData(@QueryParam("id") List<Long> ids, @QueryParam("t1") @DefaultValue("-1") long t1,
            @QueryParam("t2") @DefaultValue("-1") long t2, @QueryParam("flavor") String flavor,
            @QueryParam("n") @DefaultValue("30") int nbins) {
        try (Session sess = getSession()) {
            List<Data> result = new ArrayList<>();
            for (long id : ids) {
                Data d = getInnerData(null, id, t1, t2, flavor, nbins, sess);
                result.add(d);
            }
            return result;
        }
    }

    @GET
    @Path("/data/{id:  \\d+}")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public Data getData(@PathParam("id") long id, @QueryParam("t1") @DefaultValue("-1") long t1,
            @QueryParam("t2") @DefaultValue("-1") long t2, @QueryParam("flavor") String flavor,
            @QueryParam("n") @DefaultValue("30") int nbins) {
        try (Session sess = getSession()) {
            return getInnerData(null, id, t1, t2, flavor, nbins, sess);
        }
    }

    private Data getInnerData(String path, long id, long t1, long t2, String flavor, int nbins, Session sess) {

        long start = System.currentTimeMillis();
        if (path == null) {
            path = getPathForId(id, sess);
        }

        // handling of default values
        // t2 default is now
        // t1 default is t2 - 1 hour
        // rawPreferred is false
        // nbins is 30
        boolean useStat = false;
        boolean useRaw = false;
        if ("stat".equals(flavor)) {
            useStat = true;
        } else if ("raw".equals(flavor)) {
            useRaw = true;
        } else {
            flavor = "unspecified";
        }

        if (t2 < 0) {
            t2 = System.currentTimeMillis();
        }
        if (t1 < 0) {
            t1 = t2 - 3600000L;
        }

        // what do we have?
        long rawId = id;

        Data result = null;
        if (useRaw) {
            result = exportRawData(path, rawId, t1, t2, sess);
        } else if (useStat) {

            StatDesc statSource = null;
            long statSourceN = -1;
            List<StatDesc> stats = getAvailableStats(rawId, sess);

            for (StatDesc s : stats) {

                long n = (t2 - t1) / s.getTimeBinWidth();
                if (n > nbins / 2) {
                    // this one has enough bins
                    // TODO # of samples is not enough, for instance raw could
                    // have more samples
                    // but covering only the recent part of the time range.
                    if (statSource != null) {
                        if (n < statSourceN) {
                            statSource = s;
                            statSourceN = n;
                        }
                    } else {
                        statSource = s;
                        statSourceN = n;
                    }
                }
            }

            if (statSource != null) {
                result = exportStatDataFromStat(path, statSource, t1, t2, nbins, sess);
            }
        }

        if (result == null) {
            result = exportStatDataFromRaw(path, rawId, t1, t2, nbins, sess);
        }
        long delta = System.currentTimeMillis() - start;
        log.log(Level.FINE,
                "Request for data id: {0} interval: [{1}:{2}] delta: {6} ms flavor: {3} nbins: {4} took {5} ms for {7} entries",
                new Object[] { String.valueOf(id), String.valueOf(t1), String.valueOf(t2), flavor, nbins,
                        String.valueOf(delta), String.valueOf(t2 - t1), result.getTrendingResult().data.length });
        return result;
    }

    @SuppressWarnings("unchecked")
    @GET
    @Path("/metadata/")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public List<ChannelMetaData> getMetadataForChannel(@QueryParam("id") int channelId,
            @QueryParam("t1") @DefaultValue("-1") long t1,
            @QueryParam("t2") @DefaultValue("-1") long t2) {
        try (Session sess = getSession()) {
            return innerGetMetadataForChannel(channelId, t1, t2, sess);
        }
    }

    private List<ChannelMetaData> innerGetMetadataForChannel(int channelId, long t1, long t2, Session sess) {
        List<MetaDataData> l = null;
        // String queryStr = "select md from MetaDataData md inner join
        // md.dataGroup g
        // inner join g.members m where m.id=:id ";
        // if (t1>-1&&t2>-1) {
        // queryStr += "and ((startTime >= :t1 and startTime < :t2) or
        // (startTime < :t1
        // and (endTime > :t1 or endTime = -1)))";
        // } else if (t2<0&&t1>-1) { // No upper limit
        // queryStr += "and startTime >= :t1";
        // } else if (t2>-1&&t1<0) { // No lower limit
        // queryStr += "and startTime < :t2 or endTime = -1";
        // }

        // mysql is not efficient on queries like
        // ((startTime >= :t1 and startTime < :t2) or (startTime < :t1 and
        // (endTime >
        // :t1 or endTime = -1)))
        // one should union the two queries
        // ((startTime >= :t1 and startTime < :t2)
        // (startTime < :t1 and (endTime > :t1 or endTime = -1)))
        // no UNION in HSQL => two queries.
        // we also want to recover the list of groups for a channel.
        String groupQuery = "select g.id from DataGroup g inner join g.members m where m.id=:id";
        Query qGroup = sess.createQuery(groupQuery);
        qGroup.setLockMode("g", LockMode.NONE);
        qGroup.setReadOnly(true);
        qGroup.setLong("id", channelId);
        List<Integer> lgid = qGroup.list();

        if (lgid.isEmpty()) {
            return new ArrayList<ChannelMetaData>();
        }

        if (t1 > -1 && t2 > -1) {
            String queryStr = "select md from MetaDataData md where md.dataGroup.id in (select g.id from DataGroup g inner join g.members m where m.id=:id) "
                    + " and (startTime >= :t1 and startTime < :t2)";
            Query q = sess.createQuery(queryStr);
            q.setLockMode("md", LockMode.NONE);
            q.setReadOnly(true);
            q.setLong("id", channelId);
            q.setParameter("t1", t1);
            q.setParameter("t2", t2);
            l = q.list();

            queryStr = "select md from MetaDataData md where md.dataGroup.id in (select g.id from DataGroup g inner join g.members m where m.id=:id) "
                    + " and (startTime < :t1 and (endTime > :t1 or endTime = -1))";
            q = sess.createQuery(queryStr);
            q.setLockMode("md", LockMode.NONE);
            q.setLong("id", channelId);
            q.setParameter("t1", t1);
            q.setReadOnly(true);
            l.addAll(q.list());
        } else if (t2 < 0 && t1 > -1) { // No upper limit
            String queryStr = "select md from MetaDataData md where md.dataGroup.id in (select g.id from DataGroup g inner join g.members m where m.id=:id) "
                    + " and startTime >= :t1";
            Query q = sess.createQuery(queryStr);
            q.setLockMode("md", LockMode.NONE);
            q.setLong("id", channelId);
            q.setReadOnly(true);
            q.setParameter("t1", t1);
            l = q.list();
        } else if (t2 > -1 && t1 < 0) { // No lower limit
            String queryStr = "select md from MetaDataData md where md.dataGroup.id in (select g.id from DataGroup g inner join g.members m where m.id=:id) "
                    + " and (startTime < :t2 or endTime = -1)";
            Query q = sess.createQuery(queryStr);
            q.setLong("id", channelId);
            q.setLockMode("md", LockMode.NONE);
            q.setReadOnly(true);
            q.setParameter("t2", t2);
            l = q.list();
        } else {
            String queryStr = "select md from MetaDataData md where md.dataGroup.id in (select g.id from DataGroup g inner join g.members m where m.id=:id) ";
            Query q = sess.createQuery(queryStr);
            q.setLong("id", channelId);
            q.setLockMode("md", LockMode.NONE);
            q.setReadOnly(true);
            l = q.list();
        }

        List<ChannelMetaData> out = new ArrayList<ChannelMetaData>();
        for (MetaDataData md : l) {
            out.add(new ChannelMetaData(md));
        }
        return out;

    }

    @GET
    @Path("/channelinfo/{id}")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public ChannelMetaData.ChannelMetadataList getMetadataList(@PathParam("id") long channelId) {
        try (Session sess = getSession()) {
            long rawId = channelId;
            return new ChannelMetaData.ChannelMetadataList(innerGetMetadataForChannel((int) rawId, -1, -1, sess));
        }
    }

    @GET
    @Path("listsubsystems")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public DataSubsystem.DataSubsystemList getSubsystems() {
        try (Session sess = getSession()) {
            return new DataSubsystem.DataSubsystemList(getListOfSubsystems(sess));
        }
    }

    private List<DataSubsystem> getListOfSubsystems(Session sess) {
        List<DataSubsystem> l = null;
        Query dataQuery = sess.createQuery("select distinct dd.dataPath.agentName from DataDesc dd");
        dataQuery.setReadOnly(true);
        l = ((List<String>) dataQuery.list()).stream().map((String agentName) -> {
            return new DataSubsystem(agentName);
        }).collect(Collectors.toList());

        return l;
    }

    /**
     *
     * @return the whole channels list for all CCS.
     * 
     *         optional query parameter maxIdleSeconds with default value 86400:
     *         only returns parameters seen since that delay. Negative value:
     *         returns all channels.
     */
    @GET
    @Path("/listchannels")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public DataChannel.DataChannelList getChannels(@QueryParam("maxIdleSeconds") @DefaultValue("604800") long dt) {
        try (Session sess = getSession()) {
            return new DataChannel.DataChannelList(getListOfChannels(dt, sess), System.currentTimeMillis() - dt * 1000);
        }
    }

    /**
     *
     * @param subsystemName
     * @return a channels list for a subsystem
     * 
     *         optional query parameter maxIdleSeconds with default value 36000:
     *         only returns parameters seen since that delay. Negative value:
     *         returns all channels.
     */
    @GET
    @Path("/listchannels/{subsystem}")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public DataChannel.DataChannelList getChannels(@PathParam("subsystem") String subsystemName,
            @QueryParam("maxIdleSeconds") @DefaultValue("604800") long dt) {
        try (Session sess = getSession()) {
            List<DataChannel> channels = getListOfChannels(dt, sess);
            ArrayList<DataChannel> subChannels = new ArrayList<DataChannel>();
            for (DataChannel dc : channels) {
                if (dc.getPath()[0].equals(subsystemName)) {
                    subChannels.add(dc);
                }
            }
            return new DataChannel.DataChannelList(subChannels, System.currentTimeMillis() - dt * 1000);
        }
    }

    /**
     *
     * @param partialPath
     * @param level
     * @return the list of channels within a partial path and a level
     */
    public DataChannel[] getChannels(String partialPath, int level) {
        // TODO
        return null;
    }

    /**
     * Get the StatDesc for a given raw Id
     * 
     * @param rawId
     *              the DataDesc id
     * @return The List of StatDesc.
     */
    public List<StatDesc> getAvailableStats(long rawId, Session sess) {

        List<StatDesc> stats = sess.getNamedQuery("findStatDescById").setLong("id", rawId)
                .setHibernateFlushMode(FlushMode.COMMIT)
                .setCacheable(true)
                .setReadOnly(true)
                .setLockMode("sd", LockMode.NONE)
                .list();
        return stats;

    }

    public List<RawData> getRawData(long id, long t1, long t2, Session sess) {

        List<RawData> l = null;

        Query q = sess.createQuery(
                "from RawData r where r.dataDesc.id = :id and r.time between :t1 and :t2 order by r.time");
        q.setReadOnly(true);
        q.setLockMode("r", LockMode.NONE);
        q.setParameter("id", id);
        q.setParameter("t1", t1);
        q.setParameter("t2", t2);
        l = q.list();

        return l;
    }

    /**
     * Returns the list of StatData with the given statDesc that overlap the
     * interval [t1,t2[.
     * 
     * @param statDesc
     *                 the statistical description
     * @param t1
     * @param t2
     * @return
     */
    public List<StatData> getStatData(StatDesc statDesc, long t1, long t2, Session sess) {

        List<StatData> l = null;
        Query<StatData> q = sess.createQuery(
                "from StatData r join fetch r.statTimeInterval where r.statDesc.id = :id and r.statTimeInterval.startTime > :tlow and r.statTimeInterval.startTime < :thigh order by r.statTimeInterval.startTime");
        q.setReadOnly(true);
        q.setLockMode("r", LockMode.NONE);
        q.setParameter("id", statDesc.getId());
        q.setParameter("tlow", t1 - statDesc.getTimeBinWidth());
        q.setParameter("thigh", t2);
        l = q.list();

        return l;
    }

    protected Data exportRawData(String path, long rawId, long t1, long t2, Session sess) {
        List<RawData> l = getRawData(rawId, t1, t2, sess);

        Data d = new Data();
        d.setMetaDataData(innerGetMetadataForChannel((int) rawId, t1, t2, sess));

        TrendingData[] data = new TrendingData[l.size()];
        for (int i = 0; i < l.size(); i++) {
            RawData r = l.get(i);
            TrendingData dt = new TrendingData();
            data[i] = dt;
            long tStamp = r.getTime();
            AxisValue axisValue = new AxisValue("time", tStamp);
            dt.setAxisValue(axisValue);
            DataValue[] dataValue = new DataValue[1];
            Double dd = r.getDoubleData();
            dataValue[0] = dd == null ? new DataValue("value", r.getStringData()) : new DataValue("value", dd);
            dt.setDataValue(dataValue);
        }
        d.getTrendingResult().setTrendingDataArray(data);
        d.setId(rawId);
        d.setPath(path);

        return d;
    }

    protected static class StatEntry {
        long low, high;
        double sum1, sum2, min, max;
        long n;

        public StatEntry(long low, long high, double sum1, double sum2, long n, double min, double max) {
            super();
            this.low = low;
            this.high = high;
            this.sum1 = sum1;
            this.sum2 = sum2;
            this.min = min;
            this.max = max;
            this.n = n;
        }

        public long getLow() {
            return low;
        }

        public long getHigh() {
            return high;
        }

        public double getMin() {
            return min;
        }

        public double getMax() {
            return max;
        }

        public double getMean() {
            return n > 0 ? sum1 / n : 0;
        }

        public double getStdDev() {
            double nd = (double) n;
            return n > 0 ? Math.sqrt((sum2 / nd) - ((sum1 * sum1) / (nd * nd))) : 0;
        }

    }

    protected Data exportStatDataFromRaw(String path, long rawId, long t1, long t2, int nsamples, Session sess) {
        Query q = sess.createQuery(
                "select new org.lsst.ccs.localdb.statusdb.server.DataServer$StatEntry(min(rd.time), max(rd.time), sum(rd.doubleData), "
                        + " sum(rd.doubleData*rd.doubleData), count(1), min(rd.doubleData), max(rd.doubleData)) from"
                        + " RawData rd where rd.dataDesc.id = :id and rd.time >= :t1  and rd.time < :t2 "
                        + " group by floor(rd.time/:deltat) having count(1) > 0 ORDER BY min(rd.time) ASC");
        q.setReadOnly(true);
        long deltat = (t2 - t1) / nsamples;

        q.setParameter("id", rawId);
        q.setParameter("t1", t1);
        q.setParameter("t2", t2);
        q.setParameter("deltat", deltat);

        @SuppressWarnings("unchecked")
        List<StatEntry> l = q.list();

        List<TrendingData> trendingDataList = new ArrayList<>();

        for (StatEntry se : l) {

            long low = se.getLow();
            long high = se.getHigh();
            TrendingData dt = new TrendingData();
            dt.setAxisValue(new AxisValue("time", (low + high) / 2, low, high));
            DataValue[] dataValue = new DataValue[5];

            double value = se.getMean();
            double stddev = se.getStdDev();

            double min = se.getMin();
            double max = se.getMax();
            dataValue[0] = new DataValue("value", value);
            dataValue[1] = new DataValue("rms", stddev);
            dataValue[2] = new DataValue("stddev", stddev);
            dataValue[3] = new DataValue("min", min);
            dataValue[4] = new DataValue("max", max);
            dt.setDataValue(dataValue);
            trendingDataList.add(dt);
        }

        TrendingData[] trendingData = trendingDataList.toArray(new TrendingData[0]);

        Data d = new Data();
        d.setId(rawId);
        d.setPath(path);
        d.setMetaDataData(innerGetMetadataForChannel((int) rawId, t1, t2, sess));
        d.getTrendingResult().setTrendingDataArray(trendingData);

        return d;

    }

    // protected Data exportStatDataFromRawNative(String path, long rawId, long t1,
    // long t2, int nsamples) {
    //
    // try (Session sess = getSession()) {
    //
    // SQLQuery q = sess.createSQLQuery("select tlow, thigh, datasum/entries as
    // mean, " //
    // + " case when entries > 0 then " //
    // + " sqrt((datasumsquared - datasum*datasum/entries)/entries) " //
    // + " else 0.0 end as rms, " //
    // + " mindata, maxdata " //
    // + " from ( SELECT MIN(rd.time) AS tlow, MAX(rd.time) AS thigh, " //
    // + " SUM(rd.doubleData) AS datasum, SUM(rd.doubleData*rd.doubleData) AS
    // datasumsquared, " //
    // + " min(rd.doubleData) as mindata, max(rd.doubleData) as maxdata, " //
    // + " count(1) AS entries from ccs_rawData rd where rd.dataDescId = :id and
    // time >= :t1 " //
    // + " and time < :t2 group by floor(rd.time/:deltat) ) accumulated where
    // entries > 0 ORDER BY tlow ASC");
    // q.setReadOnly(true);
    //
    // long deltat = (t2 - t1) / nsamples;
    //
    // q.setParameter("id", rawId);
    // q.setParameter("t1", t1);
    // q.setParameter("t2", t2);
    // q.setParameter("deltat", deltat);
    //
    // List<Object[]> l = null;
    // l = q.list();
    //
    // List<TrendingData> trendingDataList = new ArrayList<>();
    // for (Object[] obj : l) {
    //
    // if (obj[2] != null) {
    // long low = ((BigInteger) obj[0]).longValue();
    // long high = ((BigInteger) obj[1]).longValue();
    // TrendingData dt = new TrendingData();
    // dt.setAxisValue(new AxisValue("time", (low + high) / 2, low, high));
    // DataValue[] dataValue = new DataValue[4];
    //
    // double value = ((Double) obj[2]).doubleValue();
    // double rms = 0;
    // if (obj[3] != null) {
    // rms = ((Double) obj[3]).doubleValue();
    // }
    // double min = ((Double) obj[4]).doubleValue();
    // double max = ((Double) obj[5]).doubleValue();
    // dataValue[0] = new DataValue("value", value);
    // dataValue[1] = new DataValue("rms", rms);
    // dataValue[2] = new DataValue("min", min);
    // dataValue[3] = new DataValue("max", max);
    // dt.setDataValue(dataValue);
    // trendingDataList.add(dt);
    // }
    // }
    //
    // TrendingData[] trendingData = trendingDataList.toArray(new TrendingData[0]);
    //
    // Data d = new Data();
    // d.setId(rawId);
    // d.setPath(path);
    // d.setMetaDataData(getMetadataForChannel((int) rawId, t1, t2));
    // d.getTrendingResult().setTrendingDataArray(trendingData);
    // d.setId(rawId);
    //
    // return d;
    // }
    // }

    protected Data exportStatDataFromStat(String path, StatDesc statDesc, long t1, long t2, int nsamples, Session sess) {
        List<StatData> in = getStatData(statDesc, t1, t2, sess);
        
        // If the number of available bins is more than 3 times larger than the number of requested points, re-bin.

        int n = in.size();
        while (n > nsamples * 3) {
            int rebin = n / nsamples;
            List<StatData>  out = new ArrayList<>((n + rebin - 1) / rebin);
            int nbin = 0;
            double sum = 0;
            double s2 = 0;
            int nsamp = 0;
            long low = 0;
            long high = 0;
            double max = -Double.MAX_VALUE;
            double min = Double.MAX_VALUE;
            ListIterator<StatData> it = in.listIterator();
            while (it.hasNext()) {
                StatData sd = it.next();
                StatTimeInterval timeInterval = sd.getStatTimeInterval();
                if (nbin > 0 && ((timeInterval.getStartTime() - high) > (2*timeInterval.getBinWidth()+1))) { // close bin if time gap
                    StatData sdOut = new StatData();
                    sdOut.setStatTimeInterval(new StatTimeInterval(low, high - low));
                    sdOut.setMax(max);
                    sdOut.setMin(min);
                    sdOut.setN(nsamp);
                    sdOut.setSum(sum);
                    sdOut.setSum2(s2);
                    out.add(sdOut);
                    sum = 0; s2 = 0; nsamp = 0; low = 0; high = 0; max = -Double.MAX_VALUE; min = Double.MAX_VALUE;
                    nbin = 0;
                }
                if (low == 0) low = timeInterval.getStartTime();
                high = timeInterval.getEndTime();
                sum += sd.getSum();
                s2 += sd.getSum2();
                if (sd.getMax() > max) max = sd.getMax();
                if (sd.getMin() < min) min = sd.getMin();
                nsamp += sd.getN();
                nbin++;
                if (nbin > 0 && (nbin == rebin || !it.hasNext())) { // close bin if enough source bins or last point
                    StatData sdOut = new StatData();
                    sdOut.setStatTimeInterval(new StatTimeInterval(low, high - low));
                    sdOut.setMax(max);
                    sdOut.setMin(min);
                    sdOut.setN(nsamp);
                    sdOut.setSum(sum);
                    sdOut.setSum2(s2);
                    out.add(sdOut);
                    sum = 0; s2 = 0; nsamp = 0; low = 0; high = 0; max = -Double.MAX_VALUE; min = Double.MAX_VALUE;
                    nbin = 0;
                }
            }
            if (out.size() < n) {
                in = out;
                n = out.size();
            } else {
                break; // cannot further reduce the number of bins
            }
        }
        
        // Convert to Data

        Data d = new Data();
        d.setMetaDataData(innerGetMetadataForChannel((int) statDesc.getDataDesc().getId(), t1, t2, sess));
        d.setPath(path);

        d.getTrendingResult().setTrendingDataArray(new TrendingData[n]);
        int iout = 0;
        for (StatData sd : in) {
            TrendingData dt = new TrendingData();
            d.getTrendingResult().getTrendingDataArray()[iout++] = dt;
            StatTimeInterval timeInterval = sd.getStatTimeInterval();
            long low = timeInterval.getStartTime();
            long high = timeInterval.getEndTime();
            dt.setAxisValue(new AxisValue("time", (low + high) / 2, low, high));
            int nsamp = sd.getN();
            double sum = sd.getSum();
            DataValue[] dataValue = new DataValue[5];
            dataValue[0] = new DataValue("value", sum / nsamp);
            double stddev = Math.sqrt(sd.getSum2() / nsamp - (sum / nsamp) * (sum / nsamp));
            dataValue[1] = new DataValue("rms", stddev);
            dataValue[2] = new DataValue("stddev", stddev);
            dataValue[3] = new DataValue("min", sd.getMin());
            dataValue[4] = new DataValue("max", sd.getMax());
            dt.setDataValue(dataValue);
        }

        d.setId(statDesc.getDataDesc().getId());
        return d;
    }

    // -- State changes
    /**
     * Lists all the existing states for a given subsystem.
     * 
     * @param subsystem
     *                  the subsystem name.
     * @param states
     *                   if empty, returns all state changes for all found states.
     * @return
     */
    @GET
    @Path("statesinfo/{subsystem}")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public StateInfo.StateInfoList getStateInfo(@PathParam("subsystem") String subsystem, @QueryParam("state") List<String> states) {
        List<StateInfo> res = new ArrayList<>();
        try (Session sess = getSession()) {

            // Fetching the internal states
            Query q = sess
                    .createQuery(
                            "select distinct isd from AgentState as ags inner join ags.componentStates as sbd inner join sbd.componentStates as isd where ags.agentDesc.agentName=:agentName")
                    .setString("agentName", subsystem).setReadOnly(true);
            Map<String, StateInfo> innerStateInfoMap = new HashMap<>();
            List<InnerStateDesc> isdl = q.list();
            for (InnerStateDesc isd : isdl) {
                String isdClassName = isd.getEnumSimpleClassName();
                StateInfo si = innerStateInfoMap.get(isdClassName);
                if (si == null) {
                    si = new StateInfo(isdClassName);
                    innerStateInfoMap.put(isdClassName, si);
                }
                si.getStateValues().add(isd.getEnumValue());
            }
            for (StateInfo si : innerStateInfoMap.values()) {
                if ( states.isEmpty() || states.contains(si.getStateClass()) ) {
                    res.add(si);
                }
            }
        }
        return new StateInfo.StateInfoList(res);
    }

    /**
     * Gets the state changes events for the given list of states, between the
     * time interval {@code [t1,t2]}, for the given subsystem.
     *
     * @param subsystem
     *                   the subsystem
     * @param states
     *                   if empty, returns all state changes for all found states.
     * @param components
     *                   if empty, return all state changes for all components.
     * @param t1
     * @param t2
     * @return
     */
    @GET
    @Path("/statechanges/{subsystem}")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public StateChange.StateChangesList getStateChangeList(@PathParam("subsystem") String subsystem,
            @QueryParam("state") List<String> states, @QueryParam("component") List<String> components,
            @QueryParam("t1") @DefaultValue("-1") long t1, @QueryParam("t2") @DefaultValue("-1") long t2) {
        try (Session sess = getSession()) {
            return innerGetStateChangeList(subsystem, components, states, t1, t2, sess);
        }
    }

    protected StateChange.StateChangesList getComponentStateChangeList(String subsystem,
            String component, @QueryParam("state") List<String> states,
            @QueryParam("t1") @DefaultValue("-1") long t1, @QueryParam("t2") @DefaultValue("-1") long t2) {
        try (Session sess = getSession()) {
            return innerGetStateChangeList(subsystem, Arrays.asList(component), states, t1, t2, sess);
        }
    }

    private StateChange.StateChangesList innerGetStateChangeList(String subsystem, List<String> components,
            List<String> states, long t1, long t2, Session sess) {
        if (t2 < 0) {
            t2 = System.currentTimeMillis();
        }
        if (t1 < 0) {
            t1 = t2 - 3600000L;
        }

        List<StateChangeNotificationData> stateChangeNotifications = getStateChanges(subsystem, t1, t2, sess);
        if (stateChangeNotifications.isEmpty()) {
            return new StateChange.StateChangesList();
        }
        List<StateChange> stateChanges = new ArrayList<>();

        // Processing the first StateChangeNotificationData
        StateChangeNotificationData firstSCND = stateChangeNotifications.get(0);
        Map<String, Map<String, String>> origStates = null;
        if (firstSCND.getTime() <= t1) {
            origStates = firstSCND.getNewState().asFlatStatesMap(components);
        } else {
            origStates = firstSCND.getOldState().asFlatStatesMap(components);
        }
        StateChange sc = new StateChange();
        sc.setTime(t1);
        for (Map.Entry<String, Map<String, String>> e1 : origStates.entrySet()) {
            for (Map.Entry<String, String> e2 : e1.getValue().entrySet()) {
                if (states == null || states.isEmpty() || states.contains(e2.getKey())) {
                    sc.addStateEvent(e1.getKey(), e2.getKey(), e2.getValue());
                }
            }
        }

        stateChanges.add(sc);
        StateChange originSE = sc;

        for (StateChangeNotificationData scnd : stateChangeNotifications) {
            if (scnd.getTime() <= t1)
                continue;
            sc = new StateChange();

            // Computing the delta between oldState and newState
            Map<String, Map<String, String>> oldState = scnd.getOldState().asFlatStatesMap(components);
            Map<String, Map<String, String>> newState = scnd.getNewState().asFlatStatesMap(components);

            for (Map.Entry<String, Map<String, String>> e1 : newState.entrySet()) {
                String component = e1.getKey();
                for (Map.Entry<String, String> e2 : e1.getValue().entrySet()) {
                    String enumClassName = e2.getKey();
                    String enumValue = e2.getValue();
                    if (states == null || states.isEmpty() || states.contains(enumClassName)) {
                        Map<String, String> prevCompStates = oldState.get(component);
                        if (prevCompStates == null || !enumValue.equals(prevCompStates.get(enumClassName))) {
                            sc.addStateEvent(component, enumClassName, enumValue);
                            if (!originSE.containsEvent(component, enumClassName)) {
                                originSE.addStateEvent(component, enumClassName, "_UNDEF_");
                            }
                        }
                    }
                }
            }

            // This should not happen
            if (sc.isEmpty())
                continue;

            sc.setTime(scnd.getTime());

            stateChanges.add(sc);
        }
        return new StateChange.StateChangesList(stateChanges);
    }

    
    /**
     * Get the state tree structure for a given subsystem.
     * 
     * @param subsystem
     *                  the subsystem name.
     * @param dt the time interval to query
     * @return
     */
    @GET
    @Path("liststates/{subsystem}")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public StateInfo.StateChannelList getListOfStates2(@PathParam("subsystem") String subsystem, @QueryParam("maxIdleSeconds") @DefaultValue("604800") long dt) {
        List<Channel> list = new ArrayList();        
        try (Session sess = getSession()) {                                
            list.addAll(innerGetListOfStates2(sess, subsystem,dt));
        }
        long start = 0L;
        if (dt > 0) {
            start = System.currentTimeMillis() - dt * 1000L;
        }
        return new StateInfo.StateChannelList(list,start);        
    }

    
    private List<Channel> innerGetListOfStates2(Session sess, String subsystem, long dt) {
        long queryStart = System.currentTimeMillis();
        List<Channel> list = new ArrayList();
        String fullPath = subsystem+"/";
        long end = System.currentTimeMillis();
        long start = end - dt * 1000;
        
        Query stateQuery = null;
        if ( subsystem != null ) {
            stateQuery = sess.createQuery("from AgentSimpleState where lastUpdate >:t and agentName =:name")
                    .setLong("t", start).setString("name", subsystem);
        } else {
            stateQuery = sess.createQuery("from AgentSimpleState where lastUpdate >:t")
                    .setLong("t", start);            
        }
        List<AgentSimpleState> states = stateQuery.list();
        for ( AgentSimpleState state : states ) {

            StateDataChannel ch = new StateDataChannel(state.getStatePath(),state.getEnumClass());
            list.add(ch);
        }
        long delta = System.currentTimeMillis()-queryStart;
        log.log(Level.FINE,"Fetching list of states for subsystem {0} over time window {1} took {2} ms", new Object[]{subsystem,dt,delta});
        return list;
    }
    
    /**
     * Get the state tree structure for a given subsystem.
     * 
     * @param subsystem
     *                  the subsystem name.
     * @return
     */
    @GET
    @Path("liststates")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public StateInfo.StateChannelList getListOfStates2(@QueryParam("maxIdleSeconds") @DefaultValue("604800") long dt) {
        List<Channel> res = new ArrayList();
        try ( Session sess = getSession()) {
            res.addAll(innerGetListOfStates2(sess, null,dt));
        }
        long start = 0L;
        if (dt > 0) {
            start = System.currentTimeMillis() - dt * 1000L;
        }
        return new StateInfo.StateChannelList(res,start);
    }
        
    /**
     * Returns the state changes events occurring from a given subsystem within
     * the time interval [t1;t2]. If none is found, it returns the closest
     * StateChangeNotificationData before t1.
     *
     * @param subsystemName
     *                      the subsystem name
     * @param t1
     *                      tstart
     * @param t2
     *                      tstop
     * @return
     */
    private List<StateChangeNotificationData> getStateChanges(String subsystemName, long t1, long t2, Session sess) {
        List<StateChangeNotificationData> res = null;
        long start = System.currentTimeMillis();
        Query q = sess
                .createQuery(
                        "from StateChangeNotificationData scnd where scnd.agentDesc.agentName=:name and scnd.time >:t1 and scnd.time <=:t2")
                .setString("name", subsystemName).setLong("t1", t1).setLong("t2", t2).setReadOnly(true);
        res = q.list();
        if (res.isEmpty()) {
            // Finding the closest previous state change notification status
            q = sess.createQuery(
                    "from StateChangeNotificationData scnd where scnd.agentDesc.agentName=:name and scnd.time <=:t1 order by scnd.time desc")
                    .setString("name", subsystemName).setLong("t1", t1).setMaxResults(1).setReadOnly(true);
            res = q.list();
        }
        long delta = System.currentTimeMillis()-start;
        log.log(Level.FINE,"Fetching state changes for {0} took {1} ms",new Object[]{subsystemName,delta});
        return res;
    }


    public AlertInfo.AlertInfoList getAlertInfo() {
        return getAlertInfo(0L);
    }
    // -- Alerts information retrieval
    @Path("/listalerts")
    @GET
    public AlertInfo.AlertInfoList getAlertInfo(@QueryParam("maxIdleSeconds") @DefaultValue("604800") long dt) {
        return getAlertInfo(null, dt);
    }

    public AlertInfo.AlertInfoList getAlertInfo(@PathParam("subsystem") String subsystem) {
        return getAlertInfo(subsystem, 0L);
    }
    @Path("/listalerts/{subsystem}")
    @GET
    public AlertInfo.AlertInfoList getAlertInfo(@PathParam("subsystem") String subsystem, @QueryParam("maxIdleSeconds") @DefaultValue("604800") long dt) {
        List<AlertInfo> res = new ArrayList<>();
        
        long time = 0;
        if (dt > 0) {
            time = System.currentTimeMillis() - dt * 1000L;
        } 
        
        try (Session sess = getSession()) {
            Query q;
            if (subsystem == null) {
                q = sess.createQuery("from AlertDesc ad where ad.id in (select adata.alertDesc.id from RaisedAlertData adata where adata.time > :t1 group by adata.alertDesc.id)").setParameter("t1", time);
            } else {
                q = sess.createQuery("from AlertDesc ad where ad.id in (select adata.alertDesc.id from RaisedAlertData adata where adata.time > :t1 and adata.agentName=:name group by adata.alertDesc.id)").setString("name",
                        subsystem).setParameter("t1", time);
            }
            q.setReadOnly(true);
            List<AlertDesc> l = q.list();

            for (AlertDesc ad : l) {
                res.add(new AlertInfo(ad.getId(), ad.getAgentDesc().getAgentName(), ad.getAlertId(),
                        ad.getAlertDescription()));
            }
        }
        return new AlertInfo.AlertInfoList(res);
    }

    /**
     * Returns a list of active alerts. Active alerts are alerts that have been
     * raised during the current subsystem run.
     *
     * @param ids
     *            a list of ids.
     * @return
     */
    @Path("activealerts")
    @GET
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public AlertEvent.AlertEventList getActiveAlerts(@QueryParam("id") List<Long> ids) {
        List<AlertEvent> res = new ArrayList<>();
        try (Session sess = getSession()) {
            Query q;
            if (ids == null || ids.isEmpty()) {
                q = sess.createQuery("from RaisedAlertData arad where arad.active=true");
            } else {
                q = sess.createQuery("from RaisedAlertData arad where arad.alertDesc.id in (:ids) and arad.active=true")
                        .setParameterList("ids", ids).setReadOnly(true);
            }
            q.setReadOnly(true);
            for (RaisedAlertData rad : (List<RaisedAlertData>) q.list()) {
                AlertEvent ae = new AlertEvent(rad.getAlertDesc().getAlertId(), rad.getTime(), rad.getSeverity(),
                        rad.getAlertCause());
                ae.setCleared(rad.getClearingAlert() != null);
                res.add(ae);
            }
        }
        return new AlertEvent.AlertEventList(res);
    }

    /**
     * Returns an history of alerts that have been raised within the given time
     * interval.
     *
     * @param ids
     *            a list of alertdesc ids.
     * @param t1
     * @param t2
     * @return
     */
    @Path("alerthistory")
    @GET
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public AlertEvent.AlertEventList getAlertHistories(@QueryParam("id") List<Long> ids,
            @QueryParam("t1") @DefaultValue("-1") long t1, @QueryParam("t2") @DefaultValue("-1") long t2) {
        if (t2 < 0) {
            t2 = System.currentTimeMillis();
        }
        if (t1 < 0) {
            t1 = t2 - 3600000L;
        }
        List<AlertEvent> res = new ArrayList<>();
        try (Session sess = getSession()) {
            Query q;
            if (ids == null || ids.isEmpty()) {
                q = sess.createQuery("from RaisedAlertData arad where arad.time >= :t1 and arad.time < :t2").setParameter("t1",t1).setParameter("t2", t2);
            } else {
                q = sess.createQuery("from RaisedAlertData arad where arad.alertDesc.id in (:ids) and arad.time >= :t1 and arad.time < :t2")
                        .setParameterList("ids", ids).setParameter("t1",t1).setParameter("t2", t2).setReadOnly(true);
            }
            q.setReadOnly(true);
            for (RaisedAlertData rad : (List<RaisedAlertData>) q.list()) {
                AlertEvent ae = new AlertEvent(rad.getAlertDesc().getAlertId(), rad.getTime(), rad.getSeverity(),
                        rad.getAlertCause());
                ae.setCleared(rad.getClearingAlert() != null);
                res.add(ae);
            }
        }
        return new AlertEvent.AlertEventList(res);
    }

}
