package org.lsst.ccs.localdb.statusdb.model;

import java.util.ArrayList;
import java.util.List;
import org.hibernate.boot.model.naming.Identifier;
import org.hibernate.boot.model.naming.ImplicitEntityNameSource;
import org.hibernate.boot.model.naming.ImplicitForeignKeyNameSource;
import org.hibernate.boot.model.naming.ImplicitNamingStrategyJpaCompliantImpl;
import org.hibernate.boot.model.naming.ImplicitUniqueKeyNameSource;
import org.hibernate.boot.model.naming.ImplicitIndexNameSource;
import org.hibernate.boot.model.naming.ImplicitJoinColumnNameSource;
import org.hibernate.boot.model.naming.ImplicitJoinTableNameSource;

/**
 * Overrides the default implicit naming strategy with custom names for foreign
 * keys.
 *
 * @author LSST CCS Team
 */
public class CCSNamingStrategy extends ImplicitNamingStrategyJpaCompliantImpl {

    private static final long serialVersionUID = 3751202026782556556L;

    private static final String CCS_PREFIX = "ccs_";

    private List<String> uniqueNames = new ArrayList();

    // The name of a table derived from an Entity.
    // It's the class name, with the first character lowercased, all prefixed by
    // "ccs_"
    @Override
    public Identifier determinePrimaryTableName(ImplicitEntityNameSource source) {
        String tableName = source.getEntityNaming().getJpaEntityName();
        tableName = Character.toLowerCase(tableName.charAt(0)) + tableName.substring(1);
        return checkIdentifier(new Identifier(
                CCS_PREFIX
                        + tableName,
                false));
    }

    @Override
    public Identifier determineForeignKeyName(ImplicitForeignKeyNameSource source) {
        // if ( source.getTableName().equals("ccs_rawData") &&
        // "true".equals(System.getProperty("org.lsst.ccs.localdb.useRawDataOldForeignKeyName"))
        // ) {
        // return new Identifier("FK3AC853D2F9EF45DB",false);
        // }

        return checkIdentifier(new Identifier(
                CCS_PREFIX
                        + shortName(source.getTableName().getText())
                        + "_"
                        + shortColumnsIdentifier(source.getColumnNames())
                        + "_"
                        + shortName(source.getReferencedTableName().getText())
                        + "_fk",
                false), false);
    }

    @Override
    public Identifier determineUniqueKeyName(ImplicitUniqueKeyNameSource source) {
        return checkIdentifier(new Identifier(
                CCS_PREFIX
                        + shortName(source.getTableName().getText())
                        + "_"
                        + shortColumnsIdentifier(source.getColumnNames())
                        + "_u",
                false));
    }

    // For some reason this is called twice, that's why checkIdentifier takes a
    // boolean.
    @Override
    public Identifier determineIndexName(ImplicitIndexNameSource source) {
        return checkIdentifier(new Identifier(
                CCS_PREFIX
                        + shortName(source.getTableName().getText())
                        + "_"
                        + shortColumnsIdentifier(source.getColumnNames())
                        + "_i",
                false), false);
    }

    // Bridge table names
    // For some reason this is called twice, that's why checkIdentifier takes a
    // boolean.
    @Override
    public Identifier determineJoinTableName(ImplicitJoinTableNameSource source) {
        return checkIdentifier(new Identifier(
                CCS_PREFIX
                        + shortName(source.getOwningPhysicalTableName())
                        + "_"
                        + shortName(source.getNonOwningPhysicalTableName()) + "_b",
                false), false);
    }

    @Override
    public Identifier determineJoinColumnName(ImplicitJoinColumnNameSource source) {
        String columnName = super.determineJoinColumnName(source).getText();
        if (columnName.endsWith("_id")) {
            columnName = columnName.substring(0, columnName.length() - 3) + "Id";
        }
        columnName = Character.toLowerCase(columnName.charAt(0)) + columnName.substring(1);
        return new Identifier(columnName, false);
    }

    private static String shortColumnsIdentifier(List<Identifier> cols) {
        String result = "";
        for (Identifier col : cols) {
            if (!result.isEmpty()) {
                result += "_";
            }
            result += shortName(col.getText());
        }
        return result;
    }

    /**
     * Return a short version of the input name. Given a name in the
     * form: ccs_SomeTableName this function will:
     * - strip any leading "ccs_"
     * - lowercase the first letter and takes up to 9 letters until the second
     * uppercase
     * - strip a trailing "_b"
     * - add the first two letters of any additional Uppercased section
     *
     * @param tableName The input table name
     * @return The short version of the name
     */
    protected static String shortName(String tableName) {
        String result = "";
        tableName = tableName.replace(CCS_PREFIX, "");

        boolean add = false;
        int nChars = 1;
        int nCharsFirstWord = 7;
        boolean firstWord = true;
        int counter = 0;

        for (int i = 0; i < tableName.length(); i++) {
            char c = tableName.charAt(i);
            if (i == 0) {
                result += Character.toLowerCase(c);
                add = true;
            } else if (Character.isUpperCase(c)) {
                firstWord = false;
                result += c;
                counter = 0;
                add = true;
            } else if (add) {
                result += c;
                counter++;
            }
            if (firstWord && counter == nCharsFirstWord || !firstWord && counter == nChars) {
                add = false;
                counter = 0;
            }

        }
        if (result.endsWith("_b")) {
            result = result.replace("_b", "");
        }
        return result;
    }

    private Identifier checkIdentifier(Identifier id) {
        return checkIdentifier(id, true);
    }

    private Identifier checkIdentifier(Identifier id, boolean checkUnique) {
        if (!uniqueNames.contains(id.getText())) {
            uniqueNames.add(id.getText());
        } else if (checkUnique) {
            throw new RuntimeException("ERROR: generated name " + id.getText() + " already exists.");
        }
        if (id.getText().length() > 64) {
            throw new RuntimeException("ERROR: name exceeds maximum length of 64 chars " + id.getText());
        }
        return id;
    }

}
