package org.lsst.ccs.localdb.statusdb.model;

import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import javax.persistence.Cacheable;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.Table;
import javax.persistence.UniqueConstraint;

import org.hibernate.annotations.Immutable;
import org.hibernate.annotations.NaturalId;

/**
 *
 * @author LSST CCS Team
 */
@Entity
@Immutable
@Cacheable
@org.hibernate.annotations.Cache(
        usage = org.hibernate.annotations.CacheConcurrencyStrategy.READ_ONLY, 
        region = "org.lsst.ccs.localdb.statusdb.model.AgentState"
)
@Table(
        uniqueConstraints = @UniqueConstraint(columnNames = {"agentName", "hashMD5"})
)
@org.hibernate.annotations.NaturalIdCache
public class AgentState implements Serializable {
	private static final long serialVersionUID = -8367786148299296222L;

	private long id;

	private AgentDesc agentDesc;

	// key = component name, or "" for top (agent) states
	private Map<String, StateBundleDesc> componentStates = new HashMap<>();

	private byte[] hashMD5;

	protected AgentState() {

	}

	public AgentState(AgentDesc ad) {
		this.agentDesc = ad;
	}

	@Id
        @GeneratedValue(generator = "ID_SEQUENCE_GENERATOR")
	public long getId() {
		return id;
	}

	public void setId(long id) {
		this.id = id;
	}

        @NaturalId(mutable = false)
	@ManyToOne(optional = false, fetch = FetchType.LAZY)
	@JoinColumn(name = "agentName")
	public AgentDesc getAgentDesc() {
		return agentDesc;
	}

	public void setAgentDesc(AgentDesc agentDesc) {
		this.agentDesc = agentDesc;
	}

	@ManyToMany(fetch = FetchType.LAZY)
	public Map<String, StateBundleDesc> getComponentStates() {
		return componentStates;
	}

	public void setComponentStates(Map<String, StateBundleDesc> componentStates) {
		this.componentStates = componentStates;
	}
        
        public static byte[] computeMD5(String agentName, Map<String, StateBundleDesc> stateBundleDescs) {
            StringBuilder sb = new StringBuilder();
            sb.append(agentName).append("*");
            for (Map.Entry<String, StateBundleDesc> e : new TreeMap<>(stateBundleDescs).entrySet()) {
                String c = e.getKey();
                byte[] md5 = e.getValue().getHashMD5();
                sb.append(c).append("/").append(Arrays.toString(md5)).append(":");
            }
            try {
                MessageDigest md = MessageDigest.getInstance("MD5");
                return md.digest(sb.toString().getBytes("UTF-8"));
            } catch (NoSuchAlgorithmException | UnsupportedEncodingException e1) {
                throw new RuntimeException(e1);
            }
        }

        @NaturalId(mutable = false)
        @Column(columnDefinition = "varbinary(100)", nullable = false)
        public byte[] getHashMD5() {
		return hashMD5 == null ? null : Arrays.copyOf(hashMD5, hashMD5.length);
	}

	public void setHashMD5(byte[] hashMD5) {
		this.hashMD5 = hashMD5 == null ? null : Arrays.copyOf(hashMD5, hashMD5.length);
	}

        /**
	 * Returns a map of state values map to their class name.
	 * 
	 * @param components
	 * @return
         */
        public Map<String, Map<String, String>> asFlatStatesMap(List<String> components) {
            Map<String, Map<String, String>> res = new HashMap<>();
		if (components == null || components.isEmpty() || components.contains("")) {
                    if (components != null && components.size() == 1) {
                        return res;
                    }
                    for (Map.Entry<String, StateBundleDesc> entry : componentStates.entrySet()) {
                        Map<String, String> flatMap = res.get(entry.getKey());
                        if (flatMap == null) {
                            flatMap = new HashMap<>();
                            res.put(entry.getKey(), flatMap);
                        }
                        for (InnerStateDesc isd : entry.getValue().getComponentStates().values()) {
                            flatMap.put(isd.getEnumClassName(), isd.getEnumValue());
                        }
                    }
                    return res;
                } else {
                    for (String component : components) {
                        StateBundleDesc sbd = componentStates.get(component);
                        if (sbd != null) {
                            Map<String, String> flatMap = new HashMap<>();
                            for (InnerStateDesc isd : sbd.getComponentStates().values()) {
                                flatMap.put(isd.getEnumClassName(), isd.getEnumValue());
                            }
                            res.put(component, flatMap);
                        }
                    }
                    return res;
                }
        }

	@Override
        public String toString() {
            StringBuilder sb = new StringBuilder(agentDesc.getAgentName()).append(":\n\t");
            Map<String, Map<String, String>> flatMap = new TreeMap<>(asFlatStatesMap(null));
            Map<String, String> topStates = flatMap.remove("");
            for(Map.Entry<String, String> sbd : topStates.entrySet()) {
                sb.append(sbd.getKey()).append(":").append(sbd.getValue()).append(", ");
            }
            sb.append("\n");
            for(Map.Entry<String, Map<String, String>> entry : flatMap.entrySet()) {
                sb.append("\t").append(entry.getKey()).append(": ");
                for(Map.Entry<String, String> sbd : entry.getValue().entrySet()) {
                    sb.append(sbd.getKey()).append(":").append(sbd.getValue()).append(", ");
                }
                sb.append("\n");
            }
            return sb.toString();
        }
}
