package org.lsst.ccs.localdb.dao;

import java.util.HashMap;
import java.util.Map;
import org.hibernate.FlushMode;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.localdb.statusdb.model.AgentDesc;
import org.lsst.ccs.localdb.statusdb.model.AlertDesc;
import org.lsst.ccs.localdb.statusdb.model.StatTimeInterval;
import org.lsst.ccs.localdb.statusdb.model.AgentState;
import org.lsst.ccs.localdb.statusdb.model.InnerStateDesc;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.model.StateBundleDesc;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Simplistic access to the database, through static methods.
 * 
 * @author LSST CCS Team
 */
public class LocaldbFacade {

	private LocaldbFacade() {
	}

	private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.statusdb");

	/**
	 * Returns the persistent AgentDesc corresponding to the AgentInfo. If not found
	 * in the first place, creates it and persists it.
	 * 
	 * @param agentInfo
	 * @param sess
	 * @return an AgentDesc in persisted state.
	 */
	public static AgentDesc getAgentDesc(AgentInfo agentInfo, Session sess) {
		String key = agentInfo.getName();
		AgentDesc ad = sess.get(AgentDesc.class, key);

		if (ad == null) {
			ad = new AgentDesc(agentInfo);
			sess.persist(ad);
			log.debug("Added Agent description for " + key + " : " + ad.getAgentName());
		}
		return ad;
	}

	/** -- Methods for Alert persistence. */

	/**
	 *
	 * @param agentDesc
	 * @param alert
	 * @param sess
	 * @return an AlertDesc in persisted state.
	 */
	public static AlertDesc getAlertDescOrPersist(AgentDesc agentDesc, Alert alert, Session sess) {
		AlertDesc ad = getAlertDesc(agentDesc, alert.getAlertId(), sess);
		if (ad == null) {
			ad = new AlertDesc();
			ad.setAgentDesc(agentDesc);
			ad.setAlertId(alert.getAlertId());
			ad.setAlertDescription(alert.getDescription());
			sess.persist(ad);
			log.debug("Added Alert description for " + agentDesc.getAgentName() + "/" + alert.getAlertId() + " : "
					+ ad.getAlertId());
		}
		return ad;
	}

	/**
	 * Looks for a persisted AlertDesc corresponding to the given parameters.
	 * Returns null if not found.
	 * 
	 * @param agentDesc
	 * @param alertId
	 * @param sess
	 * @return the corresponding AlertDesc, or null if not found.
	 */
	public static AlertDesc getAlertDesc(AgentDesc agentDesc, String alertId, Session sess) {
		Query q = sess.getNamedQuery("findAlertDesc");
		q.setString("id", alertId).setString("name", agentDesc.getAgentName());
		return (AlertDesc) q.uniqueResult();
	}

	/** -- Methods for State persistence. */

	/**
	 * Returns a persisted InnerStateDesc corresponding to the given parameters.
	 * 
	 * @param ai
	 * @param enumClassName
	 * @param enumValue
	 * @param sess
	 * @return an InnerStateDesc in persisted state.
	 */
	private static InnerStateDesc getInnerStateDesc(String enumClassName, String enumValue,
			Session sess) {
		Query q = sess.getNamedQuery("findInnerStateDesc");
		q.setString("enumClassName", enumClassName).setString("enumValue", enumValue);
		InnerStateDesc isd = (InnerStateDesc) q.uniqueResult();
		if (isd == null) {
			isd = new InnerStateDesc();
			isd.setEnumClassName(enumClassName);
			isd.setEnumValue(enumValue);
			sess.persist(isd);
			log.debug("Added Inner State description description for " + isd.toString());
		}
		return isd;
	}

	/**
	 * Reads the internal states of a StateBundle and returns a StateBundleDesc
	 * entity in persisted state.
	 *
	 * @param ad
	 * @param sb
	 * @param existingSBDs
	 * @param sess
	 * @return
	 */
	private static StateBundleDesc getStateBundleDesc(Map<String, String> states, Session sess) {
            
            Map<String, InnerStateDesc> componentStates = new HashMap<>();
            
            for (Map.Entry<String, String> state : states.entrySet()) {
                componentStates.put(state.getKey(), getInnerStateDesc(state.getKey(), state.getValue(), sess));
            }
            
            byte[] md5 = StateBundleDesc.computeMD5(componentStates);
            
            Query q = sess.getNamedQuery("findStateBundleDesc").setBinary("md5", md5);
            StateBundleDesc sbd = (StateBundleDesc)q.uniqueResult();
            if (sbd == null) {
                sbd = new StateBundleDesc();
                sbd.setComponentStates(componentStates);
                sbd.setHashMD5(md5);
                sess.persist(sbd);
            }
            return sbd;
	}

        /**
         * Returns the AgentState corresponding to the given stateBundle. An
         * AgentState is the combination of all component states. The top states are
         * associated to a component named ""
         *
         * @param ad the agent description in persisted state.
         * @param sb the StateBundle to be persisted.
         * @param sess the current session.
         * @return an AgentState in persisted state.
         */
        public static AgentState getAgentState(AgentDesc ad, StateBundle sb, Session sess) {
            
            // Computing the AgentState md5
            AgentState as = null;            
            
            Map<String, StateBundleDesc> compStates = new HashMap<>();
            
            Map<String, String> topStates = new HashMap<>(sb.getAllStatesAsStrings());
            
            compStates.put("", getStateBundleDesc(topStates, sess));
                
            for(String comp : sb.getComponentsWithStates()) {
                compStates.put(comp, getStateBundleDesc(sb.getComponentStateBundle(comp).getAllStatesAsStrings(), sess));
            }
            
            byte[] md5 = AgentState.computeMD5(ad.getAgentName(), compStates);
            
            // Looking in database by md5 
            as = sess.byNaturalId(AgentState.class)
                    .using("agentDesc", ad)
                    .using("hashMD5", md5)
                    .load();
                    
            if (as == null) {
                as = new AgentState(ad);
                as.setComponentStates(compStates);
                as.setHashMD5(md5);
                sess.persist(as);
                log.debug("Added AgentState for " + as.toString());
            }
            return as;
        }
        
//        public static synchronized StatTimeInterval getStatTimeInterval(long binWidth, long dataTime) {
//            // Get the time intervals spanning the span (t_start,t_end).
//            // This means startTime<=t_end and startTime+binWidth>t_start.
//            // Which becomes startTime<=t_end amd startTime>t_start-binWidth.
//            // Or startTime<=t2 and statTime>t1 where t2=t_end and t1=t_start-binWidth
//		long statTimeIntervalStart = (dataTime / binWidth) * binWidth;
//
//                Session sess = StatusdbUtils.getSessionForTimeIntervals();
//		// Get The time interval for this binWidth
//		StatTimeInterval statTimeInterval = sess.byNaturalId(StatTimeInterval.class)
//				.using("startTime", statTimeIntervalStart).using("binWidth", binWidth).load();
//
//		if (statTimeInterval == null) {
//                    Transaction tx = sess.beginTransaction();
//                    statTimeInterval = new StatTimeInterval(statTimeIntervalStart, binWidth);
//                    sess.persist(statTimeInterval);
//                    // No need to flush because StatTimeInterval is queried by naturalId
//                    // and therefore repeatedly readable from session cache.
//                    tx.commit();
//                    sess.close();
//		}
//		return statTimeInterval;
//	}

        public static synchronized StatTimeInterval getStatTimeInterval(long binWidth, long dataTime) {
		long statTimeIntervalStart = (dataTime / binWidth) * binWidth;

                Session s = StatusdbUtils.getSessionForTimeIntervals();
		// Get The time interval for this binWidth
		StatTimeInterval statTimeInterval = (StatTimeInterval)s.getNamedQuery("findStatTimeInterval").setLong("st", statTimeIntervalStart).setLong("bw", binWidth).setCacheable(true)
				.uniqueResult();

		if (statTimeInterval == null) {
                    Transaction tx = s.beginTransaction();
                    statTimeInterval = new StatTimeInterval(statTimeIntervalStart, binWidth);
                    s.persist(statTimeInterval);
                    tx.commit();
                    s.close();
		}
		return statTimeInterval;
	}

        /**
	 * Retrieves a StatData entity corresponding to the given {@code StatDesc} and
	 * {@code StatTimeInterval}.
	 * 
	 * @param sd
	 *            The StatDesc entity
	 * @param sti
	 *            the StatTimeInterval
	 * @return a StatData or null if no StatData was found
	 */
	public static StatData getStatDataOrNull(StatDesc sd, StatTimeInterval sti, Session sess) {
		return (StatData) sess.getNamedQuery("findStatData").setEntity("d", sd).setEntity("sti", sti).setCacheable(true)
				.uniqueResult();
	}        
}
