package org.lsst.ccs.localdb.configdb.model;

import java.util.ArrayList;
import java.util.HashMap;
import org.hibernate.Session;
import org.hibernate.SessionFactory;

import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;
import org.hibernate.CacheMode;
import org.hibernate.FlushMode;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Restrictions;
import org.hibernate.resource.transaction.spi.TransactionStatus;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.localdb.statusdb.model.AgentDesc;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Low level Data Access to Configuration Objects. An hibernate connection is
 * started using {@code begin} and ended with {@code end}. Between those calls
 * the hibernate session is persistent. An hibernate session context (session
 * and transaction) is confined to the thread that opens it, by the use of a
 * {@code ThreadLocal}
 *
 * @author LSST CCS Team
 */
public class HibernateDAO {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.configdb");

    private final SessionFactory sessionFactory;
    
    private final ThreadLocal<SessionContext> context;
    
    private static class SessionContext {
        
        private final Session session;
        private org.hibernate.Transaction transaction;
        
        private SessionContext(SessionFactory s) {
            session = s.openSession();
            session.setFlushMode(FlushMode.COMMIT);
        }
        
        private void beginTransaction() {
            transaction = session.beginTransaction();
        }
    }
    
    public HibernateDAO(SessionFactory fac) {
        sessionFactory = fac;
        
        context = new ThreadLocal<SessionContext>() {
            public SessionContext initialValue() {
                return new SessionContext(sessionFactory);
            }
        };
    }
    
    public void begin() {
        log.debug("begin transaction");
        context.get().beginTransaction();
    }

    /**
     * End of a unit of work : transaction commit.
     * @throws Exception if the persistence layer fails at committing 
     */
    public void end() throws Exception {
        log.debug("end transaction");
        context.get().transaction.commit();
        closeSession();
    }
    
    public void rollback() throws Exception {
        Transaction currTx = context.get().transaction;
        try {
            if (currTx!=null && ( currTx.getStatus() == TransactionStatus.ACTIVE || 
                    currTx.getStatus() == TransactionStatus.MARKED_ROLLBACK)) {
                currTx.rollback();
            }
        } catch (Exception rbEx) {
            log.error("Rollback of transaction failed : " + rbEx, rbEx);
            throw rbEx;
        }
    }
    
    public void closeSession() {
        if (getSession().isOpen()) {
            getSession().close();
        }
        context.remove();
    }
    
    public void persist(Object obj) {
        getSession().persist(obj);
    }
    
    private Session getSession() {
        return context.get().session;
    }
    
    public Configuration getConfigurationOrNull(BaseDescription desc, String name, String cat, int version) {
        Configuration res = null;
        Criterion baseCriterion = Restrictions.conjunction(
                Restrictions.eq("baseDescription", desc),
                Restrictions.eq("configName", name),
                Restrictions.eq("category", cat));
        
        switch(version) {
            case ConfigurationInfo.DEFAULT_VERSION:
                // getting the default configuration
                res = (Configuration)getSession().createCriteria(Configuration.class)
                        .add(baseCriterion)
                        .add(Restrictions.eq("defaultVersion", true))
                        .uniqueResult();
                if (res == null) {
                    log.fine("no default version for configuration " + desc.getAgentDesc()+":"+cat+":"+name+". Getting the latest.");
                } else {
                    break;
                }
            
            case ConfigurationInfo.LATEST_VERSION:
                res =  getLatestConfiguration(desc, cat, name);
                break;
            default:
                res =  (Configuration)getSession().createQuery("from Configuration where baseDescription=:desc and configName=:configName and category=:cat and version=:ver")
                        .setEntity("desc", desc)
                        .setString("configName", name)
                        .setString("cat", cat)
                        .setInteger("ver", version)
                        .setCacheable(true)
                        .uniqueResult();
        }
        if (res != null) {
            log.fine("found versioned configuration " + res.getConfigurationDescriptionString());
        }
        return res;
    }
    
    private Configuration getConfigurationOrNull(BaseDescription desc, String name, String cat, byte[] md5) {
        Configuration res = (Configuration)getSession().createQuery("from Configuration where baseDescription=:description and configName =:tag and category=:cat and hashMD5 =:hashMD5")
                .setEntity("description", desc).setString("tag", name).setString("cat", cat).setBinary("hashMD5", md5)
                .setCacheable(true)
                .uniqueResult();
        if (res != null) {
            log.fine("found matching configuration " + res.getConfigurationDescriptionString());
        }
        return res;
    }
    
    /**
     * Cascading process to determine which configuration is running. If the version
     * number is well defined, it is fetched by version number, otherwise its
     * checksum is computed. If not found, a new version for the given tag is
     * created.
     *
     * @param desc
     * @param name
     * @param cat
     * @param version
     * @param cpis
     * @param date
     * @param previousRun
     * @return
     */
    public Configuration getConfigurationOrCreate(BaseDescription desc, String name, String cat, int version, List<ConfigurationParameterInfo> cpis, long date, ConfigurationRun previousRun) {
        String request = desc.getId() + ": \""+cat+"\":\""+name+"\"("+version+")";
        log.debug("looking for configuration " + request);
        Configuration res = null;
        if (version != ConfigurationInfo.UNDEF_VERSION) {
            // There should be a version in db
            res = getConfigurationOrNull(desc, name, cat, version);
            if (res == null) {
                log.warn("a versioned configuration was not found : " + request + ". creating a new one");
            }
        }
        byte[] cs = null;
        if (res == null) {
            // No version number : looking by checksum
            // Final parameters are discarded from this map.
            TreeMap<String, String> currentValues = new TreeMap<>();
            for (ConfigurationParameterInfo cpi : cpis) {
                if(!cpi.isFinal()) {
                    currentValues.put(new ParameterPath(cpi.getComponentName(), cpi.getParameterName()).toString(), cpi.getConfiguredValue());
                }
            }
            try {
                cs = Configuration.computeHashMD5(desc, currentValues);
            } catch(Exception ex) {
                log.error("cannot compute configuration checksum", ex);
            }
            if (cs != null) {
                res = getConfigurationOrNull(desc, name, cat, cs);
            }
        }
        
        // TO-DO : Before creating a brand new configuration we should first compare cpis with all existing configurations with same name and category
        if (res == null) {
            res = createNextConfiguration(desc, name, cat, date, cs);
            
            if(previousRun == null) {
                for(ConfigurationParameterInfo cpi : cpis) {
                    if(!cpi.isFinal()) {
                        res.addConfigurationParameterValue(createConfigurationParameterValue(desc, cpi.getPathName(), cpi.getConfiguredValue(), date));
                    }
                }
            } else {
                for(ConfigurationParameterInfo cpi : cpis) {
                    if(!cpi.isFinal()) {
                        ParameterPath path = ParameterPath.valueOf(cpi.getPathName());
                        ConfigurationParameter cp = desc.getConfigurationParameters().get(path);
                        ConfigurationParameterValue previousValue = previousRun.getConfigurationParameterValueForRun(path);
                        if(cpi.getCurrentValue().equals(previousValue.getValue())) {
                            res.addConfigurationParameterValue(previousValue);
                        } else {
                            log.warn("parameter : " + cp.getParameterPath() + " had the following value in the previous run : " + previousValue.getValue() + " but is now : " + cpi.getCurrentValue());
                            res.addConfigurationParameterValue(createConfigurationParameterValue(desc, cpi.getPathName(), cpi.getConfiguredValue(), date));
                        }
                    }
                }
            }
        }
        return res;
    }
    
    public ConfigurationRun populateRunFromScratch(ConfigurationRun run, List<ConfigurationParameterInfo> changes) {
        for (ConfigurationParameterInfo cpi : changes) {
            ConfigurationParameterValue dirtyVal = createConfigurationParameterValue(run.getConfiguration().getBaseDescription(), cpi.getPathName(), cpi.getCurrentValue(),run.getTstart());
            run.addRuntimeChange(dirtyVal);
        }
        return persistNextConfigurationRun(run,null);
    }
    
    public ConfigurationRun populateRunFromPreviousRun(ConfigurationRun previousRun, ConfigurationRun newRun, List<ConfigurationParameterInfo> changes) {
        Configuration prevConfig = previousRun.getConfiguration();
        Configuration newConfig = newRun.getConfiguration();
        if(changes.isEmpty()) {
            GlobalConfiguration newGC = newRun.getGlobalConfiguration();
            GlobalConfiguration prevGC = previousRun.getGlobalConfiguration();
            // newRun is intended to be configured
            if ((newGC != null && prevGC != null && newGC.getId() == previousRun.getId()) || (newGC == null && prevGC == null)) {
                if (!previousRun.isDirty() && prevConfig.getConfigName().equals(newConfig.getConfigName()) && prevConfig.getVersion() == newConfig.getVersion()) {
                    return previousRun;
                }
            }
            return persistNextConfigurationRun(newRun, previousRun);
        } else {
            if (!prevConfig.getConfigName().equals(newConfig.getConfigName())) {
                log.warning("inconsistency detected");
                previousRun.setTstop(newRun.getTstart());
                return populateRunFromScratch(newRun, changes);
            } else {
                // Figuring out the delta of changes compared with the latest run
                // TO-DO : this could also be done using ci.getLatestChanges()
                List<ConfigurationParameterValue> delta = new ArrayList<>();
                for (ConfigurationParameterInfo cpi : changes) {
                    ParameterPath path = ParameterPath.valueOf(cpi.getPathName());
                    String currentValue = cpi.getCurrentValue();
                    ConfigurationParameterValue previousChange = previousRun.getRuntimeChanges().get(path);
                    if (previousChange == null || !previousChange.getValue().equals(currentValue)) {
                        // This is a new change
                        ConfigurationParameterValue cpv = createConfigurationParameterValue(newRun.getConfiguration().getBaseDescription(), cpi.getPathName(), currentValue, newRun.getTstart());
                        delta.add(cpv);
                    }
                }
                if (delta.isEmpty()) {
                    return previousRun;
                } else {
                    for (ConfigurationParameterValue cpv : delta) {
                        newRun.addRuntimeChange(cpv);
                    }
                    return persistNextConfigurationRun(newRun, previousRun);
                }
            }
        }
    }
    
    private Configuration createNextConfiguration(BaseDescription desc, String name, String cat, long date, byte[] cs) {
        Configuration latestConfiguration = getLatestConfiguration(desc, cat, name);
        int latestVersion = -1;
        if (latestConfiguration != null) {
            latestVersion = latestConfiguration.getVersion();
            latestConfiguration.setLatestVersion(false);
        }
        int nextVersion = latestVersion+1;
        Configuration res = new Configuration(desc, cat, name, date, nextVersion);
        log.fine("created new configuration : " + res);
        res.setHashMD5(cs);
        getSession().persist(res);
        return res;
    }
    
    private ConfigurationRun persistNextConfigurationRun(ConfigurationRun nextRun, ConfigurationRun previousRun) {
        if(previousRun != null) {
            previousRun.setTstop(nextRun.getTstart());
        }
        getSession().persist(nextRun);
        return nextRun;
    }
    
    private Configuration getLatestConfiguration(BaseDescription desc, String cat, String name) {
        Configuration latestConfiguration = (Configuration)getSession().createQuery(
                "from Configuration c where c.baseDescription=:desc and c.configName=:name and c.category=:cat and c.version="
                        + "(select max(co.version) from Configuration co where co.baseDescription=:desc and co.configName=:name and co.category=:cat)")
                .setEntity("desc", desc).setString("name", name).setString("cat", cat).uniqueResult();
        if(latestConfiguration == null) {
            log.debug("no configuration for \"" + cat+"\":\""+name+ "\" for description " + desc.getId());
        }
        return latestConfiguration;
    }
    
    private ConfigurationParameterValue createConfigurationParameterValue(BaseDescription desc, String path, String value, long time) {
        return new ConfigurationParameterValue(desc.fetch(path), value, time);
    }
    
    /**
     * If Configuration runs are found with no tstop, they are considered as
     * suspicious : they are removed.
     *
     * @param desc
     */
    public void cleanupConfigurationRuns(Description desc, long time) {
        Map<String, ConfigurationRun> activeRuns = getActiveConfigurationRuns(desc);
            // Those runs should have had their tstop set
            for(ConfigurationRun cr : activeRuns.values()) {
                log.info("suspicious run found for " + desc.getAgentDesc().getAgentName()+ ", starting at " + cr.getTstart() + " with no end date.");
                cr.setTstop(time);
                getSession().persist(cr);
            }
    }
    
    public Configuration getLatestRunningConfiguration(Description desc, String category) {
        return getConfigRunningAt(desc, category, PackCst.STILL_VALID);
    }
    
    public Configuration getConfigRunningAt(Description desc, String category, long date) {
        ConfigurationRun run = getConfigurationRunAt(desc, category, date);
        if (run == null) {
            return null;
        } 
        return run.getConfiguration();
    }
    
    public String getActiveValueAt(Description desc, String category, String parameterPath, long date) {
        ConfigurationRun run = getConfigurationRunAt(desc, category, date);
        if (run == null) {
            return null;
        } 
        return run.getConfigurationParameterValueForRun(ParameterPath.valueOf(parameterPath)).getValue();
    }

    public void endRun(Description description, long endTime) {
        Map<String,ConfigurationRun> currentRuns = getActiveConfigurationRuns(description);
        for(ConfigurationRun cr : currentRuns.values()) {
            cr.setTstop(endTime);
            getSession().update(cr);
        }
    }
        
    public ConfigurationParameter createConfigurationParameter(String agentName, String componentName, String parameterName, String description, String typeName, String categoryName, String finalValue) {
        log.fine("creating a new configuration parameter entity for " + agentName + " : " + componentName+"/"+parameterName + " cat : " + categoryName);
        ConfigurationParameter cp = new ConfigurationParameter(agentName,
                new ParameterPath(componentName, parameterName), typeName, description, categoryName, finalValue);
        getSession().setFlushMode(FlushMode.COMMIT);
//        getSession().persist(cp);
        return cp;
    }
    
    public Map<ParameterPath, List<ConfigurationParameter>> getAllConfigurationParameters(String agentName) {
        List<ConfigurationParameter> l = getSession().createQuery("from ConfigurationParameter where agentName=:agentName")
                .setCacheMode(CacheMode.IGNORE).setFlushMode(FlushMode.COMMIT)
                .setString("agentName", agentName).list();
        Map<ParameterPath, List<ConfigurationParameter>> res = new HashMap<>();
        for (ConfigurationParameter cp : l) {
            List<ConfigurationParameter> list = res.get(cp.getParameterPath());
            if(list == null) {
                list = new ArrayList<>();
                res.put(cp.getParameterPath(), list);
            }
            list.add(cp);
        }
        return res;
    }
    
    /**
     * Get an existing Description entity or creates it if it does not exist already.
     * @param ad
     * @param parameters
     * @return 
     */
    public Description getDescription(AgentDesc ad, List<ConfigurationParameter> parameters) {
        Description res = null;
        byte[] md5 = BaseDescription.computeHashMD5(ad, parameters);
        
        // Finding if a base description exists.
        BaseDescription desc = getBaseDescription(md5);
        if(desc == null) {
            desc = createBaseDescription(ad, parameters, md5);
            res = createDescription(desc, parameters);
        } else {
            res = getDescription(desc, parameters);
        }
        if(res == null) {
            res = createDescription(desc, parameters);
        }
        return res;
    }
    
    private BaseDescription getBaseDescription(byte[] md5) {
        BaseDescription res =  (BaseDescription)getSession().createQuery("from BaseDescription where hashMD5=:md5").setBinary("md5", md5).uniqueResult();
        if (res != null) {
            log.fine("found matching base description for " + res.getAgentDesc().getAgentName() + " : " + res.getId());
        }
        return res;
    }
    
    /**
     * Finds or creates a {@code BaseDescription} entity corresponding to the input arguments.
     * @param ad the agent description
     * @param parameters the parameters in persisted state
     * @param md5
     * @return a {@code BaseDescription} object in persisted state.
     */
    private BaseDescription createBaseDescription(AgentDesc ad, List<ConfigurationParameter> parameters, byte[] md5) {
        getSession().update(ad);
        BaseDescription desc = new BaseDescription(ad);
        // A new description has to be created
        log.fine("creating a new base description for " + ad.getAgentName());
        desc = new BaseDescription(ad);
        Map<ParameterPath, ConfigurationParameter> nonFinalParameters = new HashMap<>();
        for(ConfigurationParameter cp : parameters) {
            if (!cp.isFinal()) {
                nonFinalParameters.put(cp.getParameterPath(), cp);
            }
        }
        desc.setConfigurationParameters(nonFinalParameters);
        desc.setHashMD5(md5);
        getSession().persist(desc);
        return desc;
    }
    
    public Description getDescription(long id) {
        return getSession().get(Description.class, id);
    }
    
    private Description getDescription(BaseDescription desc, List<ConfigurationParameter> parameters) {
        Description res = null;
        Map<ParameterPath, ConfigurationParameter> finalValues = new HashMap<>();
        for(ConfigurationParameter cp : parameters) {
            if(cp.isFinal()) {
                finalValues.put(cp.getParameterPath(), cp);
            }
        }
        for(Description cd : desc.getCompatibleDescriptions()) {
            if(finalValues.equals(cd.getFinalValues())) {
                res = cd;
                break;
            }
        }
        return res;
    }
    
    private Description createDescription(BaseDescription desc, List<ConfigurationParameter> parameters) {
        Description res = new Description(desc);
        Map<ParameterPath, ConfigurationParameter> finalValues = new HashMap<>();
        for(ConfigurationParameter cp : parameters) {
            if (cp.isFinal()) {
                finalValues.put(cp.getParameterPath(), cp);
            }
        }
        res.setFinalValues(finalValues);
        getSession().persist(res);
        return res;
    }
    
    public ConfigurationRun getActiveConfigurationRun(Description desc, String category) {
        ConfigurationRun cr = (ConfigurationRun)getSession().createQuery("from ConfigurationRun where description=:desc and configuration.category=:cat and tstop=:tstop")
                .setEntity("desc", desc).setLong("tstop", PackCst.STILL_VALID)
                .setString("cat", category).uniqueResult();
        return cr;
    }
    

    public ConfigurationInfoData getConfigurationInfoData(Description desc, long date) {
        ConfigurationInfoData res = (ConfigurationInfoData)getSession().createQuery("from ConfigurationInfoData cid where cid.description=:desc and cid.time <=:date order by cid.time desc")
                .setEntity("desc", desc).setLong("date", date).setMaxResults(1).uniqueResult();
        if (res == null) {
            log.info("no ConfigurationInfoData found for date " + date);
        }
        return res;
    }
    
    public ConfigurationRun getConfigurationRunAt(Description desc, String category, long date) {        
        ConfigurationInfoData cid = getConfigurationInfoData(desc, date);
        ConfigurationRun run = null;
        if (cid != null ) {
            run = cid.getConfigurationRuns().get(category);
            if (run.getTstop() <= date) {
                log.warn("for date : " + date + ", CID was found but its run ends at " + run.getTstop());
                run = null;
            }
        }
        if (run == null) {
            log.warning( "no configuration run at " + date + " for subsystem " + desc.getAgentDesc().getAgentName() + " and category " + category);
        }
        return run;
    }
    
    public Map<String, ConfigurationRun> getActiveConfigurationRuns(Description desc) {
        List<ConfigurationRun> list = getSession().createQuery("from ConfigurationRun where description=:description and tstop=:tstop")
                .setEntity("description", desc)
                .setLong("tstop", PackCst.STILL_VALID).list();
        return list.stream().collect(Collectors.toMap((cr)->cr.getConfiguration().getCategory(), (cr)->cr));
    }
    
    public List<String> findAvailableConfigurationsForCategory(BaseDescription desc, String category) {
        return getSession().createQuery("select configName from Configuration where description=:desc and category=:cat")
                .setEntity("desc", desc).setString("cat",category).list();
    }
    
    public GlobalConfiguration findGlobalConfiguration(BaseDescription desc, String name, int version) {
        if( version < 0) {
            return (GlobalConfiguration)getSession().createQuery("from GlobalConfiguration where description=:desc and name=:name and latest=true")
                    .setEntity("desc", desc).setString("name", name).uniqueResult();
        } else {
            return (GlobalConfiguration)getSession().createQuery("from GlobalConfiguration where description=:desc and name=:name and version=:version")
                    .setEntity("desc", desc).setString("name", name).setInteger("version", version).uniqueResult();
        }
    }

    public GlobalConfiguration findGlobalConfigurationOrCreate(BaseDescription desc, String name, Map<String, Configuration> newConfigs) {
        List<GlobalConfiguration> list = getSession().createQuery("from GlobalConfiguration where description=:desc and name=:name")
                .setEntity("desc", desc).setString("name", name)
                .list();
        int nver = 0;
        GlobalConfiguration latestGC = null;
        if (!list.isEmpty()) {
            nver = list.size();
            for (GlobalConfiguration gc : list) {
                if (gc.isLatest()) {
                    latestGC = gc;
                }
                boolean equal = true;
                for(Configuration c : gc.getConfigurations().values()) {
                    if(newConfigs.get(c.getCategory()).getId() != c.getId()) {
                        equal = false;
                        break;
                    }
                }
                if (equal) {
                    return gc;
                }
            }
        } 
        // Creating it
        if(latestGC != null) {
            latestGC.setLatest(false);
        }
        GlobalConfiguration gc = new GlobalConfiguration(desc, name, nver);
        gc.setConfigurations(newConfigs);
        gc.setLatest(true);
        getSession().persist(gc);
        return gc;
                
    }
}