package org.lsst.ccs.localdb.statusdb.server;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Collectors;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.lsst.ccs.localdb.statusdb.model.AlertDesc;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.InnerStateDesc;
import org.lsst.ccs.localdb.statusdb.model.MetaDataData;
import org.lsst.ccs.localdb.statusdb.model.RaisedAlertData;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.model.StateChangeNotificationData;
import org.lsst.ccs.localdb.statusdb.server.TrendingData.AxisValue;
import org.lsst.ccs.localdb.statusdb.server.TrendingData.DataValue;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.utilities.logging.Log4JConfiguration;
import org.lsst.ccs.utilities.logging.Logger;

import com.sun.jersey.spi.resource.Singleton;
import org.lsst.ccs.localdb.statusdb.model.DataPath;

/**
 * The DataServer receives requests for trending data analysis. It delivers the
 * requested data as the result of requests to the database.
 */
@SuppressWarnings("restriction")
@Path("/dataserver")
@Singleton
public class DataServer {

    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb");

    private final SessionFactory fac;

    static {
        /*
         * Temporary : setting of logging
         */
        Log4JConfiguration.initialize();
    }

    public DataServer() {
        Properties p = new Properties();
        fac = StatusdbUtils.getSessionFactory(p);
        log.fine("Starting Data Server");
    }

    public SessionFactory getSessionFactory() {
        return fac;
    }

    private List<DataChannel> getListOfChannels(long dtSeconds) {

        log.fine("Loading Channel Information");
        long start = System.currentTimeMillis();

        ArrayList<DataChannel> channels = new ArrayList<>();
        try (Session sess = fac.openSession()) {
            // select dataDescId, max(time) from ccs_rawData group by dataDescId
            //
            // select distinct dataDescId from (select dataDescId, max(time) m
            // from
            // ccs_rawData group by dataDescId) as xx where m>15249072119
            //
            // select distinct dataDescId from (select dataDescId from
            // ccs_rawData group by
            // dataDescId having max(time)>15249072119) as xx;

            // We don't want a join to happen in the SQL subquery. Using ids
            // prevents
            // the generated SQL to have a join with DataDesc
            Query dataQuery = sess.createQuery("from DataDesc as dd where dd.id in " //
                    + " (select d.dataDesc.id from RawData d " //
                    + " group by d.dataDesc.id having max(d.time)>:t)");

            if (dtSeconds > 0) {
                dataQuery.setParameter("t", System.currentTimeMillis() - dtSeconds * 1000L);
            } else {
                dataQuery.setParameter("t", 0L);
            }
            dataQuery.setReadOnly(true);
            @SuppressWarnings("unchecked")
            List<DataDesc> l = dataQuery.list();

            double delta = (System.currentTimeMillis() - start) / 1000.;
            log.fine("Done with the query " + l.size() + " in " + delta + " seconds");
            for (DataDesc d : l) {
                DataChannel dc = new DataChannel(d);
                dc.getMetadata().put("subsystem", d.getDataPath().getAgentName());
                dc.getMetadata().put("type", d.getDataType());
                channels.add(dc);
                log.debug("retrieving Data Channel path= " + dc.getPathAsString());
            }

        }
        return channels;
    }

    private long getDataIdForPath(String path) {
        if (path.startsWith("/")) {
            path = path.substring(1);
        }
        int index = path.indexOf("/");
        String agentName = path.substring(0, index);
        String dataName = path.substring(index + 1);
        long dataId;
        try (Session sess = fac.openSession()) {
            Query q = sess.createQuery("select dd.id from DataDesc dd where dd.dataPath = :dataPath");
            q.setReadOnly(true);
            q.setParameter("dataPath", new DataPath(agentName, dataName));
            dataId = (long) q.uniqueResult();
        }
        return dataId;
    }

    private String getPathForId(long id) {
        String path = null;
        try (Session sess = fac.openSession()) {
            Query q = sess.createQuery("select dd.dataPath from DataDesc dd where dd.id = :id");
            q.setReadOnly(true);
            q.setParameter("id", id);
            path = ((DataPath) q.uniqueResult()).getFullKey();
        }
        return path;
    }

    @GET
    @Path("/data/latest/")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public TrendingData getLatestDataByPath(@QueryParam("path") String path) {
        long dataId = getDataIdForPath(path);
        TrendingData result = new TrendingData();
        try (Session sess = fac.openSession()) {
            Query q = sess.createQuery("from RawData r where r.dataDesc.id = :id order by r.time DESC");
            q.setReadOnly(true);
            q.setMaxResults(1);
            q.setParameter("id", dataId);
            RawData data = (RawData) q.uniqueResult();
            if (data != null) {
                long tStamp = data.getTime();
                AxisValue axisValue = new AxisValue("time", tStamp);
                result.setAxisValue(axisValue);
                DataValue[] dataValue = new DataValue[1];
                Double dd = data.getDoubleData();
                dataValue[0] = new DataValue("value", dd == null ? 0 : dd);
                result.setDataValue(dataValue);
            }
        }

        return result;
    }

    @GET
    @Path("/data/search/")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public List<Data> getDataByPath(@QueryParam("path") List<String> paths,
            @QueryParam("t1") @DefaultValue("-1") long t1, @QueryParam("t2") @DefaultValue("-1") long t2,
            @QueryParam("flavor") String flavor, @QueryParam("n") @DefaultValue("30") int nbins) {

        List<Data> result = new ArrayList<>();
        for (String path : paths) {
            long dataId = getDataIdForPath(path);
            Data d = getInnerData(path, dataId, t1, t2, flavor, nbins);
            result.add(d);
        }
        return result;
    }

    @GET
    @Path("/data/")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public List<Data> getData(@QueryParam("id") List<Long> ids, @QueryParam("t1") @DefaultValue("-1") long t1,
            @QueryParam("t2") @DefaultValue("-1") long t2, @QueryParam("flavor") String flavor,
            @QueryParam("n") @DefaultValue("30") int nbins) {

        List<Data> result = new ArrayList<>();
        for (long id : ids) {
            Data d = getData(id, t1, t2, flavor, nbins);
            result.add(d);
        }
        return result;
    }

    @GET
    @Path("/data/{id:  \\d+}")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public Data getData(@PathParam("id") long id, @QueryParam("t1") @DefaultValue("-1") long t1,
            @QueryParam("t2") @DefaultValue("-1") long t2, @QueryParam("flavor") String flavor,
            @QueryParam("n") @DefaultValue("30") int nbins) {

        return getInnerData(null, id, t1, t2, flavor, nbins);
    }

    private Data getInnerData(String path, long id, long t1, long t2, String flavor, int nbins) {

        if (path == null) {
            path = getPathForId(id);
        }

        // handling of default values
        // t2 default is now
        // t1 default is t2 - 1 hour
        // rawPreferred is false
        // nbins is 30
        boolean useStat = false;
        boolean useRaw = false;
        if ("stat".equals(flavor)) {
            useStat = true;
        } else if ("raw".equals(flavor)) {
            useRaw = true;
        } else {
            flavor = "unspecified";
        }

        if (t2 < 0) {
            t2 = System.currentTimeMillis();
        }
        if (t1 < 0) {
            t1 = t2 - 3600000L;
        }

        log.fine("request for data " + id + "[" + t1 + "," + t2 + "] " + flavor + " " + nbins);

        // what do we have?
        long rawId = id;

        if (useRaw) {
            log.debug("sending raw data");
            return exportRawData(path, rawId, t1, t2);
        } else if (useStat) {

            StatDesc statSource = null;
            long statSourceN = -1;
            Map<StatDesc, Long> stats = getAvailableStats(rawId, t1, t2);

            log.fine("stats :");
            // how do we create the data?
            for (Map.Entry<StatDesc, Long> s : stats.entrySet()) {
                log.debug("  " + s.getKey().getId() + " " + s.getKey().getTimeBinWidth() + " : " + s.getValue());

                long n = s.getValue();
                if (n > nbins / 2) {
                    // this one has enough bins
                    // TODO # of samples is not enough, for instance raw could
                    // have more samples
                    // but covering only the recent part of the time range.
                    if (statSource != null) {
                        if (n < statSourceN) {
                            statSource = s.getKey();
                            statSourceN = n;
                        }
                    } else {
                        statSource = s.getKey();
                        statSourceN = n;
                    }
                }
            }

            if (statSource != null) {
                log.fine(
                        "sending stat from stat sampling " + statSource.getTimeBinWidth() + " nsamples " + statSourceN);

                return exportStatDataFromStat(path, statSource, t1, t2, nbins);
            }
        }

        log.debug("sending stat from raw");
        return exportStatDataFromRaw(path, rawId, t1, t2, nbins);
    }

    @SuppressWarnings("unchecked")
    public List<ChannelMetaData> getMetadataForChannel(int channelId, long t1, long t2) {

        List<MetaDataData> l = null;
        try (Session sess = fac.openSession()) {
            // String queryStr = "select md from MetaDataData md inner join
            // md.dataGroup g
            // inner join g.members m where m.id=:id ";
            // if (t1>-1&&t2>-1) {
            // queryStr += "and ((startTime >= :t1 and startTime < :t2) or
            // (startTime < :t1
            // and (endTime > :t1 or endTime = -1)))";
            // } else if (t2<0&&t1>-1) { // No upper limit
            // queryStr += "and startTime >= :t1";
            // } else if (t2>-1&&t1<0) { // No lower limit
            // queryStr += "and startTime < :t2 or endTime = -1";
            // }

            // mysql is not efficient on queries like
            // ((startTime >= :t1 and startTime < :t2) or (startTime < :t1 and
            // (endTime >
            // :t1 or endTime = -1)))
            // one should union the two queries
            // ((startTime >= :t1 and startTime < :t2)
            // (startTime < :t1 and (endTime > :t1 or endTime = -1)))
            // no UNION in HSQL => two queries.
            // we also want to recover the list of groups for a channel.
            String groupQuery = "select g.id from DataGroup g inner join g.members m where m.id=:id";
            Query qGroup = sess.createQuery(groupQuery);
            qGroup.setReadOnly(true);
            qGroup.setLong("id", channelId);
            List<Integer> lgid = qGroup.list();

            if (lgid.isEmpty()) {
                return new ArrayList<ChannelMetaData>();
            }

            if (t1 > -1 && t2 > -1) {
                String queryStr = "select md from MetaDataData md where md.dataGroup.id in (:list) "
                        + " and (startTime >= :t1 and startTime < :t2)";
                Query q = sess.createQuery(queryStr);
                q.setReadOnly(true);
                q.setParameterList("list", lgid);
                q.setParameter("t1", t1);
                q.setParameter("t2", t2);
                l = q.list();
                queryStr = "select md from MetaDataData md where md.dataGroup.id in (:list) "
                        + " and (startTime < :t1 and (endTime > :t1 or endTime = -1))";
                q = sess.createQuery(queryStr);
                q.setParameterList("list", lgid);
                q.setParameter("t1", t1);
                q.setReadOnly(true);
                l.addAll(q.list());
            } else if (t2 < 0 && t1 > -1) { // No upper limit
                String queryStr = "select md from MetaDataData md where md.dataGroup.id in (:list) "
                        + " and startTime >= :t1";
                Query q = sess.createQuery(queryStr);
                q.setParameterList("list", lgid);
                q.setReadOnly(true);
                q.setParameter("t1", t1);
                l = q.list();
            } else if (t2 > -1 && t1 < 0) { // No lower limit
                String queryStr = "select md from MetaDataData md where md.dataGroup.id in (:list) "
                        + " and (startTime < :t2 or endTime = -1)";
                Query q = sess.createQuery(queryStr);
                q.setParameterList("list", lgid);
                q.setReadOnly(true);
                q.setParameter("t2", t2);
                l = q.list();
            } else {
                String queryStr = "select md from MetaDataData md where md.dataGroup.id in (:list) ";
                Query q = sess.createQuery(queryStr);
                q.setParameterList("list", lgid);
                q.setReadOnly(true);
                l = q.list();
            }
        }
        List<ChannelMetaData> out = new ArrayList<ChannelMetaData>();
        for (MetaDataData md : l) {
            out.add(new ChannelMetaData(md));
        }
        return out;

    }

    @GET
    @Path("/channelinfo/{id}")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public ChannelMetaData.ChannelMetadataList getMetadataList(@PathParam("id") long channelId) {
        long rawId = channelId;
        return new ChannelMetaData.ChannelMetadataList(getMetadataForChannel((int) rawId, -1, -1));
    }

    @GET
    @Path("listsubsystems")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public DataSubsystem.DataSubsystemList getSubsystems() {
        return new DataSubsystem.DataSubsystemList(getListOfSubsystems());
    }

    private List<DataSubsystem> getListOfSubsystems() {
        List<DataSubsystem> l = null;
        try (Session sess = fac.openSession()) {
            Query dataQuery = sess.createQuery("select distinct dd.dataPath.agentName from DataDesc dd");
            dataQuery.setReadOnly(true);
            l = ((List<String>) dataQuery.list()).stream().map((String agentName) -> {
                return new DataSubsystem(agentName);
            }).collect(Collectors.toList());
        }
        return l;
    }

    /**
     *
     * @return the whole channels list for all CCS.
     * 
     *         optional query parameter maxIdleSeconds with default value 86400:
     *         only returns parameters seen since that delay. Negative value:
     *         returns all channels.
     */
    @GET
    @Path("/listchannels")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public DataChannel.DataChannelList getChannels(@QueryParam("maxIdleSeconds") @DefaultValue("604800") long dt) {
        return new DataChannel.DataChannelList(getListOfChannels(dt), System.currentTimeMillis()-dt*1000);
    }

    /**
     *
     * @param subsystemName
     * @return a channels list for a subsystem
     * 
     *         optional query parameter maxIdleSeconds with default value 36000:
     *         only returns parameters seen since that delay. Negative value:
     *         returns all channels.
     */
    @GET
    @Path("/listchannels/{subsystem}")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public DataChannel.DataChannelList getChannels(@PathParam("subsystem") String subsystemName,
            @QueryParam("maxIdleSeconds") @DefaultValue("604800") long dt) {
        List<DataChannel> channels = getListOfChannels(dt);
        ArrayList<DataChannel> subChannels = new ArrayList<DataChannel>();
        for (DataChannel dc : channels) {
            if (dc.getPath()[0].equals(subsystemName)) {
                subChannels.add(dc);
            }
        }
        return new DataChannel.DataChannelList(subChannels, System.currentTimeMillis()-dt*1000);

    }

    /**
     *
     * @param partialPath
     * @param level
     * @return the list of channels within a partial path and a level
     */
    public DataChannel[] getChannels(String partialPath, int level) {
        // TODO
        return null;
    }

    /**
     * Counts for each StatDesc entity associated to the given DataDesc id, how
     * many StatData entities span the given time interval [t1,t2[.
     *
     * @param rawId
     *            the DataDesc id
     * @param t1
     * @param t2
     * @return
     */
    protected Map<StatDesc, Long> getAvailableStats(long rawId, long t1, long t2) {

        Map<StatDesc, Long> m = new HashMap<>();
        try (Session sess = fac.openSession()) {
            // Should we really perform this query?
            Query q = sess.createQuery("select d, count(x) from StatDesc d, StatData x where d.dataDesc.id = :id "
                    + "and x.statDesc = d and (x.statTimeInterval.endTime) > :t1 and x.statTimeInterval.startTime < :t2 group by d.id order by d.timeBinWidth");
            q.setReadOnly(true);
            q.setLong("id", rawId);
            q.setParameter("t1", t1);
            q.setParameter("t2", t2);
            @SuppressWarnings("unchecked")
            List<Object[]> l = q.list();
            for (Object[] r : l) {
                if (r[0] == null) {
                    continue;// crazy
                }
                m.put((StatDesc) r[0], (Long) r[1]);
            }
        }
        return m;
    }

    protected long getAvailableRawData(long rawId, long t1, long t2) {
        long n = 0;
        try (Session sess = fac.openSession()) {
            Query q = sess.createQuery(
                    "select count(r) from RawData r where r.dataDesc.id = :id and r.tstamp between :t1 and :t2 order by r.tstamp");
            q.setReadOnly(true);
            q.setParameter("id", rawId);
            q.setParameter("t1", t1);
            q.setParameter("t2", t2);
            n = (Long) q.uniqueResult();
        }
        return n;
    }

    public List<RawData> getRawData(long id, long t1, long t2) {

        List<RawData> l = null;
        try (Session sess = fac.openSession()) {

            Query q = sess.createQuery(
                    "from RawData r where r.dataDesc.id = :id and r.time between :t1 and :t2 order by r.time");
            q.setReadOnly(true);
            q.setParameter("id", id);
            q.setParameter("t1", t1);
            q.setParameter("t2", t2);
            l = q.list();
            log.debug("retrieved raw data " + id + "[" + t1 + "," + t2 + "] : " + l.size());
        }

        return l;
    }

    /**
     * Returns the list of StatData with the given statDesc that overlap the
     * interval [t1,t2[.
     * 
     * @param statDesc
     *            the statistical description
     * @param t1
     * @param t2
     * @return
     */
    protected List<StatData> getStatData(StatDesc statDesc, long t1, long t2) {

        List<StatData> l = null;
        try (Session sess = fac.openSession()) {
            Query q = sess.createQuery(
                    "from StatData r join fetch r.statTimeInterval where r.statDesc.id = :id and r.statTimeInterval.startTime > :tlow and r.statTimeInterval.startTime < :thigh order by r.statTimeInterval.startTime");
            q.setReadOnly(true);
            q.setParameter("id", statDesc.getId());
            q.setParameter("tlow", t1 - statDesc.getTimeBinWidth());
            q.setParameter("thigh", t2);
            l = q.list();
            log.debug("retrieved stat data " + statDesc.getId() + "[" + t1 + "," + t2 + "] : " + l.size());
        }
        return l;
    }

    protected Data exportRawData(String path, long rawId, long t1, long t2) {
        List<RawData> l = getRawData(rawId, t1, t2);

        Data d = new Data();
        d.setMetaDataData(getMetadataForChannel((int) rawId, t1, t2));

        TrendingData[] data = new TrendingData[l.size()];
        for (int i = 0; i < l.size(); i++) {
            RawData r = l.get(i);
            TrendingData dt = new TrendingData();
            data[i] = dt;
            long tStamp = r.getTime();
            AxisValue axisValue = new AxisValue("time", tStamp);
            dt.setAxisValue(axisValue);
            DataValue[] dataValue = new DataValue[1];
            Double dd = r.getDoubleData();
            dataValue[0] = new DataValue("value", dd == null ? 0 : dd);
            dt.setDataValue(dataValue);
        }
        d.getTrendingResult().setTrendingDataArray(data);
        d.setId(rawId);
        d.setPath(path);

        return d;
    }

    protected static class StatEntry {
        long low, high;
        double sum1, sum2, min, max;
        long n;

        public StatEntry(long low, long high, double sum1, double sum2, long n, double min, double max) {
            super();
            this.low = low;
            this.high = high;
            this.sum1 = sum1;
            this.sum2 = sum2;
            this.min = min;
            this.max = max;
            this.n = n;
        }

        public long getLow() {
            return low;
        }

        public long getHigh() {
            return high;
        }

        public double getMin() {
            return min;
        }

        public double getMax() {
            return max;
        }

        public double getMean() {
            return n > 0 ? sum1 / n : 0;
        }

        public double getStdDev() {
            double nd = (double)n;
            return n > 0 ? Math.sqrt((sum2/nd) - ((sum1*sum1) / (nd*nd))) : 0;
        }

    }

    protected Data exportStatDataFromRaw(String path, long rawId, long t1, long t2, int nsamples) {
        try (Session sess = fac.openSession()) {
            Query q = sess.createQuery(
                    "select new org.lsst.ccs.localdb.statusdb.server.DataServer$StatEntry(min(rd.time), max(rd.time), sum(rd.doubleData), "
                            + " sum(rd.doubleData*rd.doubleData), count(1), min(rd.doubleData), max(rd.doubleData)) from"
                            + " RawData rd where rd.dataDesc.id = :id and rd.time >= :t1  and rd.time < :t2 "
                            + " group by floor(rd.time/:deltat) having count(1) > 0 ORDER BY min(rd.time) ASC");
            q.setReadOnly(true);
            long deltat = (t2 - t1) / nsamples;

            q.setParameter("id", rawId);
            q.setParameter("t1", t1);
            q.setParameter("t2", t2);
            q.setParameter("deltat", deltat);

            @SuppressWarnings("unchecked")
            List<StatEntry> l = q.list();

            List<TrendingData> trendingDataList = new ArrayList<>();

            for (StatEntry se : l) {

                long low = se.getLow();
                long high = se.getHigh();
                TrendingData dt = new TrendingData();
                dt.setAxisValue(new AxisValue("time", (low + high) / 2, low, high));
                DataValue[] dataValue = new DataValue[5];

                double value = se.getMean();
                double stddev = se.getStdDev();

                double min = se.getMin();
                double max = se.getMax();
                dataValue[0] = new DataValue("value", value);
                dataValue[1] = new DataValue("rms", stddev);
                dataValue[2] = new DataValue("stddev", stddev);
                dataValue[3] = new DataValue("min", min);
                dataValue[4] = new DataValue("max", max);
                dt.setDataValue(dataValue);
                trendingDataList.add(dt);
            }

            TrendingData[] trendingData = trendingDataList.toArray(new TrendingData[0]);

            Data d = new Data();
            d.setId(rawId);
            d.setPath(path);
            d.setMetaDataData(getMetadataForChannel((int) rawId, t1, t2));
            d.getTrendingResult().setTrendingDataArray(trendingData);

            return d;
        }
    }

//    protected Data exportStatDataFromRawNative(String path, long rawId, long t1, long t2, int nsamples) {
//
//        try (Session sess = fac.openSession()) {
//
//            SQLQuery q = sess.createSQLQuery("select tlow, thigh, datasum/entries as mean, " //
//                    + " case when entries > 0 then " //
//                    + " sqrt((datasumsquared - datasum*datasum/entries)/entries) " //
//                    + " else 0.0 end as rms, " //
//                    + " mindata, maxdata " //
//                    + " from ( SELECT MIN(rd.time) AS tlow, MAX(rd.time) AS thigh, " //
//                    + " SUM(rd.doubleData) AS datasum, SUM(rd.doubleData*rd.doubleData) AS datasumsquared, " //
//                    + " min(rd.doubleData) as mindata, max(rd.doubleData) as maxdata, " //
//                    + " count(1) AS entries from ccs_rawData rd where rd.dataDescId = :id and time >= :t1 " //
//                    + " and time < :t2 group by floor(rd.time/:deltat) ) accumulated where entries > 0 ORDER BY tlow ASC");
//            q.setReadOnly(true);
//
//            long deltat = (t2 - t1) / nsamples;
//
//            q.setParameter("id", rawId);
//            q.setParameter("t1", t1);
//            q.setParameter("t2", t2);
//            q.setParameter("deltat", deltat);
//
//            List<Object[]> l = null;
//            l = q.list();
//
//            List<TrendingData> trendingDataList = new ArrayList<>();
//            for (Object[] obj : l) {
//
//                if (obj[2] != null) {
//                    long low = ((BigInteger) obj[0]).longValue();
//                    long high = ((BigInteger) obj[1]).longValue();
//                    TrendingData dt = new TrendingData();
//                    dt.setAxisValue(new AxisValue("time", (low + high) / 2, low, high));
//                    DataValue[] dataValue = new DataValue[4];
//
//                    double value = ((Double) obj[2]).doubleValue();
//                    double rms = 0;
//                    if (obj[3] != null) {
//                        rms = ((Double) obj[3]).doubleValue();
//                    }
//                    double min = ((Double) obj[4]).doubleValue();
//                    double max = ((Double) obj[5]).doubleValue();
//                    dataValue[0] = new DataValue("value", value);
//                    dataValue[1] = new DataValue("rms", rms);
//                    dataValue[2] = new DataValue("min", min);
//                    dataValue[3] = new DataValue("max", max);
//                    dt.setDataValue(dataValue);
//                    trendingDataList.add(dt);
//                }
//            }
//
//            TrendingData[] trendingData = trendingDataList.toArray(new TrendingData[0]);
//
//            Data d = new Data();
//            d.setId(rawId);
//            d.setPath(path);
//            d.setMetaDataData(getMetadataForChannel((int) rawId, t1, t2));
//            d.getTrendingResult().setTrendingDataArray(trendingData);
//            d.setId(rawId);
//
//            return d;
//        }
//    }

    protected Data exportStatDataFromStat(String path, StatDesc statDesc, long t1, long t2, int nsamples) {
        List<StatData> in = getStatData(statDesc, t1, t2);

        int n = in.size();
        int rebin = 1;
        int nout = n;

        if (n > nsamples * 3) {
            // The number of available bins is a lot (more than 3 times) larger
            // than the
            // number of requested points.
            rebin = n / nsamples;
            nout = (n + rebin - 1) / rebin;
            log.debug("will rebin stat by " + rebin + " : " + nout);
        }

        Data d = new Data();
        d.setMetaDataData(getMetadataForChannel((int) statDesc.getDataDesc().getId(), t1, t2));        
        d.setPath(path);

        d.getTrendingResult().setTrendingDataArray(new TrendingData[nout]);
        int iout = 0;
        int ibin = 0;
        double sum = 0;
        double s2 = 0;
        long nsamp = 0;
        long low = 0;
        double max = -Double.MAX_VALUE;
        double min = Double.MAX_VALUE;
        for (StatData sd : in) {
            if (ibin == 0) {
                sum = 0;
                s2 = 0;
                nsamp = 0;
                low = sd.getStatTimeInterval().getStartTime();
                max = -Double.MAX_VALUE;
                min = Double.MAX_VALUE;
            }
            sum += sd.getSum();
            s2 += sd.getSum2();
            if (sd.getMax() > max) {
                max = sd.getMax();
            }
            if (sd.getMin() < min) {
                min = sd.getMin();
            }
            nsamp += sd.getN();
            if (ibin == rebin - 1 || (iout * rebin + ibin == n - 1)) {
                log.debug("storing for " + iout + " from " + (ibin + 1) + " bins");
                TrendingData dt = new TrendingData();
                d.getTrendingResult().getTrendingDataArray()[iout] = dt;
                dt.setAxisValue(new AxisValue("time", (low + sd.getStatTimeInterval().getEndTime()) / 2, low,
                        sd.getStatTimeInterval().getEndTime()));
                DataValue[] dataValue = new DataValue[5];
                dataValue[0] = new DataValue("value", sum / nsamp);
                double stddev = Math.sqrt(s2 / nsamp - (sum / nsamp) * (sum / nsamp));
                dataValue[1] = new DataValue("rms", stddev);
                dataValue[2] = new DataValue("stddev", stddev);
                dataValue[3] = new DataValue("min", min);
                dataValue[4] = new DataValue("max", max);
                dt.setDataValue(dataValue);
                iout++;
                ibin = 0;
            } else {
                ibin++;
            }
        }

        d.setId(statDesc.getDataDesc().getId());
        return d;
    }

    // -- State changes
    /**
     * Lists all the existing states for a given subsystem.
     * 
     * @param subsystem
     *            the subsystem name.
     * @return
     */
    @GET
    @Path("liststates/{subsystem}")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public StateInfo.StateInfoList getStateInfo(@PathParam("subsystem") String subsystem) {
        List<StateInfo> res = new ArrayList<>();
        try (Session sess = fac.openSession()) {

            // Fetching the internal states
            Query q = sess
                    .createQuery(
                            "select distinct isd from AgentState as ags inner join ags.componentStates as sbd inner join sbd.componentStates as isd where ags.agentDesc.agentName=:agentName")
                    .setString("agentName", subsystem).setReadOnly(true);
            Map<String, StateInfo> innerStateInfoMap = new HashMap<>();
            List<InnerStateDesc> isdl = q.list();
            for (InnerStateDesc isd : isdl) {
                String isdClassName = isd.getEnumClassName();
                StateInfo si = innerStateInfoMap.get(isdClassName);
                if (si == null) {
                    si = new StateInfo(isdClassName);
                    innerStateInfoMap.put(isdClassName, si);
                }
                si.getStateValues().add(isd.getEnumValue());
            }
            for (StateInfo si : innerStateInfoMap.values()) {
                res.add(si);
            }
        }
        return new StateInfo.StateInfoList(res);
    }

    /**
     * Gets the state changes events for the given list of states, between the
     * time interval {@code [t1,t2]}, for the given subsystem.
     *
     * @param subsystem
     *            the subsystem
     * @param states
     *            if empty, returns all state changes for all found states.
     * @param components
     *            if empty, return all state changes for all components.
     * @param t1
     * @param t2
     * @return
     */
    @GET
    @Path("/statechanges/{subsystem}")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public StateChange.StateChangesList getStateChangeList(@PathParam("subsystem") String subsystem,
            @QueryParam("state") List<String> states, @QueryParam("component") List<String> components,
            @QueryParam("t1") @DefaultValue("-1") long t1, @QueryParam("t2") @DefaultValue("-1") long t2) {
        return innerGetStateChangeList(subsystem, components, states, t1, t2);
    }

    /**
     * Gets the state changes events for the given list of states, between the
     * time interval {@code [t1,t2]}, for a given component of a subsystem.
     *
     * @param subsystem
     *            the subsystem
     * @param component
     *            the component for which to list state changes
     * @param states
     *            if empty, returns all state changes for all found states
     * @param t1
     * @param t2
     * @return
     */
    @GET
    @Path("/statechanges/{subsystem}/{component}")
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public StateChange.StateChangesList getComponentStateChangeList(@PathParam("subsystem") String subsystem,
            @PathParam("component") String component, @QueryParam("state") List<String> states,
            @QueryParam("t1") @DefaultValue("-1") long t1, @QueryParam("t2") @DefaultValue("-1") long t2) {
        return innerGetStateChangeList(subsystem, Arrays.asList(component), states, t1, t2);
    }

    private StateChange.StateChangesList innerGetStateChangeList(String subsystem, List<String> components,
            List<String> states, long t1, long t2) {
        if (t2 < 0) {
            t2 = System.currentTimeMillis();
        }
        if (t1 < 0) {
            t1 = t2 - 3600000L;
        }
        List<StateChangeNotificationData> stateChangeNotifications = getStateChanges(subsystem, t1, t2);
        if (stateChangeNotifications.isEmpty()) {
            return new StateChange.StateChangesList();
        }
        List<StateChange> stateChanges = new ArrayList<>();

        // Processing the first StateChangeNotificationData
        StateChangeNotificationData firstSCND = stateChangeNotifications.get(0);
        Map<String, Map<String, String>> origStates = null;
        if (firstSCND.getTime() <= t1) {
            origStates = firstSCND.getNewState().asFlatStatesMap(components);
        } else {
            origStates = firstSCND.getOldState().asFlatStatesMap(components);
        }
        StateChange sc = new StateChange();
        sc.setTime(t1);
        for (Map.Entry<String, Map<String, String>> e1 : origStates.entrySet()) {
            for (Map.Entry<String, String> e2 : e1.getValue().entrySet()) {
                if (states == null || states.isEmpty() || states.contains(e2.getKey())) {
                    sc.addStateEvent(e1.getKey(), e2.getKey(), e2.getValue());
                }
            }
        }

        stateChanges.add(sc);
        StateChange originSE = sc;

        for (StateChangeNotificationData scnd : stateChangeNotifications) {
            if (scnd.getTime() <= t1)
                continue;
            sc = new StateChange();

            // Computing the delta between oldState and newState
            Map<String, Map<String, String>> oldState = scnd.getOldState().asFlatStatesMap(components);
            Map<String, Map<String, String>> newState = scnd.getNewState().asFlatStatesMap(components);

            for (Map.Entry<String, Map<String, String>> e1 : newState.entrySet()) {
                String component = e1.getKey();
                for (Map.Entry<String, String> e2 : e1.getValue().entrySet()) {
                    String enumClassName = e2.getKey();
                    String enumValue = e2.getValue();
                    if (states == null || states.isEmpty() || states.contains(enumClassName)) {
                        Map<String, String> prevCompStates = oldState.get(component);
                        if (prevCompStates == null || !enumValue.equals(prevCompStates.get(enumClassName))) {
                            sc.addStateEvent(component, enumClassName, enumValue);
                            if (!originSE.containsEvent(component, enumClassName)) {
                                originSE.addStateEvent(component, enumClassName, "_UNDEF_");
                            }
                        }
                    }
                }
            }

            // This should not happen
            if (sc.isEmpty())
                continue;

            sc.setTime(scnd.getTime());

            stateChanges.add(sc);
        }

        return new StateChange.StateChangesList(stateChanges);
    }

    /**
     * Returns the state changes events occurring from a given subsystem within
     * the time interval [t1;t2]. If none is found, it returns the closest
     * StateChangeNotificationData before t1.
     *
     * @param subsystemName
     *            the subsystem name
     * @param t1
     *            tstart
     * @param t2
     *            tstop
     * @return
     */
    private List<StateChangeNotificationData> getStateChanges(String subsystemName, long t1, long t2) {
        List<StateChangeNotificationData> res = null;
        try (Session sess = fac.openSession()) {
            Query q = sess
                    .createQuery(
                            "from StateChangeNotificationData scnd where scnd.agentDesc.agentName=:name and scnd.time >:t1 and scnd.time <=:t2")
                    .setString("name", subsystemName).setLong("t1", t1).setLong("t2", t2).setReadOnly(true);
            res = q.list();
            if (res.isEmpty()) {
                // Finding the closest previous state change notification status
                q = sess.createQuery(
                        "from StateChangeNotificationData scnd where scnd.agentDesc.agentName=:name and scnd.time <=:t1 order by scnd.time desc")
                        .setString("name", subsystemName).setLong("t1", t1).setMaxResults(1).setReadOnly(true);
                res = q.list();
            }
        }
        return res;
    }

    // -- Alerts information retrieval
    @Path("/listalerts")
    @GET
    public AlertInfo.AlertInfoList getAlertInfo() {
        return getAlertInfo(null);
    }

    @Path("/listalerts/{subsystem}")
    @GET
    public AlertInfo.AlertInfoList getAlertInfo(@PathParam("subsystem") String subsystem) {
        List<AlertInfo> res = new ArrayList<>();
        try (Session sess = fac.openSession()) {
            Query q;
            if (subsystem == null) {
                q = sess.createQuery("from AlertDesc ad");
            } else {
                q = sess.createQuery("from AlertDesc ad where ad.agentDesc.agentName=:name").setString("name",
                        subsystem);
            }
            q.setReadOnly(true);
            List<AlertDesc> l = q.list();

            for (AlertDesc ad : l) {
                res.add(new AlertInfo(ad.getId(), ad.getAgentDesc().getAgentName(), ad.getAlertId(),
                        ad.getAlertDescription()));
            }
        }
        return new AlertInfo.AlertInfoList(res);
    }

    /**
     * Returns a list of active alerts. Active alerts are alerts that have been
     * raised during the current subsystem run.
     *
     * @param ids
     *            a list of ids.
     * @return
     */
    @Path("activealerts")
    @GET
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public AlertEvent.AlertEventList getActiveAlerts(@QueryParam("id") List<Long> ids) {
        List<AlertEvent> res = new ArrayList<>();
        try (Session sess = fac.openSession()) {
            Query q;
            if (ids == null || ids.isEmpty()) {
                q = sess.createQuery("from RaisedAlertData arad where arad.active=true");
            } else {
                q = sess.createQuery("from RaisedAlertData arad where arad.alertDesc.id in (:ids) and arad.active=true")
                        .setParameterList("ids", ids).setReadOnly(true);
            }
            q.setReadOnly(true);
            for (RaisedAlertData rad : (List<RaisedAlertData>) q.list()) {
                AlertEvent ae = new AlertEvent(rad.getAlertDesc().getAlertId(), rad.getTime(), rad.getSeverity(),
                        rad.getAlertCause());
                ae.setCleared(rad.getClearingAlert() != null);
                res.add(ae);
            }
        }
        return new AlertEvent.AlertEventList(res);
    }

    /**
     * Returns an history of alerts that have been raised within the given time
     * interval.
     *
     * @param ids
     *            a list of alertdesc ids.
     * @param t1
     * @param t2
     * @return
     */
    @Path("alerthistory")
    @GET
    @Produces({ MediaType.TEXT_XML, MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
    public AlertEvent.AlertEventList getAlertHistories(@QueryParam("id") List<Long> ids,
            @QueryParam("t1") @DefaultValue("-1") long t1, @QueryParam("t2") @DefaultValue("-1") long t2) {
        if (t2 < 0) {
            t2 = System.currentTimeMillis();
        }
        if (t1 < 0) {
            t1 = t2 - 3600000L;
        }
        List<AlertEvent> res = new ArrayList<>();
        try (Session sess = fac.openSession()) {
            Query q;
            if (ids == null || ids.isEmpty()) {
                q = sess.createQuery("from RaisedAlertData arad");
            } else {
                q = sess.createQuery("from RaisedAlertData arad where arad.alertDesc.id in (:ids)")
                        .setParameterList("ids", ids).setReadOnly(true);
            }
            q.setReadOnly(true);
            for (RaisedAlertData rad : (List<RaisedAlertData>) q.list()) {
                AlertEvent ae = new AlertEvent(rad.getAlertDesc().getAlertId(), rad.getTime(), rad.getSeverity(),
                        rad.getAlertCause());
                ae.setCleared(rad.getClearingAlert() != null);
                res.add(ae);
            }
        }
        return new AlertEvent.AlertEventList(res);
    }

}
