package org.lsst.ccs.localdb.configdb.model;

import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import javax.persistence.Cacheable;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.Index;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.MapKeyJoinColumn;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.persistence.UniqueConstraint;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Represents a named configuration for a given category.
 * @author LSST CCS Team
 */
@Table(uniqueConstraints = {
    @UniqueConstraint(columnNames = {"baseDescriptionId", "category", "configName", "version"})},
        indexes = {@Index(columnList = "hashMD5")})
@Entity
@Cacheable
@org.hibernate.annotations.Cache(
        usage = org.hibernate.annotations.CacheConcurrencyStrategy.NONSTRICT_READ_WRITE, 
        region = "org.lsst.ccs.localdb.configdb.model.Configuration"
)
public class Configuration implements Serializable {

    private static final long serialVersionUID = 672642551238249224L;
    
    private long id;    

    private BaseDescription baseDescription;

    /** Name of the category for this configuration. */
    private String category;
    
    /** The configuration name for this category. */
    private String configNameForCategory;
    
    private Map<ParameterPath, ConfigurationParameterValue> parameterConfigurations = new HashMap<>();
    
    /** The time this configuration was saved. */
    private long tsaved;
    
    private int version;
    
    /** Indicates whether this named configuration is the default version. */
    private boolean defaultVersion;
    
    /** Indicates whether this named configuration is the latest version. */
    private boolean latestVersion = true;
    
    private byte[] hashMD5;
    
    ///////////////////////////// CONSTRUCTORS

    protected Configuration() {
    }
    
    public Configuration(BaseDescription subsystemDescription, String category, String configName, long date) {
        this.baseDescription = subsystemDescription;
        this.category = category;
        this.configNameForCategory = configName;
        this.tsaved = date;
    }
    
    public Configuration(BaseDescription subsystemDescription, String category, String configName, long date, int version) {
        this(subsystemDescription, category, configName, date);
        this.version = version;
    }

    /**
     * creates a copy of an active Configuration (for instance to start an engineering mode or to validate
     * a new tagged configuration from an engineering mode configuration).
     *
     * @param other
     * @param newName
     * @param date date of creation
     * @param version version number
     */
    public Configuration(Configuration other, String newName, long date, int version) {
        this(other.getBaseDescription(), other.getCategory(), 
                newName == null ? other.getConfigName() : newName, date, version);
        parameterConfigurations.putAll(other.getConfigurationParameterValues());
    }
    
    /////////////////////////////////// ACCESSORS/MUTATORS
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    public long getId() {
        return id;
    }
    
    protected void setId(long id) {
        this.id = id;
    }

    @ManyToOne (fetch=FetchType.EAGER)
    public BaseDescription getBaseDescription() {
        return baseDescription;
    }
    
    public void setBaseDescription(BaseDescription subsystemDesc) {
        this.baseDescription = subsystemDesc;
    }

    /**
     * @return an unmodifiable set
     */
    @ManyToMany(cascade = {CascadeType.ALL, CascadeType.PERSIST}, fetch=FetchType.EAGER)
    @MapKeyJoinColumn(name = "configurationParameter_parameterPath")
    public Map<ParameterPath, ConfigurationParameterValue> getConfigurationParameterValues() {
        return parameterConfigurations;
    }
    
    void setConfigurationParameterValues(Map<ParameterPath, ConfigurationParameterValue> parameterConfigurations) {
        this.parameterConfigurations = parameterConfigurations;
    }
    
    public String getCategory(){
        return category;
    }
    
    public void setCategory(String category) {
        this.category = category;
    }
    
    public String getConfigName(){
        return configNameForCategory;
    }
    
    public void setConfigName(String configName) {
        this.configNameForCategory = configName;
    }

    public long getTsaved() {
        return tsaved;
    }

    public void setTsaved(long tsaved) {
        this.tsaved = tsaved;
    }

    public int getVersion() {
        return version;
    }

    public void setVersion(int version) {
        this.version = version;
    }
    
    public boolean isDefaultVersion() {
        return defaultVersion;
    }

    public void setDefaultVersion(boolean defaultVersion) {
        this.defaultVersion = defaultVersion;
    }

    public boolean isLatestVersion() {
        return latestVersion;
    }

    public void setLatestVersion(boolean latestVersion) {
        this.latestVersion = latestVersion;
    }
    
    public void setHashMD5(byte[] md5) {
        this.hashMD5 = md5;
    }
    
    @Column(columnDefinition = "varbinary(100)", nullable = false)
    public byte[] getHashMD5() {
        return hashMD5;
    }
    
    /**
     * The md5 computation 
     * @param baseDescription
     * @param parameterValues
     * @return 
     */
    public static byte[] computeHashMD5(BaseDescription baseDescription, TreeMap<String, String> parameterValues) {
        StringBuilder sb = new StringBuilder();
        sb.append(Arrays.toString(baseDescription.getHashMD5())).append("*");
        for (Map.Entry<String, String> e : parameterValues.entrySet()) {
            String name = e.getKey();
            String value =e.getValue();
            sb.append(name).append("=").append(value).append(":");
        }
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            return md.digest(sb.toString().getBytes("UTF-8"));
        } catch (NoSuchAlgorithmException | UnsupportedEncodingException e1) {
            throw new RuntimeException(e1);
        }
    }
    
    @Transient
    public String getConfigurationDescriptionString() {
        return new StringBuilder(category).append(":")
                .append(configNameForCategory)
                .append("(").append(version).append(")")
                .toString();
    }

    /**
     * registers a list of parameter configurations
     * @param parameterConfiguration
     * @throws IllegalArgumentException if levels are incompatible
     * @throws IllegalArgumentException if the corresponding parameter descriptions are not alive in the current
 subsystem baseDescription
     */
    public void addConfigurationParameterValue(ConfigurationParameterValue parameterConfiguration) {
        ConfigurationParameter cp = parameterConfiguration.getConfigurationParameter();
        if (cp.getCategory().equals(getCategory())){
                // check for deprecation
            this.parameterConfigurations.put(parameterConfiguration.getPath(), parameterConfiguration);
        } else {
            Logger.getLogger("org.lsst.ccs.config").warn("The property " 
                    + cp.getPath().getParameterName() 
                    + " does not belong to the category specified by the property file name : " + getCategory());
        }
    }
    
    public  String getValue( ParameterPath path) {
        ConfigurationParameterValue cpv = parameterConfigurations.get(path);
        if (cpv == null) {
            return null;
        }
        return cpv.getValue();
    }
    
    @Override
    public String toString() {
        return "\""+category+"\":\""+configNameForCategory+"\"("+version+")";
    }
    
}
