package org.lsst.ccs.localdb.statusdb.utils;

import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.stat.EntityStatistics;
import org.hibernate.stat.NaturalIdCacheStatistics;
import org.hibernate.stat.QueryStatistics;
import org.hibernate.stat.SecondLevelCacheStatistics;
import org.hibernate.stat.Statistics;
import org.influxdb.InfluxDB;
import org.influxdb.dto.Point;
import org.influxdb.dto.Point.Builder;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.localdb.statusdb.model.AgentState;
import org.lsst.ccs.localdb.statusdb.model.StatTimeInterval;

/**
 * Hibernate cache statistics to be stored.
 *
 * @author LSST CCS team
 */
public class CacheStatistics {

    private static final Logger LOG = Logger.getLogger(CacheStatistics.class.getName());

    private final KeyValueDataList dataList = new KeyValueDataList("statistics");

    private static final String[] NAMED_QUERIES = new String[] {	
           "findAlertDesc","findInnerStateDesc","findDataDesc", "findStatData", "findStatDesc", "findDataGroup", "findStateBundleDesc"};    
    
    public CacheStatistics(SessionFactory fac) {
        this(fac, null);
    }

    public CacheStatistics(SessionFactory fac, InfluxDB influxDb) {
        Statistics stats = fac.getStatistics();
        // named queries
        try (Session sess = fac.openSession()) {

            // named queries
            for (String query : NAMED_QUERIES) {
                QueryStatistics queryStats = stats.getQueryStatistics(sess.getNamedQuery(query).getQueryString());

                if (influxDb != null) {
                    Builder namedQueriesPointBuilder = Point.measurement("db_named_queries")
                            .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);
                    namedQueriesPointBuilder = namedQueriesPointBuilder.
                            addField("cache_hit", queryStats.getCacheHitCount()).
                            addField("cache_miss", queryStats.getCacheMissCount()).
                            addField("cache_put", queryStats.getCachePutCount()).
                            addField("exec_time_avg", queryStats.getExecutionAvgTime()).
                            addField("exec_time_max", queryStats.getExecutionMaxTime()).
                            addField("exec_time_min", queryStats.getExecutionMinTime()).
                            addField("exec_count", queryStats.getExecutionCount()).
                            addField("exec_row_count", queryStats.getExecutionRowCount());

                    Point point = namedQueriesPointBuilder.tag("query_name", query).build();
                    influxDb.write(point);
                } else {
                    String queryPath = "statistics/queries/" + query + "/";
                    dataList.addData(queryPath + "hit", queryStats.getCacheHitCount());
                    dataList.addData(queryPath + "miss", queryStats.getCacheMissCount());
                    dataList.addData(queryPath + "put", queryStats.getCachePutCount());
                    dataList.addData(queryPath + "avgExec", queryStats.getExecutionAvgTime());
                }
            }

            // Second Level Cache
            for (String slcName : stats.getSecondLevelCacheRegionNames() ) {
                if ( !slcName.startsWith("org.lsst.ccs") ) {
                    continue;
                }
                SecondLevelCacheStatistics slcStats = stats.getSecondLevelCacheStatistics(slcName);
                String shortName = slcName.substring(slcName.lastIndexOf(".")+1);
                if (influxDb != null) {
                    Builder secondLevelCachePointBuilder = Point.measurement("db_second_level_cache")
                            .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);

                    secondLevelCachePointBuilder = secondLevelCachePointBuilder.
                            addField("cache_hit", slcStats.getHitCount()).
                            addField("cache_miss", slcStats.getMissCount()).
                            addField("cache_put", slcStats.getPutCount());

                    Point point = secondLevelCachePointBuilder.tag("class_name", shortName).build();
                    influxDb.write(point);
                } else {
                    String slcPath = "statistics/slc/" + shortName + "/";
                    dataList.addData(slcPath + "count", slcStats.getElementCountInMemory());
                    dataList.addData(slcPath + "hit", slcStats.getHitCount());
                    dataList.addData(slcPath + "miss", slcStats.getMissCount());
                    dataList.addData(slcPath + "put", slcStats.getPutCount());
                }
            }


            // Cached Entities
            for (String entityName : stats.getEntityNames()) {
                if ( !entityName.startsWith("org.lsst.ccs") ) {
                    continue;
                }
                String shortName = entityName.substring(entityName.lastIndexOf(".")+1);
                EntityStatistics entityStats = stats.getEntityStatistics(entityName);
                if (influxDb != null) {

                    Builder cachedEntitiesPointBuilder = Point.measurement("db_cached_entities")
                            .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);
                    cachedEntitiesPointBuilder = cachedEntitiesPointBuilder.
                            addField("delete", entityStats.getDeleteCount()).
                            addField("fetch", entityStats.getFetchCount()).
                            addField("insert", entityStats.getInsertCount()).
                            addField("load", entityStats.getLoadCount()).
                            addField("failure", entityStats.getOptimisticFailureCount()).
                            addField("update", entityStats.getUpdateCount());

                    Point point = cachedEntitiesPointBuilder.tag("class_name", shortName).build();
                    
                    influxDb.write(point);
                } else {
                    String entitiesPath = "statistics/entities/" + shortName + "/";
                    dataList.addData(entitiesPath + "load", entityStats.getLoadCount());
                    dataList.addData(entitiesPath + "fetch", entityStats.getFetchCount());
                }
            }


            // Cached Entities
            for (String entityName : stats.getEntityNames()) {
                if ( !entityName.startsWith("org.lsst.ccs") || !entityName.endsWith("##NaturalId") ) {
                    continue;
                }
                String shortName = entityName.substring(entityName.lastIndexOf(".")+1).replace("##NaturalId", "");
                NaturalIdCacheStatistics natIdStats = stats.getNaturalIdCacheStatistics(entityName);

                if (influxDb != null) {
                    Builder naturalIdPointBuilder = Point.measurement("db_natural_id")
                            .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);
                    naturalIdPointBuilder = naturalIdPointBuilder.
                            addField("hit", natIdStats.getHitCount()).
                            addField("miss", natIdStats.getMissCount()).
                            addField("put", natIdStats.getPutCount()).
                            addField("count_mem", natIdStats.getElementCountInMemory());

                    Point point = naturalIdPointBuilder.tag("class_name", shortName).build();
                    influxDb.write(point);
                } else {
                    String entitiesPath = "statistics/naturalId/" + shortName + "/";
                    dataList.addData(entitiesPath + "hit", natIdStats.getHitCount());
                    dataList.addData(entitiesPath + "miss", natIdStats.getMissCount());
                    dataList.addData(entitiesPath + "put", natIdStats.getPutCount());
                    dataList.addData(entitiesPath + "count", natIdStats.getElementCountInMemory());
                }
            }

            if (influxDb != null) {
                Builder sessionPointBuilder = Point.measurement("db_session")
                        .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);

                sessionPointBuilder = sessionPointBuilder.
                        addField("connect", stats.getConnectCount()).
                        addField("flush", stats.getFlushCount()).
                        addField("sess_open", stats.getSessionOpenCount()).
                        addField("sess_close", stats.getSessionCloseCount()).
                        addField("transact", stats.getTransactionCount()).
                        addField("transact_success", stats.getSuccessfulTransactionCount());
                Point point = sessionPointBuilder.build();
                influxDb.write(point);
            } else {
                dataList.addData("statistics/sessionStats/connectCount", stats.getConnectCount());
                dataList.addData("statistics/sessionStats/flushCount", stats.getFlushCount());
                dataList.addData("statistics/sessionStats/sessionOpenCount", stats.getSessionOpenCount());
                dataList.addData("statistics/sessionStats/sessionCloseCount", stats.getSessionCloseCount());
                dataList.addData("statistics/sessionStats/transactionCount", stats.getTransactionCount());
                dataList.addData("statistics/sessionStats/successfulTransactionCount", stats.getSuccessfulTransactionCount());
            }

            stats.clear();
        }
    }

    public KeyValueDataList getStatData() {
        return dataList;
    }
}
