package org.lsst.ccs.localdb.statusdb.utils;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.cache.internal.StandardQueryCache;
import org.hibernate.stat.EntityStatistics;
import org.hibernate.stat.NaturalIdCacheStatistics;
import org.hibernate.stat.QueryStatistics;
import org.hibernate.stat.SecondLevelCacheStatistics;
import org.hibernate.stat.Statistics;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.localdb.statusdb.model.AgentState;
import org.lsst.ccs.localdb.statusdb.model.StateBundleDesc;
import org.lsst.ccs.localdb.statusdb.model.AlertDesc;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.DataGroup;
import org.lsst.ccs.localdb.statusdb.model.InnerStateDesc;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.model.StatTimeInterval;

/**
 * Hibernate cache statistics to be stored.
 * @author LSST CCS team
 */
public class CacheStatistics {
    
    private static final String[] NAMED_QUERIES = new String[] {
           "findAlertDesc","findInnerStateDesc","findDataDesc", "findStatData", "findStatDesc", "findDataGroup", "findStateBundleDesc"};
    
    private static final Class[] CACHED_ENTITIES = new Class[]{
        AlertDesc.class,
        InnerStateDesc.class, 
        DataDesc.class, 
        AgentState.class, 
        StatTimeInterval.class, 
        StatData.class, 
        StatDesc.class, 
        StateBundleDesc.class, 
        DataGroup.class,
    };
    
    private static final Class[] SECOND_LEVEL_CACHE = new Class[]{
        AlertDesc.class,
        InnerStateDesc.class, 
        DataDesc.class, 
        AgentState.class, 
        StatTimeInterval.class, 
        StatData.class, 
        StatDesc.class, 
        StateBundleDesc.class, 
        DataGroup.class,
        StandardQueryCache.class        
    };
            
    private final KeyValueDataList dataList = new KeyValueDataList("statistics");
    
    public CacheStatistics(SessionFactory fac) {
        Statistics stats = fac.getStatistics();
        // named queries
        try (Session sess = fac.openSession()) {
            // named queries
            for (String query : NAMED_QUERIES) {
                String queryPath = "statistics/queries/"+query+"/";
                QueryStatistics queryStats = stats.getQueryStatistics(sess.getNamedQuery(query).getQueryString());
                dataList.addData(queryPath+"hit", queryStats.getCacheHitCount());
                dataList.addData(queryPath+"miss", queryStats.getCacheMissCount());
                dataList.addData(queryPath+"put", queryStats.getCachePutCount());
                dataList.addData(queryPath+"avgExec", queryStats.getExecutionAvgTime());
            }
            
            // Second Level Cache
            for (Class cls : SECOND_LEVEL_CACHE) {
                String slcPath = "statistics/slc/"+cls.getSimpleName()+"/";
                SecondLevelCacheStatistics slcStats = stats.getSecondLevelCacheStatistics(cls.getName());
                dataList.addData(slcPath+"count", slcStats.getElementCountInMemory());
                dataList.addData(slcPath+"hit", slcStats.getHitCount());
                dataList.addData(slcPath+"miss", slcStats.getMissCount());
                dataList.addData(slcPath+"put", slcStats.getPutCount());
            }
            
            // Cached Entities
            for (Class cls : CACHED_ENTITIES) {
                String entitiesPath = "statistics/entities/"+cls.getSimpleName()+"/";
                EntityStatistics entityStats = stats.getEntityStatistics(cls.getName());
                dataList.addData(entitiesPath+"load", entityStats.getLoadCount());
                dataList.addData(entitiesPath+"fetch", entityStats.getFetchCount());
            }
            
            // NaturalId cache statistics
            for (Class cls : new Class[]{StatTimeInterval.class, AgentState.class}) {
                String entitiesPath = "statistics/naturalId/"+cls.getSimpleName()+"/";
                NaturalIdCacheStatistics natIdStats = stats.getNaturalIdCacheStatistics(cls.getName() + "##NaturalId");
                dataList.addData(entitiesPath+"hit",natIdStats.getHitCount());
                dataList.addData(entitiesPath+"miss",natIdStats.getMissCount());
                dataList.addData(entitiesPath+"put",natIdStats.getPutCount());
                dataList.addData(entitiesPath+"count",natIdStats.getElementCountInMemory());
            }
            
            dataList.addData("statistics/sessionStats/connectCount", stats.getConnectCount());
            dataList.addData("statistics/sessionStats/flushCount", stats.getFlushCount());
            dataList.addData("statistics/sessionStats/sessionOpenCount", stats.getSessionOpenCount());
            dataList.addData("statistics/sessionStats/sessionCloseCount", stats.getSessionCloseCount());
            dataList.addData("statistics/sessionStats/transactionCount", stats.getTransactionCount());
            dataList.addData("statistics/sessionStats/successfulTransactionCount", stats.getSuccessfulTransactionCount());
            
            stats.clear();
        }
    }
        
    public KeyValueDataList getStatData() {
        return dataList;
    }
}
