/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.localdb.utils;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

/**
 *
 * @author The LSST CCS Team
 *
 */
public class BackupUtils {

    private static final Pattern PARTITION_PATTERN = Pattern.compile("p(\\d\\d)_(\\d\\d\\d\\d)");
    private static final String FUTURE_PARTITION = "futurePartition";
    private static final int PARTITIONS_TO_KEEP = 16;
    private static final int FUTURE_PARTITIONS = 52;

    /**
     * @param args the command line arguments
     * @throws java.sql.SQLException
     * @throws org.apache.commons.cli.ParseException
     */
    public static void main(String[] args) throws SQLException, ParseException {

        Properties p = BootstrapResourceUtils.getBootstrapProperties("statusPersister");

        // define the command line options
        Options commandLineOptions = new Options();
        // The help option
        commandLineOptions.addOption("h", "help", false, "Print the help message");

        // Partition the table on the time column
        commandLineOptions.addOption("p", "partition_table", false, "Partition the table");

        // Create the missing partitions.
        commandLineOptions.addOption("c", "create_missing_partitions", false, "Create missing partitions");

        // Create the missing partitions.
        commandLineOptions.addOption("r", "remove_old_partitions", false, "Remove old partitions");

        // Number of partitions to keep partitions.
        commandLineOptions.addOption("pk", "partitions_to_keep", true, "Number of partitions to keep");
        commandLineOptions.getOption("partitions_to_keep").setArgName("PARTITIONS_TO_KEEP");

        // Future partitions to create.
        commandLineOptions.addOption("fp", "future_partitions", true, "Number of partitions to create in the future");
        commandLineOptions.getOption("future_partitions").setArgName("FUTURE_PARTITIONS");

        CommandLineParser parser = new BasicParser();
        CommandLine line = parser.parse(commandLineOptions, args, false);

        if (line.hasOption("help")) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp(100, "SubsystemBoot", "", commandLineOptions, "", true);
            return;
        }

        Calendar cal = Calendar.getInstance();
        String[] connSplit = p.getProperty("hibernate.connection.url", "").split("/");
        String schemaName = connSplit[connSplit.length - 1];

        try (Connection connection = DriverManager.getConnection(p.getProperty("hibernate.connection.url") + "?user=" + p.getProperty("hibernate.connection.username") + "&password=" + p.getProperty("hibernate.connection.password"))) {

            int partitionsToKeep = PARTITIONS_TO_KEEP;
            if ( line.hasOption("partitions_to_keep") ) {
                partitionsToKeep = Integer.parseInt(line.getOptionValue("partitionsToKeep"));
            }
            int futurePartitions = FUTURE_PARTITIONS;
            if ( line.hasOption("future_partitions") ) {
                futurePartitions = Integer.parseInt(line.getOptionValue("future_partitions"));
            }
            
            //Get the date for removing partitions
            getBeginningOfWeek(cal);
            cal.add(Calendar.WEEK_OF_YEAR, -1 * partitionsToKeep);
            long removeBeforeMillis = cal.getTimeInMillis();

            System.out.println("*** Inspecting table ccs_rawData for schema " + schemaName);
            boolean abort = false;
            //Figure out if the rawData table is partitioned and if the partitions have the right name
            List<String> availablePartitions = new ArrayList<>();
            List<String> removePartitions = new ArrayList<>();
            PreparedStatement getPartitions = connection.prepareStatement("SELECT TABLE_NAME, PARTITION_NAME, TABLE_ROWS FROM INFORMATION_SCHEMA.PARTITIONS where table_name = 'ccs_rawData' and table_schema = ?;");
            getPartitions.setString(1, schemaName);
            try (ResultSet partitions = getPartitions.executeQuery()) {
                while (partitions.next()) {
                    String partitionName = partitions.getString("PARTITION_NAME");
                    long entries = partitions.getLong("TABLE_ROWS");
                    if (partitionName != null) {
                        availablePartitions.add(partitionName);
                        Matcher m = PARTITION_PATTERN.matcher(partitionName);
                        if (!m.matches() && !partitionName.equals(FUTURE_PARTITION)) {
                            System.out.println("Partition " + partitionName + " does not match the standard definition");
                            abort = true;
                        } else {
                            cal.clear();
                            if (!partitionName.equals(FUTURE_PARTITION)) {
                                cal.set(Calendar.WEEK_OF_YEAR, Integer.valueOf(m.group(1)));
                                cal.set(Calendar.YEAR, Integer.valueOf(m.group(2)));
                            } else {
                                cal.setTime(new Date());
                            }
                            Date startWeek = getBeginningOfWeek(cal);
                            cal.add(Calendar.WEEK_OF_YEAR, 1);
                            Date endWeek = cal.getTime();

                            System.out.println("Partition " + partitionName + " with " + entries + " rows \n \tStarting from " + startWeek + "(" + startWeek.getTime() + ") to " + endWeek + "(" + endWeek.getTime() + ")");
                            if (startWeek.getTime() < removeBeforeMillis) {
                                System.out.println("\t--> Can be removed");
                                removePartitions.add(partitionName);
                            }
                        }
                    } else {
                        System.out.println("The table is not partitioned and contains " + entries + " rows.");
                    }
                    System.out.println();
                }
            }
            if (abort) {
                throw new RuntimeException("The current partition names must be chanaged. We suggest you un-partition the table and start "
                        + "from scratch.\n To remove the partitioned table issue the command: ALTER TABLE ccs_rawData REMOVE PARTITIONING;");
            }

            if (availablePartitions.isEmpty()) {
                System.out.println("Table ccs_rawData is not partitioned. ");

                if (line.hasOption("partition_table")) {
                    System.out.println("Creating partitioning by range now.");
                    PreparedStatement initializePartitions = connection.prepareStatement("ALTER TABLE ccs_rawData partition by range (time) ( "
                            + " partition " + FUTURE_PARTITION + " values less than maxvalue)");
                    initializePartitions.executeUpdate();
                } else {
                    System.out.println("To partition the table by range re-run this program with the \"partition_table\" option.");
                }
                System.out.println();
            }

            //Select min/max time on raw data table
            Date startOfFirstWeek, startOfLastWeek;
            PreparedStatement getMinMaxTime = connection.prepareStatement("SELECT min(time) as minTime, max(time) as maxTime FROM ccs_rawData");
            try (ResultSet minMaxTime = getMinMaxTime.executeQuery()) {
                minMaxTime.next();

                long minTime = minMaxTime.getLong("minTime");
                long maxTime = minMaxTime.getLong("maxTime");

                cal.setTimeInMillis(minTime);
                Date startDate = cal.getTime();
                int minWeekInYear = cal.get(Calendar.WEEK_OF_YEAR);
                int minYear = cal.get(Calendar.YEAR);
                //Get the beginning of the first week:
                startOfFirstWeek = getBeginningOfWeek(cal);
                startOfFirstWeek.setTime(startOfFirstWeek.getTime());

                cal.setTimeInMillis(maxTime);                
                Date endDate = cal.getTime();
                int maxWeekInYear = cal.get(Calendar.WEEK_OF_YEAR);
                int maxYear = cal.get(Calendar.YEAR);

                System.out.println("Table time period: " + startDate + " " + endDate);
                System.out.println("Table weeks span: " + minWeekInYear + "(" + minYear + ") " + maxWeekInYear + "(" + maxYear + ") ");
                
                //Get the beginning of the last week:
                cal.add(Calendar.WEEK_OF_YEAR, futurePartitions);
                startOfLastWeek = getBeginningOfWeek(cal);  
                System.out.println("Partitions will be created up to : "+cal.get(Calendar.WEEK_OF_YEAR)+"("+cal.get(Calendar.YEAR)+")");
                
            }
            
            int lastWeekId = 1000;
            int lastYearId = 1000;
            
            Date partitionStart = startOfFirstWeek;
            boolean missingPartitions = false;
            while (partitionStart.getTime() < startOfLastWeek.getTime()) {
                cal.setTime(partitionStart);
                
                //There is a problem when the week ends on Dec 31.

                SimpleDateFormat yearWeekFormat = new SimpleDateFormat("ww");
                
                int week_of_year = cal.get(Calendar.WEEK_OF_YEAR);
                int year = cal.get(Calendar.YEAR);
                
                if ( week_of_year < lastWeekId && year == lastYearId ) {
                    year++;
                }
                lastWeekId = week_of_year;
                lastYearId = year;
                                
                String partitionName = "p" + yearWeekFormat.format(partitionStart) + "_" + year;

                cal.add(Calendar.WEEK_OF_YEAR, 1);
                Date partitionEnd = cal.getTime();

                if (!availablePartitions.remove(partitionName)) {
                    if (line.hasOption("create_missing_partitions")) {
                        System.out.println("Creating " + partitionName + " " + partitionStart + " " + partitionEnd + " " + startOfLastWeek);

                        PreparedStatement createPartition = connection.prepareStatement("ALTER TABLE ccs_rawData "
                                + "    REORGANIZE PARTITION " + FUTURE_PARTITION + " INTO ("
                                + "    partition " + partitionName + " values less than (" + partitionEnd.getTime() + "),"
                                + "    partition " + FUTURE_PARTITION + " VALUES LESS THAN maxvalue )");
                        createPartition.executeUpdate();
                    } else {
                        missingPartitions = true;
                        System.out.println("Missing " + partitionName + " " + partitionStart + " " + partitionEnd );                        
                    }
                }
                partitionStart = partitionEnd;
            }
            if ( missingPartitions ) {
                System.out.println("To create missing partition re-run this program with the \"create_missing_partitions\" option.");
            }            
            System.out.println();
            
            if ( !removePartitions.isEmpty() ) {
                if (line.hasOption("remove_old_partitions")) {
                    for ( String partitionToRemove : removePartitions) {
                        String backupTableName = "ccs_rawData_"+partitionToRemove+"_backup";
                        System.out.println("Moving partition to backup table "+backupTableName);
                        
                    }
                } else {
                    System.out.println("To remove old partition re-run this program with the \"remove_old_partitions\" option.");                    
                }
            }
            
        }
    }

    private static Date getBeginningOfWeek(Calendar cal) {
        cal.set(Calendar.HOUR_OF_DAY, 0); // ! clear would not reset the hour of day !
        cal.clear(Calendar.MINUTE);
        cal.clear(Calendar.SECOND);
        cal.clear(Calendar.MILLISECOND);
        cal.clear(Calendar.AM_PM);

        // get start of this week in milliseconds
        cal.set(Calendar.DAY_OF_WEEK, cal.getFirstDayOfWeek());
        return cal.getTime();
    }
    
    private static void backupPartition(Connection c, String partitionName, String tableName) throws SQLException {

        //Create backup table
        PreparedStatement backupPartition = c.prepareStatement("CREATE TABLE "+tableName+" (" 
                +"id bigint NOT NULL AUTO_INCREMENT, doubleData DOUBLE,stringData VARCHAR(255),"
                +"TIME bigint NOT NULL,dataDescId bigint,PRIMARY KEY (id, TIME),"
                +"INDEX "+partitionName+"_time_i (TIME),INDEX "+partitionName+"_dataDeId_i (dataDescId) )");
        backupPartition.executeUpdate();
        
        

    }

}
