package org.lsst.ccs.localdb.configdb.model;

import javax.persistence.*;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import org.hibernate.annotations.Immutable;
import org.lsst.ccs.localdb.statusdb.model.AgentDesc;

/**
 * Represents the static configuration information of an agent.
 * <BR/>
 * The description of a subsystem consists in : 
 * <UL>
 * <LI/> The name of the agent on the buses
 * <LI/> The list of {@link ConfigurationParameter} for this agent.
 * </UL>
 */
@Entity
@Immutable
@Table(uniqueConstraints = @UniqueConstraint(columnNames = "hashMD5"))
public class BaseDescription implements Serializable {

    private static final long serialVersionUID = 1931407188397328132L;
    
    private long id ;
    
    /** The subsystem name on the buses. */
    private AgentDesc agentDesc;

    /** the set containing all parameter descriptions. */
    private Map<ParameterPath,ConfigurationParameter> parameterDescriptions = new HashMap<>();
    
    private List<Description> compatibleDescriptions = new ArrayList<>();

    private byte[] hashMD5;

    protected BaseDescription() {
    }

    /**
     * @param agentDesc the corresponding agent description 
     */
    public BaseDescription(AgentDesc agentDesc) {
        this.agentDesc = agentDesc;
    }

    ///////////////////////////// ACCESSORS/MUTATORS

    /**
     * the technical id: zero if the object is not yet registered in database
     *
     * @return the id associated with the subsystem description, as given by the
     * underlying configuration service.
     */
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    public long getId() {
        return id;
    }

    /**
     * used only by reconstruction code
     *
     * @param id
     */
    void setId(long id) {
        this.id = id ;
    }
    
    @ManyToMany(cascade = {CascadeType.PERSIST,CascadeType.ALL}, fetch=FetchType.EAGER)
    @MapKey(name = "parameterPath")
    public Map<ParameterPath, ConfigurationParameter> getConfigurationParameters() {
        return this.parameterDescriptions;
    }
    
    void setConfigurationParameters(Map<ParameterPath, ConfigurationParameter> paramDescriptions) {
        this.parameterDescriptions = paramDescriptions;
    }
    
    @ManyToOne(optional = false)
    @JoinColumn(name="agentName")
    public AgentDesc getAgentDesc() {
        return agentDesc;
    }

    public void setAgentDesc(AgentDesc agentDesc) {
        this.agentDesc = agentDesc;
    }
    
    @Column(columnDefinition = "varbinary(100)", nullable = false)
    public byte[] getHashMD5() {
        return hashMD5 == null ? null : Arrays.copyOf(hashMD5, hashMD5.length);
    }
    
    public void setHashMD5(byte[] hashMD5) {
        this.hashMD5 = hashMD5 == null ? null : Arrays.copyOf(hashMD5, hashMD5.length);
    }
    
    public static byte[] computeHashMD5(AgentDesc ad, List<ConfigurationParameter> parameters) {
        Map<ParameterPath, ConfigurationParameter> map = new TreeMap<>();
        for (ConfigurationParameter cp : parameters) {
            map.put(cp.getParameterPath(), cp);
        }
        StringBuilder sb = new StringBuilder(ad.getAgentName()).append("*");
        for (Map.Entry<ParameterPath, ConfigurationParameter> e : map.entrySet()) {
            ParameterPath name = e.getKey();
            ConfigurationParameter cpi = e.getValue();
            sb.append(name).append(",").append(cpi.getCategory()).append(",")
                    .append(cpi.getType()).append(",")
                    .append(cpi.isFinal()).append(":");
        }
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            return md.digest(sb.toString().getBytes("UTF-8"));
        } catch (NoSuchAlgorithmException | UnsupportedEncodingException e1) {
            throw new RuntimeException(e1);
        }
    }
    
    @OneToMany(mappedBy = "baseDescription", fetch = FetchType.EAGER)
    public List<Description> getCompatibleDescriptions() {
        return compatibleDescriptions;
    }
    
    public void setCompatibleDescriptions(List<Description> l) {
        this.compatibleDescriptions = l;
    }
    
    @Override
    public String toString() {
        return "{" +
                "id=" + getId() +
                ";descriptions=" + this.getConfigurationParameters() +
                '}';
    }

    /**
     * Looks for a parameter description that matches the given path string
     * @param pathString
     * @return the parameter description that matches the given path String
     */
    public ConfigurationParameter fetch(String pathString) {
        for(ConfigurationParameter cp : parameterDescriptions.values()) {
            if (cp.getParameterPath().toString().equals(pathString)) {
                return cp;
            }
        }
        throw new RuntimeException("cannot find configuration parameter for " + pathString + " in BaseDescription " + toString());
    }

}

