package org.lsst.ccs.localdb.configdb.model;

import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import javax.persistence.Cacheable;
import javax.persistence.CascadeType;
import javax.persistence.FetchType;
import javax.persistence.GenerationType;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.MapKeyJoinColumn;
import javax.persistence.Transient;

/**
 * Represents the detailed configuration state of a given subsystem, within a
 * given time interval.
 * <p>
 * A configuration run records the following configuration information, for a
 * given agent, at a given time :
 * <ul>
 * <li/>The agent name
 * <li/>The running description, as a {@link Description}
 * <li/>The running named configuration for each category, as a map of
 * {@link Configuration}
 * <li/>Any run-time configuration parameter change that occurs on top of the
 * current running named configuration, as a set of
 * {@link ConfigurationParameterChange}
 * </ul>
 * Each time a configuration operation is performed on the subsystem (change,
 * commit, save, load, drop), a new entity of this object is created and
 * persisted.
 *
 * @author LSST CCS Team
 */

@Entity
@Cacheable
@org.hibernate.annotations.Cache(
        usage = org.hibernate.annotations.CacheConcurrencyStrategy.NONSTRICT_READ_WRITE, 
        region = "org.lsst.ccs.localdb.configdb.model.ConfigurationRun"
)
public class ConfigurationRun implements Serializable  {
    private static final long serialVersionUID = 927127673789827728L;
    
    private long id ;

    /** The map of running configurations, by categories. */
    private Configuration configuration;
    
    /** The description that also contains the final parameter values. */
    private Description description;
    
    /** A set of runtime changes happening during the active run. */
    private Map<ParameterPath, ConfigurationParameterValue> runtimeChanges = new HashMap<>();
    
    private long tstart ;
    
    private long tstop = PackCst.STILL_VALID;
    
    private GlobalConfiguration globalConfiguration;

    protected ConfigurationRun() {
    }
    
    public ConfigurationRun(Description description, Configuration config, long startTime) {
        this.tstart = startTime;
        configuration = config;
        this.description = description;
    }
    
    public ConfigurationRun(ConfigurationRun other, long startTime) {
        this(other.description, other.configuration, startTime);
        for(ConfigurationParameterValue cpv : other.runtimeChanges.values()) {
            runtimeChanges.put(cpv.getPath(), cpv);
        }
    }

    ///////////////////////////// ACCESSORS/MUTATORS

    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    public long getId() {
        return id;
    }
    
    void setId(long id) {
        this.id = id;
    }
    
    public void setDescription(Description desc) {
        this.description = desc;
    }
    
    @ManyToOne(optional = false, fetch=FetchType.EAGER, cascade = CascadeType.MERGE)
    public Description getDescription() {
        return description;
    }
    
    @ManyToOne(optional = false, fetch=FetchType.EAGER, cascade=CascadeType.ALL)
    public Configuration getConfiguration() {
        return configuration;
    }

    public void setConfiguration(Configuration configuration) {
        this.configuration = configuration;
    }

    @ManyToMany(cascade = {CascadeType.ALL, CascadeType.PERSIST})
    @MapKeyJoinColumn(name="configurationParameter_parameterPath")
    public Map<ParameterPath, ConfigurationParameterValue> getRuntimeChanges() {
        return runtimeChanges;
    }

    public void setRuntimeChanges(Map<ParameterPath, ConfigurationParameterValue> runtimeChanges) {
        this.runtimeChanges.putAll(runtimeChanges);
    }
    
    public void addRuntimeChange(ConfigurationParameterValue cpc) {
        runtimeChanges.put(cpc.getPath(), cpc);
    }
    
    public ConfigurationParameterValue getConfigurationParameterValueForRun(ParameterPath path) {
        if(runtimeChanges.containsKey(path)) {
            return runtimeChanges.get(path);
        } else {
            return configuration.getConfigurationParameterValues().get(path);
        }
    }
    
    public long getTstart() {
        return tstart;
    }

     void setTstart(long tstart) {
        this.tstart = tstart;
    }

    public long getTstop() {
        return tstop;
}

    public void setTstop(long tstop) {
        this.tstop = tstop;
    }

    @ManyToOne(optional = true, fetch = FetchType.EAGER)
    public GlobalConfiguration getGlobalConfiguration() {
        return  globalConfiguration;
    }

    public void setGlobalConfiguration(GlobalConfiguration globalConfiguration) {
            this.globalConfiguration = globalConfiguration;
    }
    
    @Transient
    public boolean isDirty() {
        return !runtimeChanges.isEmpty();
    }
    
    @Transient
    public String getConfigurationDescriptionString() {
        StringBuilder sb = new StringBuilder(configuration.getConfigurationDescriptionString());
        if (isDirty()) {
            sb.append("*");
        }
        return sb.toString();
    }
    
}
