package org.lsst.ccs.localdb.dao;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.hibernate.Query;
import org.hibernate.Session;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.localdb.statusdb.model.AgentDesc;
import org.lsst.ccs.localdb.statusdb.model.AlertDesc;
import org.lsst.ccs.localdb.statusdb.model.StatTimeInterval;
import org.lsst.ccs.localdb.statusdb.model.AgentState;
import org.lsst.ccs.localdb.statusdb.model.BaseState;
import org.lsst.ccs.localdb.statusdb.model.InnerStateDesc;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.model.StateBundleDesc;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Simplistic access to the database, through static methods.
 * @author LSST CCS Team
 */
public class LocaldbFacade {
    
    private LocaldbFacade() {}

    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.statusdb");

    /**
     * Returns the persistent AgentDesc corresponding to the AgentInfo.
     * If not found in the first place, creates it and persists it.
     * @param agentInfo
     * @param sess
     * @return an AgentDesc in persisted state.
     */
    public static AgentDesc getAgentDesc(AgentInfo agentInfo, Session sess) {
        String key = agentInfo.getName();
        AgentDesc ad = sess.get(AgentDesc.class, key);

        if (ad == null) {
            ad = new AgentDesc(agentInfo);
            sess.persist(ad);
            log.debug("Added Agent description for " + key +" : " + ad.getAgentName());
        }
        return ad;
    }
    
    /** -- Methods for Alert persistence. */
    
    /**
     *
     * @param agentDesc
     * @param alert
     * @param sess
     * @return an AlertDesc in persisted state.
     */
    public static AlertDesc getAlertDescOrPersist(AgentDesc agentDesc, Alert alert, Session sess) {
        AlertDesc ad = getAlertDesc(agentDesc, alert.getAlertId(), sess);
        if (ad == null) {
            ad = new AlertDesc();
            ad.setAgentDesc(agentDesc);
            ad.setAlertId(alert.getAlertId());
            ad.setAlertDescription(alert.getDescription());
            sess.persist(ad);
            log.debug("Added Alert description for " + agentDesc.getAgentName()+"/"+alert.getAlertId() + " : " + ad.getAlertId());
        }
        return ad;
    }
    
    /**
     * Looks for a persisted AlertDesc corresponding to the given parameters.
     * Returns null if not found.
     * @param agentDesc
     * @param alertId
     * @param sess
     * @return the corresponding AlertDesc, or null if not found.
     */
    public static AlertDesc getAlertDesc(AgentDesc agentDesc,  String alertId, Session sess) {
        Query q = sess.getNamedQuery("findAlertDesc");
        q.setString("id", alertId).setString("name", agentDesc.getAgentName());
        return (AlertDesc)q.uniqueResult();
    }

    /** -- Methods for State persistence. */
    
    /**
     * Returns a persisted InnerStateDesc corresponding to the given parameters.
     * @param ai
     * @param enumClassName
     * @param enumValue
     * @param sess
     * @return an InnerStateDesc in persisted state.
     */
    private static InnerStateDesc getInnerStateDesc(AgentDesc ad, String enumClassName, String enumValue, Session sess) {
        Query q = sess.getNamedQuery("findInnerStateDesc");
        q.setEntity("agentDesc", ad).setString("enumClassName", enumClassName).setString("enumValue", enumValue);
        InnerStateDesc isd = (InnerStateDesc)q.uniqueResult();
        if (isd == null) {
            isd = new InnerStateDesc(ad);
            isd.setEnumClassName(enumClassName);
            isd.setEnumValue(enumValue);
            sess.persist(isd);
            log.debug("Added Inner State description description for " + isd.toString());
        }
        return isd;
    }
    
    /**
     * Reads the internal states of a StateBundle and returns a StateBundleDesc
     * entity in persisted state.
     *
     * @param ad
     * @param sb
     * @param existingSBDs
     * @param sess
     * @return
     */
    private static StateBundleDesc getStateBundleDesc(AgentDesc ad, StateBundle sb, List<StateBundleDesc> existingSBDs, Session sess) {
        Map<String, InnerStateDesc> componentStates = new HashMap<>();
        Map<String, String> allStates = new HashMap<>(sb.getAllStatesAsStrings());
        allStates.remove("PhaseState");
        allStates.remove("OperationalState");
        allStates.remove("ConfigurationState");
        allStates.remove("CommandState");
        allStates.remove("AlertState");
        for(Map.Entry<String, String> state : allStates.entrySet()) {
            componentStates.put(state.getKey(), getInnerStateDesc(ad, state.getKey(), state.getValue(), sess));
        }
        for(StateBundleDesc sdb : existingSBDs) {
            if (sdb.getComponentStates().equals(componentStates)) {
                return sdb;
            }
        }
        // Not found : creating and persisting it
        StateBundleDesc sdb = new StateBundleDesc(ad);
        sdb.setComponentStates(componentStates);
        sess.persist(sdb);
        return sdb;
    }

    /**
     * Returns a persisted BaseState corresponding to the given StateBundle.
     * @param sb
     * @param sess
     * @return a BaseState in persisted state.
     */
    private static BaseState getBaseState(StateBundle sb, Session sess) {
        BaseState bs = new BaseState(sb);
        Query q = sess.getNamedQuery("findBaseState")
                .setString("alertSt", bs.getAlertState())
                .setString("phaseSt", bs.getPhaseState())
                .setString("commandSt", bs.getCommandState())
                .setString("operationalSt", bs.getOperationalState())
                .setString("configurationSt", bs.getConfigState());

        log.debug("sending query : " + q.getQueryString());
        BaseState persistedBS = (BaseState)q.uniqueResult();
        if (persistedBS == null) {
            sess.persist(bs);
            log.debug("Added BaseState description for " + bs.toString());
            return bs;
        } else {
            return persistedBS;
        }
    }

    /**
     * Returns the AgentState corresponding to the given stateBundle.
     * An AgentState is the combination of the 5 framework-defined states and of
     * additional subsystem-developer-defined states.
     * @param ad the agent description in persisted state
     * @param sb
     * @param sess
     * @return an AgentState in persisted state.
     */
    public static AgentState getAgentState(AgentDesc ad, StateBundle sb, Session sess) {
        // Getting the five core defined states
        BaseState bs = getBaseState(sb, sess);
        // Getting the subsystem specific states
        Map<String, String> internalStates = new HashMap<>(sb.getAllStatesAsStrings());
        // The 5 base states have to be removed since they are handled differently
        internalStates.remove("PhaseState");
        internalStates.remove("OperationalState");
        internalStates.remove("ConfigurationState");
        internalStates.remove("CommandState");
        internalStates.remove("AlertState");
        
        Map<String, InnerStateDesc> internalStateDescs = new HashMap<>();
        for (Map.Entry<String, String> entry : internalStates.entrySet()) {
            internalStateDescs.put(entry.getKey(), getInnerStateDesc(ad, entry.getKey(), entry.getValue(), sess));
        }
        // getting the component states
        Query q = sess.createQuery("from StateBundleDesc sbd where sbd.agentDesc=:ad").setEntity("ad", ad);
        List<StateBundleDesc> list = q.list();
        
        Map<String, StateBundleDesc> componentStateBundles = new HashMap<>();

        for(String comp : sb.getComponentsWithStates()) {
            StateBundle csb = sb.getComponentStateBundle(comp);
            StateBundleDesc sbd = getStateBundleDesc(ad, csb, list, sess);
            componentStateBundles.put(comp, sbd);
        }
        
        Query asq = sess.getNamedQuery("findAgentState").setEntity("agentDesc", ad).setLong("baseStateId", bs.getId());
//                .setParameterList("innerStates", internalStateDescs);
        List<AgentState> possibleStates = asq.list();
        
        AgentState as = null;
        
        // it might already be persisted the new fashion
        Map<String, StateBundleDesc> componentStateBundlesWithTop = new HashMap<>(componentStateBundles);
        componentStateBundlesWithTop.put("", getStateBundleDesc(ad, sb, list, sess));
        for (AgentState a : possibleStates) {
            if ( a.getComponentStates().equals(componentStateBundlesWithTop) ) {
                as = a;
                break;
            }
        }
        if(as == null && componentStateBundles.isEmpty()) {
            // Look if is persisted the old fashion
            for (AgentState a : possibleStates) {
                if ( a.getInnerStates().equals(internalStateDescs) ) {
                    if(!a.getInnerStates().isEmpty()) {
                        log.warn("some agent states are persisted with a deprecated schema : run the migration application \"MigrateAgentStates\"");
                    }
                    as = a;
                    break;
                }
            }
        } 
        if (as == null) {
            as = new AgentState(ad);
            as.setBaseState(bs);
            // persist it in the new fashion
            as.setComponentStates(componentStateBundlesWithTop);
            sess.persist(as);
            log.debug("Added AgentState for " + as.toString());
        }
        return as;
    }

    public static StatTimeInterval getStatTimeInterval(long binWidth, long dataTime, Session sess) {      
        //Get the time intervals spanning the span (t_start,t_end). 
        //This means startTime<=t_end and startTime+binWidth>t_start.
        //Which becomes startTime<=t_end amd startTime>t_start-binWidth.
        //Or startTime<=t2 and statTime>t1 where t2=t_end and t1=t_start-binWidth
        long statTimeIntervalStart = (dataTime / binWidth) * binWidth;
        
        //Get The time interval for this binWidth
        StatTimeInterval statTimeInterval = sess.byNaturalId(StatTimeInterval.class)
                .using("startTime", statTimeIntervalStart)
                .using("binWidth", binWidth)
                .load();
        
        if (statTimeInterval == null) {
            statTimeInterval = new StatTimeInterval(statTimeIntervalStart, binWidth);
            sess.persist(statTimeInterval);
            // No need to flush because StatTimeInterval is queried by naturalId
            // and therefore repeatedly readable from session cache.
        }
        return statTimeInterval;
    }
    
    /**
     * Retrieves a StatData entity corresponding to the given {@code StatDesc} and {@code StatTimeInterval}.
     * @param sd The StatDesc entity
     * @param sti the StatTimeInterval
     * @param flush true to flush the session before persorming the query; false otherwise
     * @return a StatData or null if no StatData was found
     */
    public static StatData getStatDataOrNull(StatDesc sd, StatTimeInterval sti, Session sess) {
        return (StatData)sess.getNamedQuery("findStatData")
                .setEntity("d", sd).setEntity("sti", sti)
                .setCacheable(true)
                .uniqueResult();
    }    
}
