package org.lsst.ccs.localdb.configdb.model;

import javax.persistence.*;
import java.io.Serializable;
import java.util.*;
import org.hibernate.annotations.Immutable;
import org.lsst.ccs.localdb.statusdb.model.AgentDesc;

/**
 * Represents the static configuration information of an agent.
 * <BR/>
 * The description of a subsystem consists in : 
 * <UL>
 * <LI/> The name of the agent on the buses
 * <LI/> The list of {@link ConfigurationParameter} for this agent.
 * </UL>
 */
@Entity
@Immutable
public class Description implements Serializable {

    private static final long serialVersionUID = 1931407188397328132L;
    
    private long id ;

    /**
     * the set containing the parameter descriptions
     */
    private Map<ParameterPath,ConfigurationParameter> parameterDescriptions = new HashMap<>();
    
    /** The subsystem name on the buses. */
    private AgentDesc agentDesc;

    protected Description() {
    }

    /**
     * @param agentDesc the corresponding agent description 
     */
    public Description(AgentDesc agentDesc) {
        this.agentDesc = agentDesc;
    }

    ///////////////////////////// ACCESSORS/MUTATORS

    /**
     * the technical id: zero if the object is not yet registered in database
     *
     * @return the id associated with the subsystem description, as given by the
     * underlying configuration service.
     */
    @Id
    @GeneratedValue
    public long getId() {
        return id;
    }

    /**
     * used only by reconstruction code
     *
     * @param id
     */
    void setId(long id) {
        this.id = id ;
    }

    @ManyToMany(cascade = CascadeType.ALL, fetch=FetchType.EAGER)
    @MapKey(name = "parameterPath")
    public Map<ParameterPath, ConfigurationParameter> getConfigurationParameters() {
        return this.parameterDescriptions;
    }
    
    void setConfigurationParameters(Map<ParameterPath, ConfigurationParameter> paramDescriptions) {
        this.parameterDescriptions = paramDescriptions;
    }
    
    @ManyToOne(optional = false)
    @JoinColumn(name="agentName")
    public AgentDesc getAgentDesc() {
        return agentDesc;
    }

    public void setAgentDesc(AgentDesc agentDesc) {
        this.agentDesc = agentDesc;
    }
    
    @Override
    public String toString() {
        return "{" +
                "id=" + getId() +
                ";descriptions=" + this.getConfigurationParameters() +
                '}';
    }

    /**
     * Looks for a parameter description that matches the given path string
     * @param pathString
     * @return the parameter description that matches the given path String
     */
    public ConfigurationParameter fetch(String pathString) {
        return parameterDescriptions.get(ParameterPath.valueOf(pathString));
    }

}

