package org.lsst.ccs.localdb.statusdb.utils;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.stat.EntityStatistics;
import org.hibernate.stat.QueryStatistics;
import org.hibernate.stat.SecondLevelCacheStatistics;
import org.hibernate.stat.Statistics;
import org.lsst.ccs.bus.annotations.SkipEncoding;
import org.lsst.ccs.localdb.statusdb.model.AgentState;
import org.lsst.ccs.localdb.statusdb.model.AlertDesc;
import org.lsst.ccs.localdb.statusdb.model.BaseState;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.InnerStateDesc;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.model.StatTimeInterval;

/**
 * Hibernate cache statistics to be sent on the buses.
 * @author LSST CCS team
 */
public class CacheStatistics implements Serializable {
    private static final long serialVersionUID = -6955983311743974515L;
    
    @SkipEncoding
    private static final String[] namedQueries = new String[] {
           "findAlertDesc","findInnerStateDesc","findDataDesc","findAgentState","findBaseState", "findStatTimeInterval", "findStatData", "findStatDesc"};
    
    @SkipEncoding
    private static final Class[] cachedEntities = new Class[]{AlertDesc.class,InnerStateDesc.class, DataDesc.class, AgentState.class, BaseState.class, StatTimeInterval.class, StatData.class, StatDesc.class};
    
    private final Map<String, QueryCacheStats> queries = new HashMap<>();
    
    private final Map<String, SLCacheStats> slc = new HashMap<>();
    
    private final Map<String, EntityStats> entities = new HashMap<>();
    
    public CacheStatistics(SessionFactory fac) {
        Statistics stats = fac.getStatistics();
        Session sess = fac.openSession();
        for (String query : namedQueries) {
            queries.put(query, new QueryCacheStats(query, stats.getQueryStatistics(
                    sess.getNamedQuery(query).getQueryString())));
        }
        
        for (Class cls : cachedEntities) {
            slc.put(cls.getSimpleName(), new SLCacheStats(cls.getSimpleName(), stats.getSecondLevelCacheStatistics(cls.getName())));
            entities.put(cls.getSimpleName(), new EntityStats(cls.getSimpleName(), stats.getEntityStatistics(cls.getName())));
        }
        sess.close();
//        stats.clear();
    }
    
    private static class QueryCacheStats implements Serializable {
        private static final long serialVersionUID = 5282982444660929176L;
    
        @SkipEncoding
        private String query=""; 
        private final long hit;
        private final long miss;
        private final long put;
        private final long avgExec;
        
        private QueryCacheStats(String query, QueryStatistics stats) {
            this.query = query;
            hit = stats.getCacheHitCount();
            miss = stats.getCacheMissCount();
            put = stats.getCachePutCount();
            avgExec = stats.getExecutionAvgTime();
        }
      
        @Override
        public String toString() {
            return query + " :\n"
                    + "\t hits : " + hit
                    + "\t misses : " + miss
                    + "\t puts : " + put
                    + "\t average execution time : " + avgExec +"\n";
        }
    };
    
    private static class SLCacheStats implements Serializable {
        private static final long serialVersionUID = -4475368776790422166L;
        @SkipEncoding
        private String region="";
        private final long hit;
        private final long miss;
        private final long put;
        private final long count;
        
        private SLCacheStats(String region, SecondLevelCacheStatistics stats) {
            this.region = region;
            count = stats.getElementCountInMemory();
            hit = stats.getHitCount();
            miss = stats.getMissCount();
            put = stats.getPutCount();
        }
        @Override
        public String toString() {
            return region + " :\n"
                    + "\t hits : " + hit
                    + "\t misses : " + miss
                    + "\t puts : " + put
                    + "\t count : " + count +"\n";
        }
    }
        
    private static class EntityStats implements Serializable {
        private static final long serialVersionUID = 105526900767640499L;

        @SkipEncoding
        private String name ="";
        private final long load;
        private final long fetch;
        
        private EntityStats(String name, EntityStatistics stats) {
            this.name = name;
            this.load = stats.getLoadCount();
            this.fetch = stats.getFetchCount();
        }
        
        @Override
        public String toString() {
            return name + " :\n"
                    + "\t load : " + load
                    + "\t fetch : " + fetch +"\n";
        }
    }
    
    public String toString() {
        StringBuilder sb = new StringBuilder("********** Hibernate statistics **********\n");
        sb.append("\n ***** Query Statistics *****\n");
        SortedSet<String> sortedKeys = new TreeSet<String>(queries.keySet());
        for (String key : sortedKeys) {
            sb.append(queries.get(key).toString());
        }
        sb.append("\n\n ***** Second Level Cache Statistics *****\n");
        sortedKeys = new TreeSet<String>(slc.keySet());
        for (String key : sortedKeys) {
            sb.append(slc.get(key).toString());
        }
        sb.append("\n\n ***** Entity Statistics *****\n");
        sortedKeys = new TreeSet<String>(entities.keySet());
        for (String key : sortedKeys) {
            sb.append(entities.get(key).toString());
        }
        return sb.toString();
    }
    
    
}
