package org.lsst.ccs.localdb.statusdb.trendServer;

import org.lsst.ccs.localdb.statusdb.server.AlertInfo;
import com.sun.jersey.spi.resource.Singleton;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import org.hibernate.Query;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.CommandState;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.bus.states.OperationalState;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.localdb.statusdb.model.AlertDesc;
import org.lsst.ccs.localdb.statusdb.model.BaseState;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.InnerStateDesc;
import org.lsst.ccs.localdb.statusdb.model.MetaDataData;
import org.lsst.ccs.localdb.statusdb.model.PlotData;
import org.lsst.ccs.localdb.statusdb.model.RaisedAlertData;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.model.StateChangeNotificationData;
import org.lsst.ccs.localdb.statusdb.server.AlertEvent;
import org.lsst.ccs.localdb.statusdb.server.ChannelMetaData;
import org.lsst.ccs.localdb.statusdb.server.Data;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.server.StateChange;
import org.lsst.ccs.localdb.statusdb.server.StateInfo;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.statusdb.server.TrendingData.AxisValue;
import org.lsst.ccs.localdb.statusdb.server.TrendingData.DataValue;
import org.lsst.ccs.localdb.statusdb.server.TrendingPlotData;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.utilities.logging.Log4JConfiguration;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * The DataServer receives requests for trending data analysis. It delivers the
 * requested data as the result of requests to the database.
 */
@SuppressWarnings("restriction")
@Path("/dataserver")
@Singleton
public class DataServer {

    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb");
    
    private final SessionFactory fac;

    static {
        /*
         Temporary : setting of logging
         */
        Log4JConfiguration.initialize();
    }
    
    public DataServer() {
        Properties p = new Properties();
        fac = StatusdbUtils.getSessionFactory(p);
        log.fine("Starting Data Server");
    }

    private List<DataChannel> getListOfChannels() {
        
        log.fine("Loading Channel Information");
        long start = System.currentTimeMillis();

        Session sess = fac.openSession();
        Query dataQuery = sess.createQuery("from DataDesc order by dataName");
        dataQuery.setReadOnly(true);
        @SuppressWarnings("unchecked")                
        List<DataDesc> l = dataQuery.list();

        ArrayList<DataChannel> channels = new ArrayList<>();
        double delta = (System.currentTimeMillis() - start)/1000.;
        log.fine("Done with the query " + l.size()+" in "+delta+" seconds");
        for (DataDesc d : l) {
            DataChannel dc = new DataChannel(d);
            dc.getMetadata().put("subsystem", d.getDataPath().getAgentName());
            dc.getMetadata().put("type", d.getDataType());
            channels.add(dc);
            log.debug("retrieving Data Channel path= " + dc.getPathAsString());
        }

        sess.close();
        return channels;
    }

    // TODO this is really a first try...
    // TODO check what happens in case of uneven binning in the db, interrupted
    // data, etc.
    // TODO when recomputing stat data either from raw or stat
    // TODO check bounds where recomputing stat
    @GET
    @Path("/data/{id}")
    public Data getData(@PathParam("id") long id,
            @QueryParam("t1")
            @DefaultValue("-1") long t1,
            @QueryParam("t2")
            @DefaultValue("-1") long t2,
            @QueryParam("flavor") String flavor,
            @QueryParam("n")
            @DefaultValue("30") int nbins) {

        // handling of default values
        // t2 default is now
        // t1 default is t2 - 1 hour
        // rawPreferred is false
        // nbins is 30
        boolean useStat = false;
        boolean useRaw = false;
        if ("stat".equals(flavor)) {
            useStat = true;
        } else if ("raw".equals(flavor)) {
            useRaw = true;
        } else {
            flavor = "unspecified";
        }

        if (t2 < 0) {
            t2 = System.currentTimeMillis();
        }
        if (t1 < 0) {
            t1 = t2 - 3600000L;
        }

        log.fine("request for data " + id + "[" + t1 + "," + t2 + "] "
                + flavor + " " + nbins);

        // what do we have?
        long rawId = id;

        if (useRaw) {
            log.debug("sending raw data");
            return exportRawData(rawId, t1, t2);
        } else if (useStat) {

            StatDesc statSource = null;
            long statSourceN = -1;
            Map<StatDesc, Long> stats = getAvailableStats(rawId, t1, t2);

            log.fine("stats :");
            // how do we create the data?
            for (Map.Entry<StatDesc, Long> s : stats.entrySet()) {
                log.debug("  " + s.getKey().getId() + " "
                        + s.getKey().getTimeBinWidth() + " : " + s.getValue());
                                
                long n = s.getValue();
                if (n > nbins / 2) {
                    // this one has enough bins
                    // TODO # of samples is not enough, for instance raw could
                    // have more samples
                    // but covering only the recent part of the time range.
                    if (statSource != null) {
                        if (n < statSourceN) {
                            statSource = s.getKey();
                            statSourceN = n;
                        }
                    } else {
                        statSource = s.getKey();
                        statSourceN = n;
                    }
                }
            }
            if (statSource != null) {
                log.fine("sending stat from stat sampling "
                        + statSource.getTimeBinWidth() + " nsamples "
                        + statSourceN);
                return exportStatDataFromStat(statSource, t1, t2, nbins);
            }
        }

        log.debug("sending stat from raw");
        return exportStatDataFromRaw(rawId, t1, t2, nbins);
    }
    
    

    public List<ChannelMetaData> getMetadata(int channelId) {
        return getMetadata(channelId, -1, -1);
    }

    public List<ChannelMetaData> getMetadata(int channelId, long t1, long t2) {

        Session sess = fac.openSession();
        String queryStr = "from MetaDataData where dataDesc.id = :id ";
        if (t1 > -1) {
            queryStr += "and (endTime > :t1 or startTime = -1) ";
        }
        if (t2 > -1) {
            queryStr += "and startTime < :t2 ";
        }

        Query q = sess.createQuery(queryStr);
        q.setReadOnly(true);
        q.setParameter("id", (long)channelId);
        if (t1 > -1) {
            q.setParameter("t1", t1);
        }
        if (t2 > -1) {
            q.setParameter("t2", t2);
        }

        @SuppressWarnings("unchecked")
        List<MetaDataData> l = q.list();

        sess.close();
        List<ChannelMetaData> out = new ArrayList<ChannelMetaData>();
        for (MetaDataData md : l) {
            out.add(new ChannelMetaData(md));
        }
        return out;

    }

    @GET
    @Path("/channelinfo/{id}")
    public ChannelMetaData.ChannelMetadataList getMetadataList(@PathParam("id") long channelId) {
        long rawId = channelId;
        return new ChannelMetaData.ChannelMetadataList(getMetadata((int) rawId));
    }

    /**
     *
     * @return the whole channels list for all CCS.
     */
    @GET
    @Path("/listchannels")
    public DataChannel.DataChannelList getChannels() {
        return new DataChannel.DataChannelList(getListOfChannels());
    }

    /**
     *
     * @param subsystemName
     * @return a channels list for a subsystem
     */
    @GET
    @Path("/listchannels/{subsystem}")
    public DataChannel.DataChannelList getChannels(
            @PathParam("subsystem") String subsystemName) {
        List<DataChannel> channels = getListOfChannels();
        ArrayList<DataChannel> subChannels = new ArrayList<DataChannel>();
        for (DataChannel dc : channels) {
            if (dc.getPath()[0].equals(subsystemName)) {
                subChannels.add(dc);
            }
        }
        return new DataChannel.DataChannelList(subChannels);

    }
    
    /**
     *
     * @param partialPath
     * @param level
     * @return the list of channels within a partial path and a level
     */
    public DataChannel[] getChannels(String partialPath, int level) {
        // TODO
        return null;
    }

    /**
     * Return all available channels for a given keyword.
     *
     * @param keyword
     * @return channels list
     */
    public DataChannel[] getChannelsByKeywork(String keyword) {
        // TODO
        return null;
        // the keyword can be the name of the published value or a substring of
        // the name.

    }

    protected Map<StatDesc, Long> getAvailableStats(long rawId, long t1, long t2) {
        Session sess = fac.openSession();
        Map<StatDesc, Long> m = new HashMap<StatDesc, Long>();
        
        //Should we really perform this query?
        Query q = sess
                .createQuery("select d, count(x) from StatDesc d, StatData x where d.dataDesc.id = :id "
                        + "and x.statDesc = d and (x.statTimeInterval.startTime+d.timeBinWidth) >= :t1 and x.statTimeInterval.startTime <= :t2 group by d.id order by d.timeBinWidth");
        q.setReadOnly(true);
        q.setLong("id", rawId);
        q.setParameter("t1", t1);
        q.setParameter("t2", t2);
        @SuppressWarnings("unchecked")
        List<Object[]> l = q.list();
        for (Object[] r : l) {
            if (r[0] == null) {
                continue;// crazy
            }
            m.put((StatDesc) r[0], (Long) r[1]);
        }
        sess.close();
        return m;
    }

    protected long getAvailableRawData(long rawId, long t1, long t2) {
        Session sess = fac.openSession();
        Query q = sess
                .createQuery("select count(r) from RawData r where r.dataDesc.id = :id and r.tstamp between :t1 and :t2 order by r.tstamp");
        q.setReadOnly(true);
        q.setParameter("id", rawId);
        q.setParameter("t1", t1);
        q.setParameter("t2", t2);
        long n = (Long) q.uniqueResult();
        sess.close();
        return n;
    }

    protected List<RawData> getRawData(long id, long t1, long t2) {
        Session sess = fac.openSession();

        Query q = sess
                .createQuery("from RawData r where r.dataDesc.id = :id and r.time between :t1 and :t2 order by r.time");
        q.setReadOnly(true);
        q.setParameter("id", id);
        q.setParameter("t1", t1);
        q.setParameter("t2", t2);
        @SuppressWarnings("unchecked")
        List<RawData> l = q.list();
        log.debug("retrieved raw data " + id + "[" + t1 + "," + t2 + "] : "
                + l.size());
        sess.close();

        return l;
    }

    protected List<StatData> getStatData(StatDesc statDesc, long t1, long t2) {
        Session sess = fac.openSession();

        Query q = sess
                .createQuery("from StatData r where r.statDesc.id = :id and r.statTimeInterval.startTime >= :t1 and r.statTimeInterval.startTime <= :t2 order by r.statTimeInterval.startTime");
        q.setReadOnly(true);
        q.setParameter("id", statDesc.getId());
        q.setParameter("t1", t1-statDesc.getTimeBinWidth());
        q.setParameter("t2", t2);
        @SuppressWarnings("unchecked")
        List<StatData> l = q.list();
        log.debug("retrieved stat data " + statDesc.getId() + "[" + t1 + "," + t2 + "] : "
                + l.size());
        sess.close();
        return l;
    }

    protected Data exportRawData(long rawId, long t1, long t2) {
        List<RawData> l = getRawData(rawId, t1, t2);

        Data d = new Data();
        d.setMetaDataData(getMetadata((int) rawId, t1, t2));

        TrendingData[] data = new TrendingData[l.size()];
        for (int i = 0; i < l.size(); i++) {
            RawData r = l.get(i);
            TrendingData dt = new TrendingData();
            data[i] = dt;
            long tStamp = r.getTime();
            AxisValue axisValue = new AxisValue("time", tStamp);
            dt.setAxisValue(axisValue);
            DataValue[] dataValue = new DataValue[1];
            Double dd = r.getDoubleData();
            dataValue[0] = new DataValue("value", dd == null ? 0 : dd);
            dt.setDataValue(dataValue);
        }
        d.getTrendingResult().setTrendingDataArray(data);

        return d;
    }

    protected Data exportStatDataFromRaw(long rawId, long t1, long t2, int nsamples) {

        Session sess = fac.openSession();

        SQLQuery q = sess
                .createSQLQuery("select tlow, thigh, datasum/entries as mean, "
                        + " case when entries > 1 then "
                        + " sqrt((datasumsquared - datasum*datasum/entries)/(entries-1)) "
                        + " else 0.0 end as rms, "
                        + " mindata, maxdata "
                        + " from ( SELECT MIN(rd.time) AS tlow, MAX(rd.time) AS thigh, "
                        + " SUM(rd.doubleData) AS datasum, SUM(rd.doubleData*rd.doubleData) AS datasumsquared, "
                        + " min(rd.doubleData) as mindata, max(rd.doubleData) as maxdata, "
                        + " count(1) AS entries from ccs_rawData rd where rd.dataDescId = :id and time >= :t1 "
                        + " and time <= :t2 group by floor(rd.time/:deltat) ) accumulated where entries > 0 ORDER BY tlow DESC");
        q.setReadOnly(true);

        long deltat = (t2 - t1) / nsamples;
        
        q.setParameter("id", rawId);
        q.setParameter("t1", t1);
        q.setParameter("t2", t2);
        q.setParameter("deltat", deltat);
        @SuppressWarnings("unchecked")
        List<Object[]> l = q.list();

        sess.close();

        List<TrendingData> trendingDataList = new ArrayList<>();
        for (Object[] obj : l) {

            if (obj[2] != null) {
                long low = ((BigInteger) obj[0]).longValue();
                long high = ((BigInteger) obj[1]).longValue();
                TrendingData dt = new TrendingData();
                dt.setAxisValue(new AxisValue("time", (low + high) / 2, low, high));
                DataValue[] dataValue = new DataValue[4];

                double value = ((Double) obj[2]).doubleValue();
                double rms = 0;
                if (obj[3] != null) {
                    rms = ((Double) obj[3]).doubleValue();
                }
                double min = ((Double) obj[4]).doubleValue();
                double max = ((Double) obj[5]).doubleValue();
                dataValue[0] = new DataValue("value", value);
                dataValue[1] = new DataValue("rms", rms);
                dataValue[2] = new DataValue("min", min);
                dataValue[3] = new DataValue("max", max);
                dt.setDataValue(dataValue);
                trendingDataList.add(dt);
            }
        }

        TrendingData[] trendingData = trendingDataList.toArray(new TrendingData[0]);
        
        Data d = new Data();
        d.setMetaDataData(getMetadata((int) rawId, t1, t2));
        d.getTrendingResult().setTrendingDataArray(trendingData);

        return d;
    }

    protected Data exportStatDataFromStat(StatDesc statDesc, long t1, long t2, int nsamples) {
        List<StatData> in = getStatData(statDesc, t1, t2);

        int n = in.size();
        int rebin = 1;
        int nout = n;
        if (n > nsamples * 3) {
            rebin = n / nsamples;
            nout = (n + rebin - 1) / rebin;
            log.debug("will rebin stat by " + rebin + " : " + nout);
        }

        Data d = new Data();
        d.setMetaDataData(getMetadata((int) statDesc.getDataDesc().getId(), t1, t2));

        d.getTrendingResult().setTrendingDataArray(new TrendingData[nout]);
        int iout = 0;
        int ibin = 0;
        double sum = 0;
        double s2 = 0;
        long nsamp = 0;
        long low = 0;
        double max = Double.MIN_NORMAL;
        double min = Double.MAX_VALUE;
        for (StatData sd : in) {
            if ( ibin == 0 ) {
                sum = 0;
                s2 = 0;
                nsamp = 0;
                low = sd.getStatTimeInterval().getStartTime();
                max = Double.MIN_NORMAL;
                min = Double.MAX_VALUE;
            }
            sum += sd.getSum();
            s2 += sd.getSum2();
            if (sd.getMax() > max) {
                max = sd.getMax();
            }
            if (sd.getMin() < min) {
                min = sd.getMin();
            }
            nsamp += sd.getN();
            if (ibin == rebin - 1 || (iout * rebin + ibin == n - 1)) {
                log.debug("storing for " + iout + " from " + (ibin + 1)
                        + " bins");
                TrendingData dt = new TrendingData();
                d.getTrendingResult().getTrendingDataArray()[iout] = dt;
                dt.setAxisValue(new AxisValue("time",
                        (low + sd.getStatTimeInterval().getEndTime()) / 2, low, sd.getStatTimeInterval().getEndTime()));
                DataValue[] dataValue = new DataValue[4];
                dataValue[0] = new DataValue("value", sum / nsamp);
                dataValue[1] = new DataValue("rms", s2 / nsamp - (sum / nsamp)
                        * (sum / nsamp));
                dataValue[2] = new DataValue("min", min);
                dataValue[3] = new DataValue("max", max);
                dt.setDataValue(dataValue);
                iout++;
                ibin = 0;
            } else {
                ibin++;
            }
        }

        return d;
    }
    
    // -- State changes
    /**
     * Lists all the existing states for a given subsystem.
     * @param subsystem the subsystem name.
     * @return 
     */
    @GET
    @Path("liststates/{subsystem}")
    public StateInfo.StateInfoList getStateInfo(
            @PathParam("subsystem") String subsystem) {
        Session sess = fac.openSession();
        List<StateInfo> res = new ArrayList<>();
        
        // Fetching the core states
        Query q = sess.createQuery("from BaseState bs").setReadOnly(true);
        List<BaseState> bsl = q.list();
        Map<Class, StateInfo> baseStateInfoMap = new HashMap<>();
        baseStateInfoMap.put(AlertState.class, new StateInfo(AlertState.class.getName()));
        baseStateInfoMap.put(ConfigurationState.class, new StateInfo(ConfigurationState.class.getName()));
        baseStateInfoMap.put(CommandState.class, new StateInfo(CommandState.class.getName()));
        baseStateInfoMap.put(PhaseState.class, new StateInfo(PhaseState.class.getName()));
        baseStateInfoMap.put(OperationalState.class, new StateInfo(OperationalState.class.getName()));
        for (BaseState bs : bsl) {
            baseStateInfoMap.get(AlertState.class).getStateValues().add(bs.getAlertState());
            baseStateInfoMap.get(ConfigurationState.class).getStateValues().add(bs.getConfigState());
            baseStateInfoMap.get(CommandState.class).getStateValues().add(bs.getCommandState());
            baseStateInfoMap.get(PhaseState.class).getStateValues().add(bs.getPhaseState());
            baseStateInfoMap.get(OperationalState.class).getStateValues().add(bs.getOperationalState());
        }
        for (StateInfo si : baseStateInfoMap.values()) {
            res.add(si);
        }
        
        // Fetching the internal states
        q = sess.createQuery("from InnerStateDesc isd where isd.agentDesc.agentName=:agentName")
                .setString("agentName",subsystem).setReadOnly(true);
        Map<String, StateInfo> innerStateInfoMap = new HashMap<>();
        List<InnerStateDesc> isdl = q.list();
        for (InnerStateDesc isd : isdl) {
            String isdClassName = isd.getEnumClassName();
            StateInfo si = innerStateInfoMap.get(isdClassName);
            if (si == null) {
                si = new StateInfo(isdClassName);
                innerStateInfoMap.put(isdClassName, si);
            }
            si.getStateValues().add(isd.getEnumValue());
        }
        for (StateInfo si : innerStateInfoMap.values()) {
            res.add(si);
        }
        sess.close();
        return new StateInfo.StateInfoList(res);
    }
    
    /**
     * Gets the state changes events for the given list of states, between the
     * time interval {@code [t1,t2]}
     *
     * @param subsystem
     * @param states if empty, returns all state changes for all found states
     * @param t1
     * @param t2
     * @return
     */
    @GET
    @Path("/statechanges/{subsystem}")
    public StateChange.StateChangesList getStateChangeList (
            @PathParam("subsystem") String subsystem,
            @QueryParam("state")List<String> states,
            @QueryParam("t1") @DefaultValue("-1") long t1,
            @QueryParam("t2") @DefaultValue("-1") long t2) {
        if (t2 < 0) {
            t2 = System.currentTimeMillis();
        }
        if (t1 < 0) {
            t1 = t2 - 3600000L;
        }
        List<StateChangeNotificationData> stateChangeNotifications = getStateChanges(subsystem, t1, t2);
         if (stateChangeNotifications.isEmpty()) {
            return new StateChange.StateChangesList();
        }
        List<StateChange> stateChanges = new ArrayList<>();
        
        // Processing the first state's old state
        StateChange sc = new StateChange();
        sc.setTime(t1);
        Map<String, String> origStates = stateChangeNotifications.get(0).getOldState().asFlatStatesMap();
        if (states == null || states.isEmpty()) {
            for (Map.Entry<String, String> entry : origStates.entrySet()) {
                sc.addStateEvent(entry.getKey(), entry.getValue());
            }
        } else {
            for (String state : states) {
                String stateVal = origStates.get(state);
                if (stateVal == null) {
                    sc.addStateEvent(state, "__UNDEF__");
                } else {
                    sc.addStateEvent(state, stateVal);
                }
            }
        }
        
        stateChanges.add(sc);
        
        for (StateChangeNotificationData scnd : stateChangeNotifications) {

            sc = new StateChange();
            
            // Computing the delta between oldState and newState
            Map<String, String> oldState = scnd.getOldState().asFlatStatesMap();
            Map<String, String>  newState = scnd.getNewState().asFlatStatesMap();
            
            for (Map.Entry<String, String> entry : newState.entrySet()) {
                if (states == null || states.isEmpty() || states.contains(entry.getKey())) {
                    String prevState = oldState.get(entry.getKey());
                    if (prevState == null || !prevState.equals(entry.getValue())) {
                        sc.addStateEvent(entry.getKey(), entry.getValue());
                    }
                }
            }

            // This should not happen
            if (sc.isEmpty()) continue;
            
            sc.setTime(scnd.getTime());
            
            stateChanges.add(sc);
        }
        
        return new StateChange.StateChangesList(stateChanges);
    }
    
    /**
     * Returns the state changes events occurring from a given subsystem between
     * the time intervale [t1;t2]
     *
     * @param subsystemName the subsystem name
     * @param t1 tstart
     * @param t2 tstop
     * @return
     */
    private List<StateChangeNotificationData> getStateChanges(String subsystemName, long t1, long t2) {
        Session sess = fac.openSession();
        Query q = sess.createQuery("from StateChangeNotificationData scnd where scnd.agentDesc.agentName=:name and scnd.time >=:t1 and scnd.time <=:t2")
                .setString("name", subsystemName).setLong("t1", t1).setLong("t2", t2)
                .setReadOnly(true);
        List<StateChangeNotificationData> res = q.list();
        sess.close();
        return res;
    }
    
    // -- Alerts information retrieval
    @Path("/listalerts")
    @GET
    public AlertInfo.AlertInfoList getAlertInfo() {
        return getAlertInfo(null);
    }

    @Path("/listalerts/{subsystem}")
    @GET
    public AlertInfo.AlertInfoList getAlertInfo( @PathParam("subsystem") String subsystem) {
        Session sess = fac.openSession();
        Query q;
        if (subsystem == null) {
            q = sess.createQuery("from AlertDesc ad");
        } else {
            q = sess.createQuery("from AlertDesc ad where ad.agentDesc.agentName=:name").setString("name", subsystem);
        }
        q.setReadOnly(true);
        List<AlertDesc> l = q.list();
        
        List<AlertInfo> res = new ArrayList<>();
        for (AlertDesc ad : l) {
            res.add(new AlertInfo(ad.getId(),ad.getAgentDesc().getAgentName(),ad.getAlertId(),ad.getAlertDescription()));
        }
        sess.close();
        return new AlertInfo.AlertInfoList(res);
    }
    
    /**
     * Returns a list of active alerts. Active alerts are alerts that have been
     * raised during the current subsystem run.
     *
     * @param ids a list of ids.
     * @return
     */
    @Path("activealerts")
    @GET
    public AlertEvent.AlertEventList getActiveAlerts(@QueryParam("id") List<Long> ids) {
        Session sess = fac.openSession();
        Query q;
        if (ids == null || ids.isEmpty()) {
            q = sess.createQuery("from RaisedAlertData arad where arad.active=true");
        } else {
        q = sess.createQuery("from RaisedAlertData arad where arad.alertDesc.id in (:ids) and arad.active=true")
                .setParameterList("ids", ids).setReadOnly(true);
        }
        q.setReadOnly(true);
        List<AlertEvent> res = new ArrayList<>();
        for (RaisedAlertData rad : (List<RaisedAlertData>)q.list()) {
             AlertEvent ae = new AlertEvent(rad.getAlertDesc().getAlertId(), rad.getTime(), rad.getSeverity(), rad.getAlertCause());
            ae.setCleared(rad.getClearingAlert() != null);
            res.add(ae);
        }
        return new AlertEvent.AlertEventList(res);
    }
    
    /**
     * Returns an history of alerts that have been raised within the given time
     * interval.
     *
     * @param ids a list of alertdesc ids.
     * @param t1
     * @param t2
     * @return
     */
    @Path("alerthistory")
    @GET
    public AlertEvent.AlertEventList getAlertHistories(
            @QueryParam("id") List<Long> ids,
            @QueryParam("t1") @DefaultValue("-1") long t1,
            @QueryParam("t2") @DefaultValue("-1") long t2) {
        if (t2 < 0) {
            t2 = System.currentTimeMillis();
        }
        if (t1 < 0) {
            t1 = t2 - 3600000L;
        }
        Session sess = fac.openSession();
        Query q;
        if (ids == null || ids.isEmpty()) {
            q = sess.createQuery("from RaisedAlertData arad");
        } else {
        q = sess.createQuery("from RaisedAlertData arad where arad.alertDesc.id in (:ids)")
                .setParameterList("ids", ids).setReadOnly(true);
        }
        q.setReadOnly(true);
        List<AlertEvent> res = new ArrayList<>();
        for (RaisedAlertData rad : (List<RaisedAlertData>)q.list()) {
            AlertEvent ae = new AlertEvent(rad.getAlertDesc().getAlertId(), rad.getTime(), rad.getSeverity(), rad.getAlertCause());
            ae.setCleared(rad.getClearingAlert() != null);
            res.add(ae);
        }
        return new AlertEvent.AlertEventList(res);
    }
    
    
    
    
    /**
     * Returns a list of trended plots for the given data desc id, between the
     * time interval {@code [t1,t2]}.
     *
     * @param id the datadesc id
     * @param t1
     * @param t2
     * @return
     */
    @GET
    @Path("plotdata/{id}")
    public TrendingPlotData.TrendingPlotDataList getPlotData(
            @PathParam("id") long id, 
            @QueryParam("t1") @DefaultValue("-1") long t1, 
            @QueryParam("t2") @DefaultValue("-1") long t2) {
        Session sess = fac.openSession();
        if (t2 < 0) {
            t2 = System.currentTimeMillis();
        }
        if (t1 < 0) {
            t1 = t2 - 3600000L;
        }
        Query q = sess.createQuery("from PlotData pd where pd.dataDesc.id=:id and pd.time >=:t1 and pd.time <=:t2")
                .setLong("id", id).setLong("t1", t1).setLong("t2", t2)
                .setReadOnly(true);
        List<PlotData> plotdata = q.list();
        List<TrendingPlotData> res = new ArrayList<>();
        for(PlotData pd : plotdata) {
            TrendingPlotData tpd = new TrendingPlotData();
            tpd.setId(id);
            tpd.setPlotData(pd.getPlotData());
            tpd.setTime(pd.getTime());
            res.add(tpd);
        }
        sess.close();
        return new TrendingPlotData.TrendingPlotDataList(res);
    }
    
}
