package org.lsst.ccs.localdb.statusdb.model;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.persistence.Cacheable;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.Index;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.MapKey;
import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import javax.persistence.QueryHint;
import javax.persistence.Table;
import org.hibernate.annotations.Immutable;

/**
 *
 * @author LSST CCS Team
 */
@Entity
@Immutable
@Cacheable
@org.hibernate.annotations.Cache( 
        usage = org.hibernate.annotations.CacheConcurrencyStrategy.READ_WRITE
        ,region = "org.lsst.ccs.localdb.statusdb.model.AgentState"
)
@NamedQueries ( {
    @NamedQuery (
            name = "findAgentState",
            query = "from AgentState ast where "
                    + "ast.agentDesc=:agentDesc "
                    + "and ast.baseState.id=:baseStateId ",
//                    + "and ast.innerStates=:innerStates ",
            hints = {@QueryHint(name="org.hibernate.cacheable", value="true")}
    )
})
@Table(
        indexes = {@Index(columnList = "agentName"),@Index(columnList = "baseStateId")}
)
public class AgentState implements Serializable {
    private static final long serialVersionUID = -8367786148299296222L;
    
    private long id;
    
    private AgentDesc agentDesc; 
    
    private BaseState baseState;
    
    private Map<String, InnerStateDesc> innerStates = new HashMap<>();
    
    private Map<String, StateBundleDesc> componentStates = new HashMap<>();

    protected AgentState() {
        
    }
    
    public AgentState(AgentDesc ad) {
        this.agentDesc = ad;
    }
    
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    public long getId() {
        return id;
    }

    public void setId(long id) {
        this.id = id;
    }

    @ManyToOne(optional = false)
    @JoinColumn(name="agentName")
    public AgentDesc getAgentDesc() {
        return agentDesc;
    }

    public void setAgentDesc(AgentDesc agentDesc) {
        this.agentDesc = agentDesc;
    }

    @ManyToOne(optional = false)
    public BaseState getBaseState() {
        return baseState;
    }

    public void setBaseState(BaseState baseState) {
        this.baseState = baseState;
    }

    @ManyToMany(fetch = FetchType.EAGER)
    @MapKey(name = "enumClassName")
    public Map<String, InnerStateDesc> getInnerStates() {
        return innerStates;
    }

    public void setInnerStates(Map<String, InnerStateDesc> innerStates) {
        this.innerStates = innerStates;
    }

    @ManyToMany(fetch = FetchType.EAGER)
    public Map<String, StateBundleDesc> getComponentStates() {
        return componentStates;
    }

    public void setComponentStates(Map<String, StateBundleDesc> componentStates) {
        this.componentStates = componentStates;
    }
    
    /**
     * Returns a map of state values map to their class name.
     * @param components
     * @return 
     */
    public Map<String, Map<String, String>> asFlatStatesMap(List<String> components) {
        Map<String, Map<String, String>> res = new HashMap<>();
        if(components == null || components.isEmpty() || components.contains("")) {
            Map<String, String> subStates = new HashMap<>();
            subStates.put("org.lsst.ccs.bus.states.AlertState", baseState.getAlertState());
            subStates.put("org.lsst.ccs.bus.states.PhaseState", baseState.getPhaseState());
            subStates.put("org.lsst.ccs.bus.states.CommandState", baseState.getCommandState());
            subStates.put("org.lsst.ccs.bus.states.OperationalState", baseState.getOperationalState());
            subStates.put("org.lsst.ccs.bus.states.ConfigState", baseState.getConfigState());
            if(!innerStates.isEmpty()) {
                for (InnerStateDesc isd : innerStates.values()) {
                    subStates.put(isd.getEnumClassName(), isd.getEnumValue());
                }
            }
            res.put("", subStates);
            if(components != null && components.size() == 1) {
                return res;
            }
            for(Map.Entry<String, StateBundleDesc> entry : componentStates.entrySet()) {
                Map<String, String> flatMap = res.get(entry.getKey());
                if(flatMap == null) {
                    flatMap = new HashMap<>();
                    res.put(entry.getKey(), flatMap);
                }
                for(InnerStateDesc isd : entry.getValue().getComponentStates().values()) {
                    flatMap.put(isd.getEnumClassName(), isd.getEnumValue());
                }
            }
            return res;
        } else {
            for(String component : components) {
                StateBundleDesc sbd = componentStates.get(component);
                if(sbd != null) {
                    Map<String, String> flatMap = new HashMap<>();
                    for(InnerStateDesc isd : sbd.getComponentStates().values()) {
                        flatMap.put(isd.getEnumClassName(), isd.getEnumValue());
                    }
                    res.put(component, flatMap);
                }
            }
            return res;
        }
    }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(agentDesc.getAgentName());
        sb.append(":").append(baseState.toString()).append(",");
        for (InnerStateDesc isd : innerStates.values()) {
            sb.append(isd.toString()).append(",");
        }
        return sb.toString();
    }
    
}
