package org.lsst.ccs.localdb.configdb.model;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;

/**
 * Utility class to compute checksum of {@code Configuration} entities.
 * @author LSST CCS Team
 */
public class ConfigurationUtils {

    private ConfigurationUtils() {}
    
    /**
     * The computed checksum of a {@code Configuration} entity only takes in
     * account the name and value of each parameter that is part of that
     * configuration.
     *
     * @param c the configuration
     * @return
     * @throws NoSuchAlgorithmException
     * @throws IOException
     */
    public static long computeChecksum(Configuration c) throws NoSuchAlgorithmException, IOException {
          Map<String, String> orderedMap = new TreeMap<>();
        for (Map.Entry<ParameterPath,ConfigurationParameterValue> e : c.getConfigurationParameterValues().entrySet()) {
            orderedMap.put(e.getKey().toString(), e.getValue().getValue());
        }
        return doComputeChecksum(c.getDescription().getId(), orderedMap);
    }
    
    public static long computeCheckSum(long descriptionId, List<ConfigurationParameterInfo> cpis) throws NoSuchAlgorithmException, IOException {
        Map<String, String> orderedMap = new TreeMap<>();
        for (ConfigurationParameterInfo cpi: cpis) {
            if (!cpi.isFinal()) {
                orderedMap.put(new ParameterPath(cpi.getComponentName(), cpi.getParameterName()).toString(), cpi.getConfiguredValue());
            }
        }
        return doComputeChecksum(descriptionId, orderedMap);
    }
    
    private static long doComputeChecksum(long descId, Map<String, String> parms) throws NoSuchAlgorithmException, IOException {
        MessageDigest md = MessageDigest.getInstance("SHA");
        md.update(ByteBuffer.allocate(Long.SIZE).putLong(descId));
      
        ByteArrayOutputStream bas = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(bas);
        oos.writeObject(parms);
        md.update(bas.toByteArray());
        bas.close();
        oos.close();
        return new BigInteger(md.digest()).longValue();
        
    }
}
