package org.lsst.ccs.localdb.configdb.model;

import java.io.Serializable;
import java.util.*;
import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.Index;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.MapKeyJoinColumn;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.persistence.UniqueConstraint;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Represents a named configuration for a given category.
 * @author LSST CCS Team
 */
@Table(uniqueConstraints = {
    @UniqueConstraint(columnNames = {"descriptionId", "category", "configName", "version"})},
            indexes = {@Index(columnList = "checkSum")})
@Entity
public class Configuration implements Serializable {

    private static final long serialVersionUID = 672642551238249224L;
    
    private long id;    

    private Description description;

    /** Name of the category for this configuration. */
    private String category;
    
    /** The configuration name for this category. */
    private String configNameForCategory;
    
    private Map<ParameterPath, ConfigurationParameterValue> parameterConfigurations = new HashMap<>();
    
    /** The time this configuration was saved. */
    private long tsaved;
    
    private int version;
    
    /** Indicates whether this named configuration is the default version. */
    private boolean defaultVersion;
    
    /** Indicates whether this named configuration is the latest version. */
    private boolean latestVersion = true;
    
    private Long checkSum; 
    
    ///////////////////////////// CONSTRUCTORS

    protected Configuration() {
    }
    
    public Configuration(Description subsystemDescription, String category, String configName, long date) {
        this.description = subsystemDescription;
        this.category = category;
        this.configNameForCategory = configName;
        this.tsaved = date;
    }
    
    public Configuration(Description subsystemDescription, String category, String configName, long date, int version) {
        this(subsystemDescription, category, configName, date);
        this.version = version;
    }

    /**
     * creates a copy of an active Configuration (for instance to start an engineering mode or to validate
     * a new tagged configuration from an engineering mode configuration).
     *
     * @param other
     * @param newName
     * @param date date of creation
     * @param version version number
     */
    public Configuration(Configuration other, String newName, long date, int version) {
        this(other.getDescription(), other.getCategory(), 
                newName == null ? other.getConfigName() : newName, date, version);
        parameterConfigurations.putAll(other.getConfigurationParameterValues());
    }
    
    /////////////////////////////////// ACCESSORS/MUTATORS
    @Id
    @GeneratedValue
    public long getId() {
        return id;
    }
    
    protected void setId(long id) {
        this.id = id;
    }

    @ManyToOne (fetch=FetchType.EAGER)
    public Description getDescription() {
        return description;
    }
    
    public void setDescription(Description subsystemDesc) {
        this.description = subsystemDesc;
    }

    /**
     * @return an unmodifiable set
     */
    @ManyToMany(cascade = CascadeType.ALL, fetch=FetchType.EAGER)
    @MapKeyJoinColumn(name = "configurationParameter_parameterPath")
    public Map<ParameterPath, ConfigurationParameterValue> getConfigurationParameterValues() {
        return parameterConfigurations;
    }
    
    void setConfigurationParameterValues(Map<ParameterPath, ConfigurationParameterValue> parameterConfigurations) {
        this.parameterConfigurations = parameterConfigurations;
    }
    
    public String getCategory(){
        return category;
    }
    
    public void setCategory(String category) {
        this.category = category;
    }
    
    public String getConfigName(){
        return configNameForCategory;
    }
    
    public void setConfigName(String configName) {
        this.configNameForCategory = configName;
    }

    public long getTsaved() {
        return tsaved;
    }

    public void setTsaved(long tsaved) {
        this.tsaved = tsaved;
    }

    public int getVersion() {
        return version;
    }

    public void setVersion(int version) {
        this.version = version;
    }
    
    public boolean isDefaultVersion() {
        return defaultVersion;
    }

    public void setDefaultVersion(boolean defaultVersion) {
        this.defaultVersion = defaultVersion;
    }

    public boolean isLatestVersion() {
        return latestVersion;
    }

    public void setLatestVersion(boolean latestVersion) {
        this.latestVersion = latestVersion;
    }
    
    public void setCheckSum(Long sha) {
        this.checkSum = sha;
    }
    
    public Long getCheckSum() {
        return checkSum;
    }
    
    @Transient
    public String getConfigurationDescriptionString() {
        return new StringBuilder(category).append(":")
                .append(configNameForCategory)
                .append("(").append(version).append(")")
                .toString();
    }

    /**
     * registers a list of parameter configurations
     * @param parameterConfiguration
     * @throws IllegalArgumentException if levels are incompatible
     * @throws IllegalArgumentException if the corresponding parameter descriptions are not alive in the current
     * subsystem description
     */
    public void addConfigurationParameterValue(ConfigurationParameterValue parameterConfiguration) {
        ConfigurationParameter cp = parameterConfiguration.getConfigurationParameter();
        if (cp.getCategory().equals(getCategory())){
                // check for deprecation
            this.parameterConfigurations.put(parameterConfiguration.getPath(), parameterConfiguration);
        } else {
            Logger.getLogger("org.lsst.ccs.config").warn("The property " 
                    + cp.getPath().getParameterName() 
                    + " does not belong to the category specified by the property file name : " + getCategory());
        }
    }
    
    public  String getValue( ParameterPath path) {
        ConfigurationParameterValue cpv = parameterConfigurations.get(path);
        if (cpv == null) {
            return null;
        }
        return cpv.getValue();
    }
    
    @Override
    public String toString() {
        return "\""+category+"\":\""+configNameForCategory+"\"("+version+")";
    }
    
}
