package org.lsst.ccs.localdb.configdb.model;

import java.util.ArrayList;
import java.util.HashMap;
import org.hibernate.Session;
import org.hibernate.SessionFactory;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Restrictions;
import org.hibernate.resource.transaction.spi.TransactionStatus;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.localdb.statusdb.model.AgentDesc;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Low level Data Access to Configuration Objects. An hibernate connection is
 * started using {@code begin} and ended with {@code end}. Between those calls
 * the hibernate session is persistent. An hibernate session is confined to the
 * thread that opens it, by the use of a {@code ThreadLocal}
 *
 * @author LSST CCS Team
 */
public class HibernateDAO {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.configdb");

    private final SessionFactory sessionFactory;
    private org.hibernate.Transaction currentTransaction ;
    private final ThreadLocal<Session> session;
    
    public HibernateDAO(SessionFactory fac) {
        sessionFactory = fac;
        
        session = new ThreadLocal<Session>() {
            public Session initialValue() {
                return sessionFactory.openSession();
            }
        };
    }
    
    public void begin() {
        log.debug("begin transaction");
        currentTransaction = getSession().beginTransaction() ;
    }

    /**
     * 
     * @throws Exception if the persistence layer fails at committing 
     */
    public void end() throws Exception {
        log.debug("end transaction");
        try {
            currentTransaction.commit() ;
        } catch (Exception ex) {
            log.error("caught exception when persisting : " + ex, ex);
            try {
                if (currentTransaction.getStatus() == TransactionStatus.ACTIVE
                        || currentTransaction.getStatus() == TransactionStatus.MARKED_ROLLBACK) {
                    currentTransaction.rollback();
                }
            } catch (Exception rbEx) {
                log.error("Rollback of transaction failed : " + rbEx, rbEx);
                throw rbEx;
            }
            throw ex;
        } finally {
            if (getSession().isOpen()) {
                getSession().close();
            }
            currentTransaction = null ;
            session.remove();
        }
    }
    
    public void close() {
        getSession().close();
        sessionFactory.close();
    }
    
    public void persist(Object obj) {
        getSession().persist(obj);
    }
    
    private Session getSession() {
        return session.get();
    }

    public Description getDescription(long id) {
        return getSession().get(Description.class, id);
    }
    
    public Configuration getConfigurationOrNull(Description desc, String name, String cat, int version) {
        Configuration res = null;
        Criterion baseCriterion = Restrictions.conjunction(
                Restrictions.eq("description", desc),
                Restrictions.eq("configName", name),
                Restrictions.eq("category", cat));
        
        switch(version) {
            case ConfigurationInfo.DEFAULT_VERSION:
                // getting the default configuration
                res = (Configuration)getSession().createCriteria(Configuration.class)
                        .add(baseCriterion)
                        .add(Restrictions.eq("defaultVersion", true))
                        .uniqueResult();
                if (res == null) {
                    log.fine("no default version for configuration " + desc.getAgentDesc()+":"+cat+":"+name+". Getting the latest.");
                } else {
                    return res;
                }
            
            case ConfigurationInfo.LATEST_VERSION:
                return getLatestConfiguration(desc, cat, name);
            default:
                return (Configuration)getSession().createCriteria(Configuration.class)
                        .add(baseCriterion)
                        .add(Restrictions.eq("version", version))
                        .uniqueResult();
        }
    }
    
    private Configuration getConfigurationOrNull(Description desc, String name, String cat, Long checkSum) {
        Configuration res = (Configuration)getSession().createQuery("from Configuration where description=:description and configName =:tag and category=:cat and checkSum =:checkSum")
                .setEntity("description", desc).setString("tag", name).setString("cat", cat).setLong("checkSum", checkSum)
                .uniqueResult();
        if (res != null) {
            log.fine("found matching configuration " + res.getConfigurationDescriptionString());
        }
        return res;
    }
    
    /**
     * Cascading process to determine which configuration is running. If the version
     * number is well defined, it is fetched by version number, otherwise its
     * checksum is computed. If not found, a new version for the given tag is
     * created.
     *
     * @param desc
     * @param name
     * @param cat
     * @param version
     * @param cpis
     * @param date
     * @param previousRun
     * @return
     */
    public Configuration getConfigurationOrCreate(Description desc, String name, String cat, int version, List<ConfigurationParameterInfo> cpis, long date, ConfigurationRun previousRun) {
        String request = desc.getId() + ": \""+cat+"\":\""+name+"\"("+version+")";
        log.debug("looking for configuration " + request);
        Configuration res = null;
        if (version != ConfigurationInfo.UNDEF_VERSION) {
            // There should be a version in db
            res = getConfigurationOrNull(desc, name, cat, version);
            if (res == null) {
                log.warn("a versioned configuration was not found : " + request + ". creating a new one");
            }
        }
        Long cs = null;
        if (res == null) {
            // No version number : looking by checksum
            try {
                cs = ConfigurationUtils.computeCheckSum(desc.getId(), cpis);
                res = getConfigurationOrNull(desc, name, cat, cs);
            } catch(Exception ex) {
                log.error("cannot compute configuration checksum", ex);
                return null;
            }
        }
        if (res == null) {
            res = createNextConfiguration(desc, name, cat, date, cs);
            
            if(previousRun == null) {
                for(ConfigurationParameterInfo cpi : cpis) {
                    ConfigurationParameter cp = desc.getConfigurationParameters().get(ParameterPath.valueOf(cpi.getPathName()));
                    res.addConfigurationParameterValue(
                            new ConfigurationParameterValue(cp, cpi.getConfiguredValue(), date));
                }
            } else {
                for(ConfigurationParameterInfo cpi : cpis) {
                    ParameterPath path = ParameterPath.valueOf(cpi.getPathName());
                    ConfigurationParameter cp = desc.getConfigurationParameters().get(path);
                    ConfigurationParameterValue previousValue = previousRun.getConfigurationParameterValueForRun(path);
                    if(cpi.getConfiguredValue().equals(previousValue.getValue())) {
                        res.addConfigurationParameterValue(previousValue);
                    } else {
                        log.warn("parameter : " + cp.getParameterPath() + " had the following value in the previous run : " + previousValue.getValue() + " but is now : " + cpi.getCurrentValue());
                        res.addConfigurationParameterValue(new ConfigurationParameterValue(previousValue.getConfigurationParameter(), cpi.getConfiguredValue(), date));
                    }
                }
            }
        }
        return res;
    }
    
    public ConfigurationRun populateRunFromScratch(ConfigurationRun run, List<ConfigurationParameterInfo> changes) {
        for (ConfigurationParameterInfo cpi : changes) {
            ConfigurationParameterValue dirtyVal = createConfigurationParameterValue(run.getConfiguration().getDescription(), cpi.getPathName(), cpi.getCurrentValue(),run.getTstart());
            run.addRuntimeChange(dirtyVal);
        }
        return persistNextConfigurationRun(run,null);
    }
    
    public ConfigurationRun populateRunFromPreviousRun(ConfigurationRun previousRun, ConfigurationRun newRun, List<ConfigurationParameterInfo> changes) {
        Configuration prevConfig = previousRun.getConfiguration();
        Configuration newConfig = newRun.getConfiguration();
        if(changes.isEmpty()) {
            GlobalConfiguration newGC = newRun.getGlobalConfiguration();
            GlobalConfiguration prevGC = previousRun.getGlobalConfiguration();
            // newRun is intended to be configured
            if ((newGC != null && prevGC != null && newGC.getId() == previousRun.getId()) || (newGC == null && prevGC == null)) {
                if (!previousRun.isDirty() && prevConfig.getConfigName().equals(newConfig.getConfigName()) && prevConfig.getVersion() == newConfig.getVersion()) {
                    return previousRun;
                }
            }
            return persistNextConfigurationRun(newRun, previousRun);
        } else {
            if (!prevConfig.getConfigName().equals(newConfig.getConfigName())) {
                log.warning("inconsistency detected");
                previousRun.setTstop(newRun.getTstart());
                return populateRunFromScratch(newRun, changes);
            } else {
                // Figuring out the delta of changes compared with the latest run
                // TO-DO : this could also be done using ci.getLatestChanges()
                List<ConfigurationParameterValue> delta = new ArrayList<>();
                for (ConfigurationParameterInfo cpi : changes) {
                    ParameterPath path = ParameterPath.valueOf(cpi.getPathName());
                    String currentValue = cpi.getCurrentValue();
                    ConfigurationParameterValue previousChange = previousRun.getRuntimeChanges().get(path);
                    if (previousChange == null || !previousChange.getValue().equals(currentValue)) {
                        // This is a new change
                        ConfigurationParameterValue cpv = new ConfigurationParameterValue(newRun.getConfiguration().getDescription().getConfigurationParameters().get(path), currentValue, newRun.getTstart());
                        delta.add(cpv);
                    }
                }
                if (delta.isEmpty()) {
                    return previousRun;
                } else {
                    for (ConfigurationParameterValue cpv : delta) {
                        newRun.addRuntimeChange(cpv);
                    }
                    return persistNextConfigurationRun(newRun, previousRun);
                }
            }
        }
    }
    
    private Configuration createNextConfiguration(Description desc, String name, String cat, long date, Long cs) {
        Configuration latestConfiguration = getLatestConfiguration(desc, cat, name);
        int latestVersion = -1;
        if (latestConfiguration != null) {
            latestVersion = latestConfiguration.getVersion();
            latestConfiguration.setLatestVersion(false);
        }
        int nextVersion = latestVersion+1;
        Configuration res = new Configuration(desc, cat, name, date, nextVersion);
        log.fine("created new configuration : " + res);
        res.setCheckSum(cs);
        getSession().persist(res);
        return res;
    }
    
    private ConfigurationRun persistNextConfigurationRun(ConfigurationRun nextRun, ConfigurationRun previousRun) {
        if(previousRun != null) {
            previousRun.setTstop(nextRun.getTstart());
        }
        getSession().persist(nextRun);
        return nextRun;
    }
    
    private Configuration getLatestConfiguration(Description desc, String cat, String name) {
        Configuration latestConfiguration = (Configuration)getSession().createQuery(
                "from Configuration c where c.description=:desc and c.configName=:name and c.category=:cat and c.version="
                        + "(select max(co.version) from Configuration co where co.description=:desc and co.configName=:name and co.category=:cat)")
                .setEntity("desc", desc).setString("name", name).setString("cat", cat).uniqueResult();
        if(latestConfiguration == null) {
            log.debug("no configuration for \"" + cat+"\":\""+name+ "\" for description " + desc.getId());
        }
        return latestConfiguration;
    }
    
    private ConfigurationParameterValue createConfigurationParameterValue(Description desc, String path, String value, long time) {
        ConfigurationParameterValue cpv = new ConfigurationParameterValue(desc.fetch(path), value, time);
        getSession().persist(cpv);
        return cpv;
    }
    
    /**
     * If Configuration runs are found with no tstop, they are considered as
     * suspicious : they are removed.
     *
     * @param desc
     */
    public void cleanupConfigurationRuns(Description desc, long time) {
        Map<String, ConfigurationRun> activeRuns = getActiveConfigurationRuns(desc);
            // Those runs should have had their tstop set
            for(ConfigurationRun cr : activeRuns.values()) {
                log.info("suspicious run found for " + desc.getAgentDesc().getAgentName()+ ", starting at " + cr.getTstart() + " with no end date.");
                cr.setTstop(time);
                getSession().persist(cr);
            }
    }
    
    public Configuration getLatestRunningConfiguration(Description desc, String category) {
        return getConfigRunningAt(desc, category, PackCst.STILL_VALID);
    }
    
    public Configuration getConfigRunningAt(Description desc, String category, long date) {
        ConfigurationRun run = getConfigurationRunAt(desc, category, date);
        if (run == null) {
            return null;
        } 
        return run.getConfiguration();
    }
    
    public String getActiveValueAt(Description desc, String category, String parameterPath, long date) {
        ConfigurationRun run = getConfigurationRunAt(desc, category, date);
        if (run == null) {
            return null;
        }
        return run.getConfigurationParameterValueForRun(ParameterPath.valueOf(parameterPath)).getValue();
    }

    public void endRun(Description description, long endTime) {
        Map<String,ConfigurationRun> currentRuns = getActiveConfigurationRuns(description);
        for(ConfigurationRun cr : currentRuns.values()) {
            cr.setTstop(endTime);
            getSession().update(cr);
        }
    }
    
    private ConfigurationParameter getConfigurationParameter(String agentName, ConfigurationParameterInfo cpi) {
        ConfigurationParameter cp = (ConfigurationParameter)getSession().createQuery("from ConfigurationParameter where agentName=:agentName and componentName=:componentName and parameterName=:parameterName and typeName=:typeName and category=:category")
                .setString("agentName", agentName).setString("componentName",cpi.getComponentName())
                .setString("parameterName", cpi.getParameterName()).setString("typeName", cpi.getType())
                .setString("category",cpi.getCategoryName()).uniqueResult();
        if (cp == null) {
            log.fine("creating a new configuration parameter entity for " + agentName + " : " + cpi.getComponentName()+"/"+cpi.getParameterName());
            cp = new ConfigurationParameter(agentName,
                    new ParameterPath(cpi.getComponentName(), cpi.getParameterName()), cpi.getType(), cpi.getDescription(), "", cpi.getCategoryName(), 0);
            getSession().persist(cp);
        }
        return cp;
    }
    
    /**
     * Finds or creates a {@code Description} entity corresponding to the input arguments.
     * @param ad the agent description
     * @param cpis
     * @return a {@code Description} object in persisted state.
     */
    public Description findDescriptionOrCreate(AgentDesc ad, List<ConfigurationParameterInfo> cpis) {
        getSession().update(ad);
        Description desc = findDescriptionOrNull(ad.getAgentName(), cpis);
        if (desc == null) {
            // A new description has to be created
            log.fine("no matching description found for " + ad.getAgentName() + " : creating a new one");
            desc = new Description(ad);
            Map<ParameterPath, ConfigurationParameter> cpMap = new HashMap<>();
            for (ConfigurationParameterInfo cpi : cpis) {
                ConfigurationParameter cp = getConfigurationParameter(ad.getAgentName(), cpi);
                cpMap.put(cp.getParameterPath(), cp);
            }
            desc.setConfigurationParameters(cpMap);
            getSession().persist(desc);
        }
        return desc;
    }
    
    public Description findDescriptionOrNull(String agentName, List<ConfigurationParameterInfo> cpis) {
        // Remove final configuration parameters from the list
        List<ConfigurationParameterInfo> cpisNonFinal = cpis.stream()
                .filter(cp -> !cp.isFinal()).collect(Collectors.toList());
        
        List<Description> list = getSession().createQuery("from Description where agentName=:agentName")
                .setString("agentName", agentName).list();
        for (Description desc : list) {
            Map<ParameterPath, ConfigurationParameter> map = desc.getConfigurationParameters();
            if (map.size() != cpis.size()) continue;
            boolean matching = true;
            for(ConfigurationParameterInfo cpi : cpisNonFinal) {
                ConfigurationParameter cp = map.get(ParameterPath.valueOf(cpi.getPathName()));
                if (cp == null || !cp.getCategory().equals(cpi.getCategoryName()) || !cp.getTypeName().equals(cpi.getType())) {
                    matching = false;
                    break;
                } 
            }
            // In the future we can do more stuff here
            if (matching) {
                log.fine("found matching description for " + agentName + " : " + desc.getId());
                return desc;
            }
        }
        return null;
    }
    
    public ConfigurationRun getActiveConfigurationRun(Description desc, String category) {
        ConfigurationRun cr = (ConfigurationRun)getSession().createQuery("from ConfigurationRun where configuration.description=:desc and configuration.category=:cat and tstop=:tstop")
                .setEntity("desc", desc).setLong("tstop", PackCst.STILL_VALID)
                .setString("cat", category).uniqueResult();
        return cr;
    }
    

    public ConfigurationInfoData getConfigurationInfoData(Description desc, long date) {
        ConfigurationInfoData res = (ConfigurationInfoData)getSession().createQuery("from ConfigurationInfoData cid where cid.description=:desc and cid.time <=:date order by cid.time desc")
                .setEntity("desc", desc).setLong("date", date).setMaxResults(1).uniqueResult();
        if (res == null) {
            log.info("no ConfigurationInfoData found for date " + date);
        }
        return res;
    }
    
    public ConfigurationRun getConfigurationRunAt(Description desc, String category, long date) {        
        ConfigurationInfoData cid = getConfigurationInfoData(desc, date);
        ConfigurationRun run = null;
        if (cid != null ) {
            run = cid.getConfigurationRuns().get(category);
            if (run.getTstop() <= date) {
                log.warn("for date : " + date + ", CID was found but its run ends at " + run.getTstop());
                run = null;
            }
        }
        if (run == null) {
            log.warning( "no configuration run at " + date + " for subsystem " + desc.getAgentDesc().getAgentName() + " and category " + category);
        }
        return run;
    }
    
    public Map<String, ConfigurationRun> getActiveConfigurationRuns(Description desc) {
        List<ConfigurationRun> list = getSession().createQuery("from ConfigurationRun where configuration.description=:description and tstop=:tstop")
                .setEntity("description", desc)
                .setLong("tstop", PackCst.STILL_VALID).list();
        return list.stream().collect(Collectors.toMap((cr)->cr.getConfiguration().getCategory(), (cr)->cr));
    }
    
    public List<String> findAvailableConfigurationsForCategory(Description desc, String category) {
        return getSession().createQuery("select configName from Configuration where description=:desc and category=:cat")
                .setEntity("desc", desc).setString("cat",category).list();
    }
    
    public GlobalConfiguration findGlobalConfiguration(Description desc, String name, int version) {
        if( version < 0) {
            return (GlobalConfiguration)getSession().createQuery("from GlobalConfiguration where description=:desc and name=:name and latest=true")
                    .setEntity("desc", desc).setString("name", name).uniqueResult();
        } else {
            return (GlobalConfiguration)getSession().createQuery("from GlobalConfiguration where description=:desc and name=:name and version=:version")
                    .setEntity("desc", desc).setString("name", name).setInteger("version", version).uniqueResult();
        }
    }

    public GlobalConfiguration findGlobalConfigurationOrCreate(Description desc, String name, Map<String, Configuration> newConfigs) {
        List<GlobalConfiguration> list = getSession().createQuery("from GlobalConfiguration where description=:desc and name=:name")
                .setEntity("desc", desc).setString("name", name)
                .list();
        int nver = 0;
        GlobalConfiguration latestGC = null;
        if (!list.isEmpty()) {
            nver = list.size();
            for (GlobalConfiguration gc : list) {
                if (gc.isLatest()) {
                    latestGC = gc;
                }
                boolean equal = true;
                for(Configuration c : gc.getConfigurations().values()) {
                    if(newConfigs.get(c.getCategory()).getId() != c.getId()) {
                        equal = false;
                        break;
                    }
                }
                if (equal) {
                    return gc;
                }
            }
        } 
        // Creating it
        if(latestGC != null) {
            latestGC.setLatest(false);
        }
        GlobalConfiguration gc = new GlobalConfiguration(desc, name, nver);
        gc.setConfigurations(newConfigs);
        gc.setLatest(true);
        getSession().persist(gc);
        return gc;
                
    }
}