package org.lsst.ccs.localdb.configdb.model;

import java.io.Serializable;
import java.util.Collection;

/**
 * A set of static methods to be used on Configuration service client's side : they are hiding some implementation details to the "outside"
 * world.
 * To be used in conjunction with <TT>ConfigurationFacade</TT> (but
 * these codes may be called locally without referring to a server)
 * @author bamade
 */
// Date: 05/06/12

public class Factories {
    
    private Factories() {
    }
    
    ////////// creation of "raw" subsystemDescriptions
    //TODO: normally the creation of all descriptions should be made from the text file
    // so no more public factories for these ?

    /**
     * Creates a new {@code SubsystemDescription} object without any {@code ParameterDescription} list creation. 
     * The object is not registered in the database since it is to be populated by <TT>ParameterDescription</TT> objects.
     * <p/>
     * To populate the descriptions get the base descriptions from the Subsystemdescription by calling a version
     * of <TT>getBaseParameters</TT> or <TT>getParameterDescriptions</TT> and generate the descriptions from these bases.
     *
     * @param subsystemName
     * @param tag
     * @param configurationData
     * @return a {@code SubsystemDescription} object without parameter descriptions
     */
    public static SubsystemDescription createRawSubsystemDescription(String subsystemName, String tag,
                                                              Serializable configurationData) {
        return new ASubsystemDescription(subsystemName, tag, configurationData);
    }

    /**
     * Creates a new {@code SubsystemDescription} object and populates it with the
     * given list of parameter descriptions.
     * @param subsystemName
     * @param tag
     * @param configurationData
     * @param parameterDescriptions
     * @return 
     */
    public static SubsystemDescription createSubsystemDescription(String subsystemName, String tag,
            Serializable configurationData, Collection<ParameterDescription> parameterDescriptions) {
        SubsystemDescription res = createRawSubsystemDescription(subsystemName, tag, configurationData);
        res.addParameterDescriptions(parameterDescriptions);
        return res;
    }

    /**
     * creates a new SubsystemDescription from another. The other one could be active of deprecated data:
     * this method is meant to create another subsystemdescription were only the ParameterDescription are to be changed.
     * not that the list of ParameterDescription may be populated (if the model have some) and then that this set
     * is to be modified (the resulting object is not yet persisted in the database).
     *
     * @param desc
     * @return a deep copy of a the object.
     */
    public static SubsystemDescription createSubsystemDescriptionCopy(SubsystemDescription desc) {
        return new ASubsystemDescription(desc);
    }


    /**
     * factory method to create a new ParameterDescription. To be used by tools such as GUI
     *
     * @param parameterBase
     * @param description
     * @param simpleName
     * @param category
     * @param level
     * @return a {@code ParameterDescription} object built out of the issued
     * arguments
     */
    public static ParameterDescription createParameterDescription(ParameterBase parameterBase, String description, String simpleName, String category, int level) {
        return new AParameterDescription(parameterBase, description, simpleName,category, level);
    }

    /**
     * Factory method to create a new ParameterDescription from another one.
     *
     * @param other
     * @return a copy of other.
     */
    public static ParameterDescription createParameterDescription(ParameterDescription other) {
        return new AParameterDescription(other);
    }

    /**
     * tries to copy <TT>ParameterDescription</TT> from a model to a new subsystem description.
     * each <TT>ParameterDescription</TT> is checked against the configData (if not compatible it is rejected
     * and a Listener is notified).
     * <p/>
     * NOT IMPLEMENTED YET
     * </P>
     *
     * @param newDescription   a subsystem description <B>without</B> any <TT>ParameterDescription</TT>
     * @param model
     * @param mismatchListener optional code that will be warned when inconsistencies occur
     */
    public static void tryCopyParameters(SubsystemDescription newDescription, SubsystemDescription model,
                                  DescriptionMismatchListener mismatchListener) {
        // generate a bogus ParameterBase list for newDescription
        // test if type is the same ?(default value may be changed)
        // new defaultValue tested against "constraints"
        // level set to old level
        System.err.println("Try copy Parameters not implemented yet");
    }


    ////////////// creation of "raw" configProfiles

    /**
     * Creates an empty ConfigProfile.
     * This object is associated to a subsystem description, is specified the 
     * category of parameters it referes to and is associated a configuration name
     * but it does not contain any {@code ParameterConfiguration} list.
     *
     * @param subsystemDesc should  be read from the database
     * @param category
     * @param configName
     * @return an empty {@code ConfigProfile}
     * @throws IllegalArgumentException if subsystemdescription not in database
     */
    public static ConfigProfile createRawConfigProfile(SubsystemDescription subsystemDesc, String category
            , String configName) {
        if (!(subsystemDesc instanceof ASubsystemDescription)) {
            throw new IllegalArgumentException("deprecated Description");
        }
        return createRawConfigProfile((ASubsystemDescription) subsystemDesc, category, configName);
    }

    static AConfigProfile createRawConfigProfile(ASubsystemDescription subsystemDesc, String categoryName
            , String configName) {
        //TODO: check for preconditions
        return new AConfigProfile(subsystemDesc, categoryName, configName);
    }

    /**
     * to be used to create a copy of configuration with different tag, etc.
     * This is used mostly to go back and forth in Engineering mode: for instance start
     * an Engineering mode with modification of parameters "on the fly"
     * <p/>
     * when this is used to create an "engineering mode" profile it is mandatory that the profile
     * to be copied is an active one (not a deprecated one)
     *
     * @param toBeCopied    for the moment should be an "active" ConfigProfile (not a deprecated one)
     * @param newName
     * @return a copy of the specified configuration profile with different name,
     * user name and level.
     */
    public static ConfigProfile copyProfile(ConfigProfile toBeCopied, String newName) {
        if (toBeCopied instanceof AConfigProfile) {
            return new AConfigProfile((AConfigProfile)toBeCopied, newName);
        }
        throw new UnsupportedOperationException("copy with deprecated profile not yet implemented");
    }

    /**
     * Factory method to create a ParameterConfiguration object.
     * The object stands for the {@code description} parameter description and
     * attempts to assign a configured value to it.
     * This factory method is to be used statically exclusively (ie not when a 
     * subsystem is running).
     *
     * @param description
     * @param value
     * @return a new parameter configuration the value of which is {@code value}
     */
    public static ParameterConfiguration createParameterConfiguration(ParameterDescription description, String value) {
        if (!(description instanceof AParameterDescription)) {
            throw new IllegalArgumentException("deprecated description");
        }
        if(description.getId() == 0L) {
            throw new IllegalArgumentException("ParameterDescription not registered in base" + description) ;
        }
        return new AParameterConfiguration((AParameterDescription) description, value);
    }

    /**
     * Factory method to create a ParameterConfiguration object.
     *
     * @param description
     * @return a parameter configuration object, the value of which is set to 
     * the default value as defined in the parameter description
     */
    public static ParameterConfiguration createParameterConfiguration(ParameterDescription description) {
        if (!(description instanceof AParameterDescription)) {
            throw new IllegalArgumentException("deprecated description");
        }
        if(description.getId() == 0L) {
            throw new IllegalArgumentException("ParameterDescription not registered in base" + description) ;
        }
        return new AParameterConfiguration((AParameterDescription) description);
    }
    
}
