package org.lsst.ccs.localdb.dao;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.hibernate.Query;
import org.hibernate.Session;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.localdb.statusdb.model.AgentDesc;
import org.lsst.ccs.localdb.statusdb.model.AlertDesc;
import org.lsst.ccs.localdb.statusdb.model.StatTimeInterval;
import org.lsst.ccs.localdb.statusdb.model.AgentState;
import org.lsst.ccs.localdb.statusdb.model.BaseState;
import org.lsst.ccs.localdb.statusdb.model.InnerStateDesc;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Simplistic access to the database, through static methods.
 * @author LSST CCS Team
 */
public class LocaldbFacade {
    
    private LocaldbFacade() {}

    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb");

    /**
     * Returns the persistent AgentDesc corresponding to the AgentInfo.
     * If not found in the first place, creates it and persists it.
     * @param agentInfo
     * @param sess
     * @return an AgentDesc in persisted state.
     */
    public static AgentDesc getAgentDesc(AgentInfo agentInfo, Session sess) {
        String key = agentInfo.getName();
        AgentDesc ad = sess.get(AgentDesc.class, key);

        if (ad == null) {
            ad = new AgentDesc(agentInfo);
            sess.persist(ad);
            log.debug("Added Agent description for " + key +" : " + ad.getAgentName());
        }
        return ad;
    }

    /** -- Methods for Alert persistence. */
    
    /**
     *
     * @param agentDesc
     * @param alert
     * @param sess
     * @return an AlertDesc in persisted state.
     */
    public static AlertDesc getAlertDescOrPersist(AgentDesc agentDesc, Alert alert, Session sess) {
        AlertDesc ad = getAlertDesc(agentDesc, alert.getAlertId(), sess);
        if (ad == null) {
            ad = new AlertDesc();
            ad.setAgentDesc(agentDesc);
            ad.setAlertId(alert.getAlertId());
            ad.setAlertDescription(alert.getDescription());
            sess.persist(ad);
            log.debug("Added Alert description for " + agentDesc.getAgentName()+"/"+alert.getAlertId() + " : " + ad.getAlertId());
        }
        return ad;
    }

    /**
     * Looks for a persisted AlertDesc corresponding to the given parameters.
     * Returns null if not found.
     * @param agentDesc
     * @param alertId
     * @param sess
     * @return the corresponding AlertDesc, or null if not found.
     */
    public static AlertDesc getAlertDesc(AgentDesc agentDesc,  String alertId, Session sess) {
        Query q = sess.getNamedQuery("findAlertDesc");
        q.setString("id", alertId).setString("name", agentDesc.getAgentName());
        return (AlertDesc)q.uniqueResult();
    }

    /** -- Methods for State persistence. */
    
    /**
     * Returns a persisted InnerStateDesc corresponding to the given parameters.
     * @param ai
     * @param enumClassName
     * @param enumValue
     * @param sess
     * @return an InnerStateDesc in persisted state.
     */
    public static InnerStateDesc getInnerStateDesc(AgentInfo ai, String enumClassName, String enumValue, Session sess) {
        Query q = sess.getNamedQuery("findInnerStateDesc");
        q.setString("subsystemName", ai.getName()).setString("enumClassName", enumClassName).setString("enumValue", enumValue);
        InnerStateDesc isd = (InnerStateDesc)q.uniqueResult();
        if (isd == null) {
            isd = new InnerStateDesc();
            AgentDesc ad = getAgentDesc(ai, sess);
            isd.setAgentDesc(ad);
            isd.setEnumClassName(enumClassName);
            isd.setEnumValue(enumValue);
            sess.persist(isd);
            log.debug("Added Inner State description description for " + isd.toString());
        }
        return isd;
    }

    /**
     * Returns a persisted BaseState corresponding to the given StateBundle.
     * @param sb
     * @param sess
     * @return a BaseState in persisted state.
     */
    public static BaseState getBaseState(StateBundle sb, Session sess) {
        BaseState bs = new BaseState(sb);
        Query q = sess.getNamedQuery("findBaseState")
                .setString("alertSt", bs.getAlertState())
                .setString("phaseSt", bs.getPhaseState())
                .setString("commandSt", bs.getCommandState())
                .setString("operationalSt", bs.getOperationalState())
                .setString("configurationSt", bs.getConfigState());

        log.debug("sending query : " + q.getQueryString());
        BaseState persistedBS = (BaseState)q.uniqueResult();
        if (persistedBS == null) {
            sess.persist(bs);
            log.debug("Added BaseState description for " + bs.toString());
            return bs;
        } else {
            return persistedBS;
        }
    }

    /**
     * Returns the AgentState corresponding to the given stateBundle.
     * An AgentState is the combination of the 5 framework-defined states and of
     * additional subsystem-developer-defined states.
     * @param ai
     * @param sb
     * @param sess
     * @return an AgentState in persisted state.
     */
    public static AgentState getAgentState(AgentInfo ai, StateBundle sb, Session sess) {
        BaseState bs = getBaseState(sb, sess);
        Map<String, String> internalStates = sb.getInternalStates();
        Map<String, InnerStateDesc> internalStateDescs = new HashMap<>();
        for (Map.Entry<String, String> entry : internalStates.entrySet()) {
            internalStateDescs.put(entry.getKey(), getInnerStateDesc(ai, entry.getKey(), entry.getValue(), sess));
        }

        Query q = sess.getNamedQuery("findAgentState")
                .setString("subsystemName", ai.getName())
                .setLong("baseStateId", bs.getId());
//                .setParameterList("innerStates", internalStateDescs);

        AgentState as = null;

        List l = q.list();
        for (Object o : l) {
            AgentState a = (AgentState)o;
            if ( a.getInnerStates().equals(internalStateDescs) ) {
                as = a;
                break;
            }
        }

        if (as == null) {
            as = new AgentState();
            as.setAgentDesc(getAgentDesc(ai, sess));
            as.setBaseState(bs);
            as.setInnerStates(internalStateDescs);
            sess.persist(as);
            log.debug("Added AgentState for " + as.toString());
        }

        return as;
    }

    
    public static StatTimeInterval getStatTimeInterval(long binWidth, long dataTime, Session sess) {
        //Get The time interval for this binWidth
        Query statTimeIntervalQuery = sess.getNamedQuery("findStatTimeInterval");
        statTimeIntervalQuery.setLong("binWidth", binWidth);
        statTimeIntervalQuery.setLong("t1", dataTime-binWidth);
        statTimeIntervalQuery.setLong("t2", dataTime);
        StatTimeInterval statTimeInterval = (StatTimeInterval) statTimeIntervalQuery.uniqueResult();

        if (statTimeInterval == null) {
            long statTimeIntervalStart = (dataTime / binWidth) * binWidth;
            statTimeInterval = new StatTimeInterval(statTimeIntervalStart, binWidth);
            sess.persist(statTimeInterval);
        }
        return statTimeInterval;
    }

}
