package org.lsst.ccs.localdb.statusdb.trendServer;

import com.sun.jersey.spi.resource.Singleton;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpServer;
import java.io.IOException;
import java.math.BigInteger;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;

import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Application;
import javax.ws.rs.ext.RuntimeDelegate;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementWrapper;
import javax.xml.bind.annotation.XmlRootElement;
import org.hibernate.Query;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.DataMetaData;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.model.StatData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.server.ChannelMetaData;
import org.lsst.ccs.localdb.statusdb.server.Data;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.statusdb.server.TrendingData.AxisValue;
import org.lsst.ccs.localdb.statusdb.server.TrendingData.DataValue;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.utilities.logging.Log4JConfiguration;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *
 * The DataServer receives requests for trending data analysis. It delivers the
 * requested data as the result of requests to the database.
 *
 */
@SuppressWarnings("restriction")
@Path("/dataserver")
@Singleton
public class DataServer { // has to be completed

    private static SessionFactory fac;

    static {
        /*
         Temporary : setting of logging
         */
        Log4JConfiguration.initialize();

    }
    static Logger log = Logger.getLogger("org.lsst.ccs.localdb");

    public DataServer() {

        Properties p = new Properties();
        String hcdName = "hibernate.connection.datasource";

        try {
            Context env = new InitialContext();
            try {
                if (env.lookup("jdbc/css") != null) {
                    p.setProperty(hcdName, "jdbc/ccs");
                }
            } catch (NamingException e) {
            }
            try {
                if (env.lookup("java:comp/env/jdbc/ccs") != null) {
                    p.setProperty(hcdName, "java:comp/env/jdbc/ccs");
                }
            } catch (NamingException e) {
            }
//            if ( p.getProperty(hcdName) == null ) {
//                throw new RuntimeException("Unsupported lookup datasource");
//            }
        } catch (NamingException ne) {
        }

        fac = StatusdbUtils.getSessionFactory(p);

        log.fine("Starting Data Server");

    }

    private List<DataChannel> getListOfChannels() {

        log.fine("Loading Channel Information");

        Session sess = fac.openSession();
        @SuppressWarnings("unchecked")
        List<DataDesc> l = sess.createQuery("from DataDesc order by srcName").list();

        ArrayList<DataChannel> channels = new ArrayList<>();
        log.fine("Done with the query " + l.size());
        for (DataDesc d : l) {
            DataChannel dc = new DataChannel(d);
            dc.getMetadata().put("subsystem", d.getSrcSubsystem());
            channels.add(dc);
            log.debug("retrieving Data Channel path= " + dc.getPathAsString());
        }

        sess.close();
        return channels;
    }

    // TODO this is really a first try...
    // TODO check what happens in case of uneven binning in the db, interrupted
    // data, etc.
    // TODO when recomputing stat data either from raw or stat
    // TODO check bounds where recomputing stat
    @GET
    @Path("/data/{id}")
    public Data getData(@PathParam("id") long id,
            @QueryParam("t1")
            @DefaultValue("-1") long t1,
            @QueryParam("t2")
            @DefaultValue("-1") long t2,
            @QueryParam("flavor") String flavor,
            @QueryParam("n")
            @DefaultValue("30") int nbins) {

        // handling of default values
        // t2 default is now
        // t1 default is t2 - 1 hour
        // rawPreferred is false
        // nbins is 30
        boolean useStat = false;
        boolean useRaw = false;
        if ("stat".equals(flavor)) {
            useStat = true;
        } else if ("raw".equals(flavor)) {
            useRaw = true;
        } else {
            flavor = "unspecified";
        }

        log.fine("request for data " + id + "[" + t1 + "," + t2 + "] "
                + flavor + " " + nbins);

        if (t2 < 0) {
            t2 = System.currentTimeMillis();
        }
        if (t1 < 0) {
            t1 = t2 - 3600000L;
        }

        log.fine("request for data " + id + "[" + t1 + "," + t2 + "] "
                + flavor + " " + nbins);

        // what do we have?
        long rawId = id;

        if (useRaw) {
            log.debug("sending raw data");
            return exportRawData(rawId, t1, t2);
        } else if (useStat) {

            StatDesc statSource = null;
            long statSourceN = -1;
            Map<StatDesc, Long> stats = getAvailableStats(rawId, t1, t2);

            log.fine("stats :");
            for (Map.Entry<StatDesc, Long> s : stats.entrySet()) {
                log.fine(s);
                log.fine("  " + s.getKey().getId() + " "
                        + s.getKey().getTimeBinWidth() + " : " + s.getValue());
            }

            // how do we create the data?
            for (Map.Entry<StatDesc, Long> s : stats.entrySet()) {
                long n = s.getValue();
                if (n > nbins / 2) {
                    // this one has enough bins
                    // TODO # of samples is not enough, for instance raw could
                    // have more samples
                    // but covering only the recent part of the time range.
                    if (statSource != null) {
                        if (n < statSourceN) {
                            statSource = s.getKey();
                            statSourceN = n;
                        }
                    } else {
                        statSource = s.getKey();
                        statSourceN = n;
                    }
                }
            }
            if (statSource != null) {

                log.debug("sending stat from stat sampling "
                        + statSource.getTimeBinWidth() + " nsamples "
                        + statSourceN);
                return exportStatDataFromStat(rawId, statSource.getId(), t1, t2, nbins);
            }
        }

        log.debug("sending stat from raw");
        return exportStatDataFromRaw(rawId, t1, t2, nbins);
    }

    public List<ChannelMetaData> getMetadata(int channelId) {
        return getMetadata(channelId, -1, -1);
    }

    public List<ChannelMetaData> getMetadata(int channelId, long t1, long t2) {

        Session sess = fac.openSession();
        String queryStr = "from DataMetaData where rawDescr_id = :id ";
        if (t1 > -1) {
            queryStr += "and (tstopmillis > :t1 or tstopmillis = -1) ";
        }
        if (t2 > -1) {
            queryStr += "and tstartmillis < :t2 ";
        }

        Query q = sess.createQuery(queryStr);
        q.setParameter("id", channelId);
        if (t1 > -1) {
            q.setParameter("t1", t1);
        }
        if (t2 > -1) {
            q.setParameter("t2", t2);
        }

        @SuppressWarnings("unchecked")
        List<DataMetaData> l = q.list();

        sess.close();
        List<ChannelMetaData> out = new ArrayList<ChannelMetaData>();
        for (DataMetaData md : l) {
            out.add(new ChannelMetaData(md));
        }
        return out;

    }

    @XmlRootElement(name = "channelinfo")
    public static class ChannelMetadataList {

        @XmlElementWrapper(name = "channelmetadata")
        @XmlElement(name = "channelmetadatavalue")
        public List<ChannelMetaData> list;

        public ChannelMetadataList(List<ChannelMetaData> list) {
            this.list = list;
        }

        public ChannelMetadataList() {
        }
    }

    @GET
    @Path("/channelinfo/{id}")
    public DataServer.ChannelMetadataList getMetadataList(@PathParam("id") long channelId) {
        long rawId = channelId;
        return new DataServer.ChannelMetadataList(getMetadata((int) rawId));
    }

    /**
     *
     * @return the whole channels list for all CCS.
     */
    @GET
    @Path("/listchannels")
    public DataChannel.DataChannelList getChannels() {
        return new DataChannel.DataChannelList(getListOfChannels());
    }

    /**
     *
     * @param subsystemName
     * @return a channels list for a subsystem
     */
    @GET
    @Path("/listchannels/{subsystem}")
    public DataChannel.DataChannelList getChannels(
            @PathParam("subsystem") String subsystemName) {
        List<DataChannel> channels = getListOfChannels();
        ArrayList<DataChannel> subChannels = new ArrayList<DataChannel>();
        for (DataChannel dc : channels) {
            if (dc.getPath()[0].equals(subsystemName)) {
                subChannels.add(dc);
            }
        }
        return new DataChannel.DataChannelList(subChannels);

    }

    /**
     *
     * @param partialPath
     * @param level
     * @return the list of channels within a partial path and a level
     */
    public DataChannel[] getChannels(String partialPath, int level) {
        // TODO
        return null;
    }

    /**
     * Return all available channels for a given keyword.
     *
     * @param keyword
     * @return channels list
     */
    public DataChannel[] getChannelsByKeywork(String keyword) {
        // TODO
        return null;
        // the keyword can be the name of the published value or a substring of
        // the name.

    }

    protected Map<StatDesc, Long> getAvailableStats(long rawId, long t1, long t2) {
        Session sess = fac.openSession();
        Map<StatDesc, Long> m = new HashMap<StatDesc, Long>();
        Query q = sess
                .createQuery("select d, count(x) from StatDesc d, StatData x where d.rawDescr.id = :id "
                        + "and x.descr = d and x.tstampFirst >= :t1 and x.tstampLast <= :t2 order by d.timeBinWidth");
        q.setParameter("id", rawId);
        q.setParameter("t1", t1);
        q.setParameter("t2", t2);
        @SuppressWarnings("unchecked")
        List<Object[]> l = q.list();
        for (Object[] r : l) {
            if (r[0] == null) {
                continue;// crazy
            }
            m.put((StatDesc) r[0], (Long) r[1]);
        }
        sess.close();
        return m;
    }

    protected long getAvailableRawData(long rawId, long t1, long t2) {
        Session sess = fac.openSession();
        Query q = sess
                .createQuery("select count(r) from RawData r where r.descr.id = :id and r.tstamp between :t1 and :t2 order by r.tstamp");
        q.setParameter("id", rawId);
        q.setParameter("t1", t1);
        q.setParameter("t2", t2);
        long n = (Long) q.uniqueResult();
        sess.close();
        return n;
    }

    protected List<RawData> getRawData(long id, long t1, long t2) {
        Session sess = fac.openSession();

        Query q = sess
                .createQuery("from RawData r where r.descr.id = :id and r.tstamp between :t1 and :t2 order by r.tstamp");
        q.setParameter("id", id);
        q.setParameter("t1", t1);
        q.setParameter("t2", t2);
        @SuppressWarnings("unchecked")
        List<RawData> l = q.list();
        log.debug("retrieved raw data " + id + "[" + t1 + "," + t2 + "] : "
                + l.size());
        sess.close();

        return l;
    }

    protected List<StatData> getStatData(long id, long t1, long t2) {
        Session sess = fac.openSession();

        Query q = sess
                .createQuery("from StatData r where r.descr.id = :id and r.tstampFirst >= :t1 and r.tstampLast <= :t2 order by r.tstampFirst");
        q.setParameter("id", id);
        q.setParameter("t1", t1);
        q.setParameter("t2", t2);
        @SuppressWarnings("unchecked")
        List<StatData> l = q.list();
        log.debug("retrieved stat data " + id + "[" + t1 + "," + t2 + "] : "
                + l.size());
        sess.close();
        return l;
    }

    protected Data exportRawData(long rawId, long t1, long t2) {
        List<RawData> l = getRawData(rawId, t1, t2);

        Data d = new Data();
        d.setDataMetaData(getMetadata((int) rawId, t1, t2));

        TrendingData[] data = new TrendingData[l.size()];
        for (int i = 0; i < l.size(); i++) {
            RawData r = l.get(i);
            TrendingData dt = new TrendingData();
            data[i] = dt;
            long tStamp = r.getTstamp();
            AxisValue axisValue = new AxisValue("time", tStamp);
            dt.setAxisValue(axisValue);
            DataValue[] dataValue = new DataValue[1];
            Double dd = r.getDoubleData();
            dataValue[0] = new DataValue("value", dd == null ? 0 : dd);
            dt.setDataValue(dataValue);
        }
        d.getTrendingResult().setTrendingDataArray(data);

        return d;
    }

    protected Data exportStatDataFromRaw(long rawId, long t1, long t2, int nsamples) {

        Session sess = fac.openSession();

        SQLQuery q = sess
                .createSQLQuery("select tlow, thigh, datasum/entries as mean, "
                        + " sqrt((datasumsquared - datasum*datasum/entries)/(entries-1)) as rms, "
                        + " mindata, maxdata "
                        + " from ( SELECT MIN(rd.tstampmills) AS tlow, MAX(rd.tstampmills) AS thigh, "
                        + " SUM(rd.doubleData) AS datasum, SUM(rd.doubleData*rd.doubleData) AS datasumsquared, "
                        + " min(rd.doubleData) as mindata, max(rd.doubleData) as maxdata, "
                        + " count(1) AS entries from rawdata rd where descr_id = :id and tstampmills >= :t1 "
                        + " and tstampmills <= :t2 group by floor(rd.tstampmills/:deltat) ) accumulated where entries > 0 ");

        long deltat = (t2 - t1) / nsamples;

        q.setParameter("id", rawId);
        q.setParameter("t1", t1);
        q.setParameter("t2", t2);
        q.setParameter("deltat", deltat);
        @SuppressWarnings("unchecked")
        List<Object[]> l = q.list();

        sess.close();

        List<TrendingData> trendingDataList = new ArrayList<>();
        for (Object[] obj : l) {

            if (obj[2] != null) {
                long low = ((BigInteger) obj[0]).longValue();
                long high = ((BigInteger) obj[1]).longValue();
                TrendingData dt = new TrendingData();
                dt.setAxisValue(new AxisValue("time", (low + high) / 2, low, high));
                DataValue[] dataValue = new DataValue[4];

                double value = ((Double) obj[2]).doubleValue();
                double rms = 0;
                if (obj[3] != null) {
                    rms = ((Double) obj[3]).doubleValue();
                }
                double min = ((Double) obj[4]).doubleValue();
                double max = ((Double) obj[5]).doubleValue();
                dataValue[0] = new DataValue("value", value);
                dataValue[1] = new DataValue("rms", rms);
                dataValue[2] = new DataValue("min", min);
                dataValue[3] = new DataValue("max", max);
                dt.setDataValue(dataValue);
                trendingDataList.add(dt);
            }
        }

        TrendingData[] trendingData = trendingDataList.toArray(new TrendingData[0]);
        
        Data d = new Data();
        d.setDataMetaData(getMetadata((int) rawId, t1, t2));
        d.getTrendingResult().setTrendingDataArray(trendingData);

        return d;
    }

    protected Data exportStatDataFromStat(long rawId, long id, long t1, long t2,
            int nsamples) {
        List<StatData> in = getStatData(id, t1, t2);

        int n = in.size();
        int rebin = 1;
        int nout = n;
        if (n > nsamples * 3) {
            rebin = n / nsamples;
            nout = (n + rebin - 1) / rebin;
            log.debug("will rebin stat by " + rebin + " : " + nout);
        }

        Data d = new Data();
        d.setDataMetaData(getMetadata((int) rawId, t1, t2));

        d.getTrendingResult().setTrendingDataArray(new TrendingData[nout]);
        int iout = 0;
        int ibin = 0;
        double sum = 0;
        double s2 = 0;
        long nsamp = 0;
        long low = 0;
        double max = Double.MIN_NORMAL;
        double min = Double.MAX_VALUE;
        for (StatData sd : in) {
            sum += sd.getSum();
            s2 += sd.getSum2();
            if (sd.getMax() > max) {
                max = sd.getMax();
            }
            if (sd.getMin() < min) {
                min = sd.getMin();
            }
            nsamp += sd.getN();
            if (ibin == 0) {
                low = sd.getTstampFirst();
            }
            if (ibin == rebin - 1 || (iout * rebin + ibin == n - 1)) {
                log.debug("storing for " + iout + " from " + (ibin + 1)
                        + " bins");
                TrendingData dt = new TrendingData();
                d.getTrendingResult().getTrendingDataArray()[iout] = dt;
                dt.setAxisValue(new AxisValue("time",
                        (low + sd.getTstampLast()) / 2, low, sd.getTstampLast()));
                DataValue[] dataValue = new DataValue[4];
                dataValue[0] = new DataValue("value", sum / n);
                dataValue[1] = new DataValue("rms", s2 / n - (sum / n)
                        * (sum / n));
                dataValue[2] = new DataValue("min", min);
                dataValue[3] = new DataValue("max", max);
                dt.setDataValue(dataValue);
                iout++;
                ibin = 0;
            } else {
                ibin++;
            }
        }

        return d;
    }

// TODO change that to a static method that can be called from the StatusPersisterSubsystem	
    public static void main(String[] args) throws IOException {
        class MyApplication extends Application {

            @Override
            public Set<Class<?>> getClasses() {
                Set<Class<?>> s = new HashSet<Class<?>>();
                s.add(DataServer.class);
                return s;
            }
        }

        int socket = 8080;
        if (args.length > 0) {
            socket = Integer.decode(args[0]);
        }
        MyApplication app = new MyApplication();
        HttpHandler h = RuntimeDelegate.getInstance().createEndpoint(app, HttpHandler.class);
        HttpServer s = HttpServer.create(new InetSocketAddress(socket), 5);
        s.createContext("/rest/data/", h);
        s.start();
    }

}
