/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.subsystem.shutter;

import java.nio.ByteBuffer;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.shutter.PLCVariableDictionary;
import org.lsst.ccs.subsystem.shutter.SimMessage;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.EncoderSample;
import org.lsst.ccs.subsystem.shutter.common.HallTransition;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.CloseShutter;
import org.lsst.ccs.subsystem.shutter.plc.GoToProd;
import org.lsst.ccs.subsystem.shutter.plc.HomeAxisPLC;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.plc.MoveAxisAbsolutePLC;
import org.lsst.ccs.subsystem.shutter.plc.MsgToPLC;
import org.lsst.ccs.subsystem.shutter.plc.OpenShutter;
import org.lsst.ccs.subsystem.shutter.plc.PLCMsg;
import org.lsst.ccs.subsystem.shutter.plc.ShutterStatusPLC;
import org.lsst.ccs.subsystem.shutter.plc.TakeExposure;
import org.lsst.ccs.subsystem.shutter.plc.ToggleSafetyCheck;
import org.lsst.ccs.subsystem.shutter.sim.CubicSCurve;
import org.lsst.ccs.subsystem.shutter.sim.MotionProfile;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.utilities.scheduler.Scheduler;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

public final class SimulatedShutter {
    private static final Logger LOG = Logger.getLogger(SimulatedShutter.class.getName());
    private final Scheduler scheduler;
    private final BlockingQueue<SimMessage> outQueue;
    private final BlockingQueue<MsgToPLC> inQueue;
    private final PLCVariableDictionary dictionary;
    private final Map<ShutterSide, SideInfo> sdinfo;
    private ShutterSide closingSide;
    private boolean safetyOn;
    private int openCount = 0;
    private static final long[] MOTION_TIME = new long[]{730L, 1000L, 2000L};
    private int timeCycle = 0;

    SimulatedShutter(Scheduler scheduler, BlockingQueue<SimMessage> outQueue, PLCVariableDictionary dictionary) {
        this.scheduler = scheduler;
        this.inQueue = new ArrayBlockingQueue<MsgToPLC>(1000);
        this.outQueue = outQueue;
        this.dictionary = dictionary;
        this.sdinfo = new EnumMap<ShutterSide, SideInfo>(ShutterSide.class);
        this.safetyOn = true;
    }

    void start() {
        LOG.info("Starting controller simulation.");
        this.scheduler.scheduleWithFixedDelay(this::taskBody, 0L, 10L, TimeUnit.MILLISECONDS);
    }

    void accept(MsgToPLC msg) {
        LOG.info(String.format("Accepting message %s.", msg.getClass().getSimpleName()));
        if (!this.inQueue.offer(msg)) {
            LOG.warning("Input queue is full.");
        }
    }

    private void taskBody() {
        this.setInitialPositions();
        this.sendStatus();
        LOG.info("In event loop for simulated shutter.");
        try {
            while (true) {
                MsgToPLC msg;
                PLCVariableDictionary.InVariable var;
                if ((var = this.dictionary.getInVariable((msg = this.inQueue.take()).getClass())) == null) {
                    LOG.warning(String.format("Incoming message %s not in dictionary.", msg.getClass().getName()));
                    continue;
                }
                this.sendMessage(var.ackName, msg);
                this.interpretMessage(msg);
            }
        }
        catch (InterruptedException exc) {
            LOG.info("Normal termination of shutter controller simulation.");
            return;
        }
    }

    private void sendMessage(String varName, PLCMsg msg) {
        LOG.info(String.format("Sending message %s.", msg.getClass().getSimpleName()));
        ByteBuffer data = ByteBuffer.allocate(10240);
        msg.encode(data);
        data.limit(data.position());
        data.rewind();
        SimMessage reply = new SimMessage(varName, data);
        if (!this.outQueue.offer(reply)) {
            LOG.warning("Output queue is full.");
        }
    }

    private void interpretMessage(MsgToPLC eventMsg) {
        if (eventMsg instanceof OpenShutter) {
            ShutterSide side = this.leastExtended().opposite();
            this.simulateMotion(side, this.sdinfo.get((Object)((Object)side)).homePos);
            this.closingSide = side.opposite();
        } else if (eventMsg instanceof CloseShutter) {
            ShutterSide side = this.closingSide;
            this.simulateMotion(side, this.sdinfo.get((Object)((Object)side)).deployedPos);
            this.closingSide = null;
        } else if (eventMsg instanceof TakeExposure) {
            ShutterSide side = this.leastExtended().opposite();
            this.simulateMotion(side, this.sdinfo.get((Object)((Object)side)).homePos);
            this.simulateMotion(side.opposite(), this.sdinfo.get((Object)((Object)side.opposite())).deployedPos);
        } else if (eventMsg instanceof MoveAxisAbsolutePLC) {
            MoveAxisAbsolute abs = ((MoveAxisAbsolutePLC)eventMsg).getStatusBusMessage();
            ShutterSide side = ShutterSide.fromAxis(Axis.fromName(abs.getAxisName()));
            this.simulateMotion(side, abs.getPosition());
            this.closingSide = null;
        } else if (eventMsg instanceof GoToProd) {
            this.setInitialPositions();
            this.sendStatus();
        } else if (eventMsg instanceof ChangeBrakeState) {
            this.simulateChangeBrakeState((ChangeBrakeState)eventMsg);
        } else if (eventMsg instanceof HomeAxisPLC) {
            this.simulateHomeAxis((HomeAxisPLC)eventMsg);
        } else if (eventMsg instanceof ToggleSafetyCheck) {
            this.safetyOn = !this.safetyOn;
            this.sendStatus();
        }
    }

    private void simulateChangeBrakeState(ChangeBrakeState msg) {
        boolean brakeEngaged;
        ShutterSide side = msg.getAxis().isPlusXSide() ? ShutterSide.PLUSX : ShutterSide.MINUSX;
        this.sdinfo.get((Object)((Object)side)).brakeEngaged = brakeEngaged = msg.getState() == ChangeBrakeState.State.ENGAGED;
        this.sendStatus();
    }

    private void setInitialPositions() {
        this.sdinfo.put(ShutterSide.PLUSX, new SideInfo(750.0, false, 0.0, 750.0));
        this.sdinfo.put(ShutterSide.MINUSX, new SideInfo(750.0, false, 750.0, 0.0));
        this.closingSide = null;
    }

    private ShutterSide leastExtended() {
        if (this.sdinfo.get((Object)ShutterSide.PLUSX).extent() < this.sdinfo.get((Object)ShutterSide.MINUSX).extent()) {
            return ShutterSide.PLUSX;
        }
        return ShutterSide.MINUSX;
    }

    private void simulateMotion(ShutterSide side, double targetPos) {
        double startPos = this.sdinfo.get((Object)((Object)side)).position;
        Duration dur = Duration.ofMillis(MOTION_TIME[this.timeCycle++ % MOTION_TIME.length]);
        CCSTimeStamp startTime = CCSTimeStamp.currentTime();
        MotionDone.Builder motionBuilder = new MotionDone.Builder().side(side).startPosition(startPos).targetPosition(targetPos).startTime(startTime).endPosition(targetPos).targetDuration(dur).actualDuration(dur).hallTransitions(new ArrayList<HallTransition>()).encoderSamples(new ArrayList<EncoderSample>());
        CubicSCurve profile = new CubicSCurve(targetPos - startPos, 0.001 * (double)dur.toMillis());
        motionBuilder.hallTransitions(this.makeHallTransitions(startTime, startPos, dur, profile));
        motionBuilder.encoderSamples(this.makeEncoderSamples(startTime, startPos, dur, profile));
        this.sdinfo.get((Object)((Object)side)).position = targetPos;
        MotionDonePLC plcMotion = new MotionDonePLC(0, motionBuilder.build());
        PLCVariableDictionary.OutVariable var = this.dictionary.getOutVariable(plcMotion.getClass());
        if (var == null) {
            LOG.warning(String.format("Outgoing message %s not in dictionary.", plcMotion.getClass().getName()));
        } else {
            this.sendMessage(var.varName, plcMotion);
        }
        this.sendStatus();
    }

    private List<EncoderSample> makeEncoderSamples(CCSTimeStamp t0Stamp, double startPos, Duration moveTime, MotionProfile profile) {
        int NUM_SAMPLES = 10;
        Instant t0 = t0Stamp.getUTCInstant();
        Duration dt = moveTime.dividedBy(11L);
        Instant t = t0.plus(dt);
        ArrayList<EncoderSample> samp = new ArrayList<EncoderSample>(10);
        for (int i = 0; i < 10; ++i) {
            CCSTimeStamp stamp = CCSTimeStamp.currentTimeFromMillis((long)t.toEpochMilli());
            double elapsed = 0.001 * (double)Duration.between(t0, stamp.getUTCInstant()).toMillis();
            samp.add(new EncoderSample(stamp, startPos + profile.distance(elapsed)));
            t = t.plus(dt);
        }
        return samp;
    }

    private List<HallTransition> makeHallTransitions(CCSTimeStamp t0Stamp, double startPos, Duration moveTime, MotionProfile profile) {
        int NUM_SENSORS = 8;
        Duration hallWidth = moveTime.dividedBy(50L);
        Instant t0 = t0Stamp.getUTCInstant();
        Duration dt = moveTime.dividedBy(9L);
        Instant t = t0.plus(dt);
        ArrayList<HallTransition> hall = new ArrayList<HallTransition>(8);
        for (int i = 0; i < 8; ++i) {
            CCSTimeStamp stamp = CCSTimeStamp.currentTimeFromMillis((long)t.toEpochMilli());
            double elapsed = 0.001 * (double)Duration.between(t0, stamp.getUTCInstant()).toMillis();
            hall.add(new HallTransition(stamp, i, startPos + profile.distance(elapsed) + (double)(i + 1) * 1.1, true));
            stamp = CCSTimeStamp.currentTimeFromMillis((long)t.plus(hallWidth).toEpochMilli());
            elapsed = 0.001 * (double)Duration.between(t0, stamp.getUTCInstant()).toMillis();
            hall.add(new HallTransition(stamp, i, startPos + profile.distance(elapsed), false));
            t = t.plus(dt);
        }
        return hall;
    }

    private void sendStatus() {
        EnumMap<ShutterSide, ShutterStatusPLC.AxisStatusPLC> axes = new EnumMap<ShutterSide, ShutterStatusPLC.AxisStatusPLC>(ShutterSide.class);
        for (ShutterSide side : ShutterSide.values()) {
            ShutterStatusPLC.AxisStatusPLC axstat = new ShutterStatusPLC.AxisStatusPLC(this.sdinfo.get((Object)((Object)side)).position, side == ShutterSide.PLUSX ? 1667.0 : -1234.0, 0.0, true, this.sdinfo.get((Object)((Object)side)).brakeEngaged, false, false, true, 0, side == ShutterSide.PLUSX ? 30.0 : -100.0);
            axes.put(side, axstat);
        }
        ShutterStatusPLC plcStatus = new ShutterStatusPLC(0, 0, true, 0, axes, this.safetyOn, Arrays.asList(405, 420, 307));
        PLCVariableDictionary.OutVariable var = this.dictionary.getOutVariable(plcStatus.getClass());
        if (var == null) {
            LOG.warning(String.format("Outgoing message %s not in dictionary.", plcStatus.getClass().getName()));
        } else {
            this.sendMessage(var.varName, plcStatus);
        }
    }

    private void simulateHomeAxis(HomeAxisPLC msg) {
        Axis ax = Axis.fromName(msg.getAxisName());
        MotionDone.Builder bld = new MotionDone.Builder().actualDuration(Duration.ofSeconds(1L)).encoderSamples(Collections.emptyList()).endPosition(0.0).hallTransitions(Collections.emptyList()).side(ax.isPlusXSide() ? ShutterSide.PLUSX : ShutterSide.MINUSX).startPosition(0.0).startTime(CCSTimeStamp.currentTime()).targetDuration(Duration.ofSeconds(1L)).targetPosition(0.0);
        MotionDone motion = bld.build();
        MotionDonePLC mplc = new MotionDonePLC(0, motion);
        PLCVariableDictionary.OutVariable var = this.dictionary.getOutVariable(mplc.getClass());
        if (var == null) {
            LOG.warning(String.format("Outgoing message %s not in dictionary.", mplc.getClass().getName()));
        } else {
            this.sendMessage(var.varName, mplc);
        }
    }

    private static final class SideInfo {
        double position;
        boolean brakeEngaged;
        final double homePos;
        final double deployedPos;
        final double fullStroke;
        final double direction;

        SideInfo(double position, boolean brakeEngaged, double homePos, double deployedPos) {
            this.position = position;
            this.brakeEngaged = brakeEngaged;
            this.homePos = homePos;
            this.deployedPos = deployedPos;
            double s = deployedPos - homePos;
            this.direction = Math.signum(s);
            this.fullStroke = Math.abs(s);
        }

        boolean isExtended() {
            return this.extent() > 0.99;
        }

        boolean isRetracted() {
            return this.extent() < 0.01;
        }

        double extent() {
            return this.direction * (this.position - this.homePos) / this.fullStroke;
        }
    }
}

