/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.drivers.gpio;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileLock;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.lsst.ccs.drivers.commons.DriverException;

public class GPIODriver {
    private final Path root;
    private final Map<Integer, GPIOChannel> channelMap = new HashMap<Integer, GPIOChannel>();
    private final Predicate<String> chipPattern = Pattern.compile("gpiochip\\d+").asPredicate();
    private final Predicate<String> channelPattern = Pattern.compile("gpio\\d+").asPredicate();

    public GPIODriver() throws DriverException {
        File rootFile = new File("/sys/class/gpio");
        if (!rootFile.isDirectory()) {
            throw new DriverException("GPIO is not supported on this system");
        }
        this.root = rootFile.toPath();
    }

    private GPIOChip readChip(Path dir) throws WrappedDriverException {
        try {
            return new GPIOChip(dir);
        }
        catch (DriverException ex) {
            throw new WrappedDriverException(ex);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public List<GPIOChip> enumerateChips() throws DriverException {
        try (Stream<Path> stream = Files.list(this.root).filter(t -> this.chipPattern.test(t.getFileName().toString()));){
            List<GPIOChip> list = stream.map(p -> this.readChip((Path)p)).collect(Collectors.toList());
            return list;
        }
        catch (WrappedDriverException ex) {
            throw ex.getWrappedException();
        }
        catch (IOException ex) {
            throw new DriverException("Unexpected IO exception", (Throwable)ex);
        }
    }

    private GPIOChannel getChannel(Path dir) {
        int channel = Integer.parseInt(dir.getFileName().toString().substring(4));
        return this.getChannelFromMap(channel, dir);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public List<GPIOChannel> enumerateExportedChannels() throws DriverException {
        try (Stream<Path> stream = Files.list(this.root).filter(t -> this.channelPattern.test(t.getFileName().toString()));){
            List<GPIOChannel> list = stream.map(p -> this.getChannel((Path)p)).collect(Collectors.toList());
            return list;
        }
        catch (IOException ex) {
            throw new DriverException("Unexpected IO exception", (Throwable)ex);
        }
    }

    public GPIOChannel getChannel(int channel) throws DriverException {
        Path path = this.root.resolve("gpio" + channel);
        if (!path.toFile().isDirectory()) {
            throw new DriverException("Invalid channel " + channel + " (not exported?)");
        }
        return this.getChannelFromMap(channel, path);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private GPIOChannel getChannelFromMap(int channel, Path path) {
        Map<Integer, GPIOChannel> map = this.channelMap;
        synchronized (map) {
            GPIOChannel result = this.channelMap.get(channel);
            if (result == null) {
                result = new GPIOChannel(channel, path);
                this.channelMap.put(channel, result);
            }
            return result;
        }
    }

    public boolean isExported(int channel) {
        Path path = this.root.resolve("gpio" + channel);
        return path.toFile().isDirectory();
    }

    public GPIOChannel export(int channel) throws DriverException {
        GPIODriver.writeToFile(this.root.resolve("export"), Integer.toString(channel));
        return this.getChannel(channel);
    }

    public List<GPIOChannel> exportRange(int base, int ngpio) throws DriverException {
        ArrayList<GPIOChannel> result = new ArrayList<GPIOChannel>();
        for (int channel = base; channel < base + ngpio; ++channel) {
            if (!this.isExported(channel)) {
                result.add(this.export(channel));
                continue;
            }
            result.add(this.getChannel(channel));
        }
        return result;
    }

    public void unexportRange(int base, int ngpio) throws DriverException {
        for (int channel = base; channel < base + ngpio; ++channel) {
            if (!this.isExported(channel)) continue;
            this.unexport(channel);
        }
    }

    public void unexport(int channel) throws DriverException {
        GPIODriver.writeToFile(this.root.resolve("unexport"), Integer.toString(channel));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static String getFirstLine(Path file) throws DriverException {
        try (Stream<String> lines = Files.lines(file);){
            String string = lines.findFirst().get();
            return string;
        }
        catch (IOException | NoSuchElementException x) {
            throw new DriverException("Error reading " + file, (Throwable)x);
        }
    }

    private static void writeToFile(Path file, String line) throws DriverException {
        try {
            Files.write(file, line.getBytes(), new OpenOption[0]);
        }
        catch (IOException x) {
            throw new DriverException("Error writing " + line + " to " + file, (Throwable)x);
        }
    }

    private static final class WrappedDriverException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        WrappedDriverException(DriverException x) {
            this.initCause((Throwable)x);
        }

        private DriverException getWrappedException() {
            return (DriverException)this.getCause();
        }
    }

    public final class GPIOChip {
        private final int base;
        private final int ngpio;
        private final String label;

        private GPIOChip(Path dir) throws DriverException {
            try {
                this.base = Integer.parseInt(GPIODriver.getFirstLine(dir.resolve("base")));
                this.ngpio = Integer.parseInt(GPIODriver.getFirstLine(dir.resolve("ngpio")));
                this.label = GPIODriver.getFirstLine(dir.resolve("label"));
            }
            catch (NumberFormatException ex) {
                throw new DriverException("Error parsing chip " + dir, (Throwable)ex);
            }
        }

        public int getBase() {
            return this.base;
        }

        public int getNgpio() {
            return this.ngpio;
        }

        public String getLabel() {
            return this.label;
        }

        public List<GPIOChannel> exportAll() throws DriverException {
            return GPIODriver.this.exportRange(this.base, this.ngpio);
        }

        public void unexportAll() throws DriverException {
            GPIODriver.this.unexportRange(this.base, this.ngpio);
        }

        public String toString() {
            return "GPIOChip{base=" + this.base + ", ngpio=" + this.ngpio + ", label=" + this.label + '}';
        }
    }

    public static final class GPIOChannel {
        private final int channel;
        private final Path path;
        private RandomAccessFile randomAccessValueFile;
        private final Path valuePath;
        private static final byte ONE_BYTE = "1".getBytes()[0];
        private static final byte ZERO_BYTE = "0".getBytes()[0];
        private FileLock lock;

        private GPIOChannel(int channel, Path path) {
            this.channel = channel;
            this.path = path;
            this.valuePath = path.resolve("value");
        }

        public Direction getDirection() throws DriverException {
            return Direction.valueOf(GPIODriver.getFirstLine(this.path.resolve("direction")).toUpperCase());
        }

        public void setDirection(Direction dir) throws DriverException {
            GPIODriver.writeToFile(this.path.resolve("direction"), dir.toString().toLowerCase());
        }

        public void set() throws DriverException {
            this.write(true);
        }

        public void clear() throws DriverException {
            this.write(false);
        }

        public boolean read() throws DriverException {
            if (this.randomAccessValueFile != null) {
                try {
                    this.randomAccessValueFile.seek(0L);
                    return ONE_BYTE == this.randomAccessValueFile.readByte();
                }
                catch (IOException ex) {
                    throw new DriverException("Error reading channel", (Throwable)ex);
                }
            }
            return "1".equals(GPIODriver.getFirstLine(this.valuePath));
        }

        public void write(boolean value) throws DriverException {
            if (this.randomAccessValueFile != null) {
                try {
                    this.randomAccessValueFile.seek(0L);
                    this.randomAccessValueFile.write(value ? ONE_BYTE : ZERO_BYTE);
                }
                catch (IOException ex) {
                    throw new DriverException("Error reading channel", (Throwable)ex);
                }
            } else {
                GPIODriver.writeToFile(this.valuePath, value ? "1" : "0");
            }
        }

        public void lock() throws DriverException {
            try {
                this.randomAccessValueFile = new RandomAccessFile(this.valuePath.toFile(), "rws");
                this.lock = this.randomAccessValueFile.getChannel().lock();
            }
            catch (IOException ex) {
                throw new DriverException("Unable to lock file", (Throwable)ex);
            }
        }

        public void unlock() throws DriverException {
            try {
                RandomAccessFile local = this.randomAccessValueFile;
                if (local != null) {
                    this.randomAccessValueFile = null;
                    this.lock.release();
                    local.close();
                }
            }
            catch (IOException ex) {
                throw new DriverException("Unable to unlock file", (Throwable)ex);
            }
        }

        public int getChannel() {
            return this.channel;
        }

        public String toString() {
            return "GPIOChannel{channel=" + this.channel + '}';
        }

        public static enum Direction {
            IN,
            OUT;

        }
    }
}

