package org.lsst.ccs.utilities.image;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import nom.tam.fits.FitsException;

/**
 * A specification for a fits file header. Typically read from a .spec file.
 * For a given HeaderSpecification it is possible to read multiple spec files
 * to allow for overwriting of information on the part of subsystems.
 * 
 * Any uploaded spec file will overwrite the existing information.
 * 
 * @author tonyj
 */
public class HeaderSpecification {

    private String name;
    private final Map<String, HeaderLine> headers = new LinkedHashMap<>();
    private final static Pattern EXPRESSION_PATTERN = Pattern.compile("\\$\\{(?<exp>[^\\|]*)(?<default>\\|.*)?}");
    private final static Pattern INNER_EXPRESSION_PATTERN = Pattern.compile("(?<pre>.*)(?<exp>(\\$\\{(?<value>[^\\$\\{\\}]*)\\}))(?<post>.*)");
    private final static Pattern SYSTEM_PROPERTY_PATTERN = Pattern.compile(".*(\\[System\\(([^,]+)(,.*)?\\)\\]).*");
    private final static Pattern LINE_PATTERN = Pattern.compile("\\s*(\\S+)\\s+(\\w+)\\s+(?:\"(.*)\"|(\\S+))\\s*(.*)");
    private static final Logger LOG = Logger.getLogger(HeaderSpecification.class.getName());

    public enum DataType {
        Boolean, Integer, String, Float, Date, MJD
    };

    public HeaderSpecification(String name, InputStream in) throws IOException {
        this(name, new InputStreamReader(in));
    }

    public HeaderSpecification(String name, Reader in) throws IOException {
        this.name = name;
        loadHeaderSpecification(in);
    }

    /**
     * Load Header Specification information from an InputStream.
     * The content of the HeaderSpecification will be updated based on the 
     * provided information.
     * Existing information will be over-written so that the end result will
     * be the superposition of what already exists with what has been provided.
     * 
     * @param in The input stream with the HeadSpecification information.
     * @throws java.io.IOException For any problems with the InputStream.
     */    
    final void loadHeaderSpecification(InputStream in) throws IOException {
        loadHeaderSpecification(new InputStreamReader(in));
        
    }

    final void loadHeaderSpecification(Reader in) throws IOException {
        BufferedReader reader = in instanceof BufferedReader
                ? (BufferedReader) in : new BufferedReader(in);
        for (;;) {
            String line = reader.readLine();
            if (line == null) {
                break;
            } else if (line.startsWith("#") || line.trim().isEmpty() ) {
                continue;
            }
            try {
                while (true) {
                    Matcher sysPropMatcher = SYSTEM_PROPERTY_PATTERN.matcher(line);
                    if (sysPropMatcher.matches()) {
                        String def = sysPropMatcher.group(3);
                        if ( def == null ) {
                            def = "";
                        } else {
                            def = def.substring(1);
                        }
                        line = line.replace(sysPropMatcher.group(1), System.getProperty(sysPropMatcher.group(2), def));
                    } else {
                        break;
                    }
                }
                HeaderLine headerLine = new HeaderLine(line);
                if ( headers.containsKey(headerLine.getKeyword()) ) {
                    LOG.log(Level.INFO,"Duplicate definition for header keyword: {0}. Overriding current definition {1}.", new Object[] {headerLine.getKeyword(), headers.get(headerLine.getKeyword()).getMetaName()});
                }
                headers.put(headerLine.getKeyword(), headerLine);
            } catch (IOException x) {
                throw new IOException("Error while reading line:\n\t" + line, x);
            }
        }        
    }
    
    /**
     * The name of the HeaderSpecification. It is used to map the HeaderSpecification
     * with the fits header being written out.
     * 
     * @return 
     */
    public String getName() {
        return name;
    }

    /**
     * Get a specific HeaderLine object specifying the full definition for
     * a given header keyword.
     * 
     * @param keyWord The keyword name for the HeaderLine
     * @return        The HeaderLine defining the content of the header.
     */
    public HeaderLine getHeader(String keyWord) {
        return headers.get(keyWord);
    }
    
    /**
     * Get the full collection of HeaderLines for this HeaderSpecification.
     * The returned collection is unmodifiable.
     * @return The unmodifiable collection of HeaderLine objects.
     */
    public Collection<HeaderLine> getHeaders() {
        return Collections.unmodifiableCollection(headers.values());
    }

    /**
     * Get the full collection of HeaderLines for this HeaderSpecification.
     * The returned collection is unmodifiable.
     * @return The unmodifiable collection of HeaderLine objects.
     */
    Collection<HeaderLine> getRequiredHeaders() {
        return headers.values().stream().filter(h -> h.isRequired).collect(Collectors.toCollection(ArrayList<HeaderLine>::new));
    }

    /**
     * A Class defining each Header line contained in a HeaderSpecification.
     * A HeaderLine is built from each of the lines contained in a header specification
     * file (.spec file).
     */
    public static class HeaderLine {

        private final String keyword;
        private final DataType dataType;
        private final String comment;
        private String metaName;
        private Object value;
        private String metaMap;
        private String valueExpression;
        private final boolean isExpression;
        private boolean evaluateOnTheFly = false;
        private final boolean isRequired;
        private final String originalLine;
        private final Object defaultValue;

        HeaderLine(String line) throws IOException {
            originalLine = line;
            try {
                Matcher lineMatcher = LINE_PATTERN.matcher(line);
                if (!lineMatcher.matches()) {
                    throw new IOException("Invalid line in header specification");
                }
                String tmpKeyword = lineMatcher.group(1);
                isRequired = tmpKeyword.endsWith("!");
                keyword = isRequired ? tmpKeyword.substring(0,tmpKeyword.length()-1) : tmpKeyword;

                dataType = DataType.valueOf(lineMatcher.group(2));
                valueExpression = lineMatcher.group(3) == null ? lineMatcher.group(4) : lineMatcher.group(3);
                Matcher matcher = EXPRESSION_PATTERN.matcher(valueExpression);
                isExpression = matcher.matches();
                if (isExpression) {
                    String expression = matcher.group("exp");
                    String defaultExp = matcher.group("default");
                    
                    if ( defaultExp != null && !defaultExp.isEmpty() ) {
                        valueExpression = valueExpression.replace(defaultExp, "");
                        defaultExp = defaultExp.replace("|", "");
                        defaultValue = coerce(defaultExp,dataType);                        
                    } else {
                        defaultValue = null;
                    }                    
                    
                    Matcher inner_matcher = INNER_EXPRESSION_PATTERN.matcher(valueExpression);
                    if ( inner_matcher.matches() ) {
                        if (!expression.equals(inner_matcher.group("value"))) {
                            evaluateOnTheFly = true;
                        }
                    }
                    if ( ! evaluateOnTheFly ) {
                        String[] split = splitKey(expression);
                        metaMap = split[0];
                        metaName = split[1];
                        value = null;
                    }
                } else {
                    metaMap = null;
                    metaName = null;
                    value = coerce(valueExpression,dataType);
                    defaultValue = null;
                }
                comment = lineMatcher.group(5).trim();
            } catch (IllegalArgumentException | FitsException x) {
                throw new IOException("Illegal token found while reading header specification", x);
            }
            FitsHeadersSpecificationsBuilder.log.debug("HeaderLine "+keyword+" "+metaMap+" "+metaName+" ["+originalLine+"]");
        }

        /**
         * The keyword name as it will be written in the fits file.
         * 
         * @return The keyword name.
         */
        public String getKeyword() {
            return keyword;
        }

        /**
         * The data type for this HeaderLine.
         * 
         * @return The data type.
         */
        public DataType getDataType() {
            return dataType;
        }

        public String getMetaName() {
            return metaName;
        }
        
        String getMetaMap() {
            return metaMap;
        }
        /**
         * The Comment as it will be written in the fits file.
         * It describes the keyword.
         * 
         * @return The comment line.
         */
        public String getComment() {
            return comment;
        }
        
        boolean isExpression() {
            return isExpression;
        }

        boolean isRequired() {
            return isRequired;
        }
        
        String getHeaderDefinition() {
            return originalLine;
        }

        Object getValue(MetaDataSet metaDataSet) {
            Object val = null;
            if (evaluateOnTheFly) {
                String expressionToBeParsed = valueExpression;
                boolean done = false;
                while (true) {
                    Matcher matcher = INNER_EXPRESSION_PATTERN.matcher(expressionToBeParsed);
                    if (matcher.matches()) {

                        String expressionToReplace = matcher.group("exp");
                        String key = matcher.group("value");
                        done = expressionToReplace.equals(expressionToBeParsed);

                        if (done) {
                            val = findMetaDataValue(key, metaDataSet, dataType);
                            break;
                        }
                        Object value = findMetaDataValue(key, metaDataSet, DataType.String);
                        if (value == null) {
                            throw new RuntimeException("Could not replace value " + key + " [" + expressionToReplace + "]");
                        } else {
                            expressionToBeParsed = expressionToBeParsed.replace(expressionToReplace, (String) value);
                        }
                    } else {
                        break;
                    }
                }
                if (!done) {
                    val = findMetaDataValue(expressionToBeParsed, metaDataSet, dataType);
                }
            } else {
                if (isExpression) {
                    val = findMetaDataValue(metaMap, metaName, metaDataSet, dataType);
                } else {
                    val = value;
                }
            }
            return val != null ? val : (defaultValue != null ? defaultValue : val);
        }
        
        
        //Split a key in map, name
        private String[] splitKey(String key) {
            String[] result = new String[2];
            int dotIndex = key.indexOf(".");
            int slashIndex = key.indexOf("/");
            if (dotIndex == -1 || (dotIndex != -1 && slashIndex != -1 && slashIndex < dotIndex)) {
                result[0] = null;
                result[1] = key;
            } else {
                result[0] = key.substring(0, dotIndex);
                result[1] = key.substring(dotIndex + 1);
            }
            return result;
        }

        private Object findMetaDataValue(String key, MetaDataSet metaDataSet, DataType type) {            
            String[] split = splitKey(key);
            String map = split[0];
            String name = split[1];            
            return findMetaDataValue(map, name, metaDataSet, type);
        }
        
        private Object findMetaDataValue(String map, String name, MetaDataSet metaDataSet, DataType type) {
            Object obj = metaDataSet.getValue(map, name);
            if ( obj instanceof String ) {
                try {
                    obj = coerce((String)obj, type);
                } catch (Exception e) {}
            }
            return obj;
        }

    }   
    
        /**
         * Work around annoying feature of Integer.decode interpreting leading zeros as octal.
         */
        private static Integer sensibleDecode(String valueExpression) {
            if (valueExpression.startsWith("0x")) return Integer.decode(valueExpression);
            else return Integer.valueOf(valueExpression);
        }
    
        private static Object coerce(String valueExpression, DataType dataType) throws FitsException, NumberFormatException {
            switch (dataType) {
                case Integer: return sensibleDecode(valueExpression);
                case Float: return Double.valueOf(valueExpression);
                case Boolean: return Boolean.valueOf(valueExpression);
                case Date: return DateUtils.convertStringToDate(valueExpression);
                case MJD: return Double.valueOf(valueExpression);
                default: return valueExpression;
            }
        }
    public static void main(String[] argv) throws Exception {
        Map<String,String> map = new HashMap();
        map.put("something/23","1");
        map.put("else","2");
        map.put("again", "3");
        String expressionVal = "${something/${else}${again}}";
//        HeaderValueParser p = new HeaderValueParser(expressionVal, DataType.String, map);
        
        
        
    }
       
}
