package org.lsst.ccs.utilities.ccd;

/**
 * An object that defines CCD specific geometrical constants.
 * 
 * We assume that all the Segments within a CCD are described by the same 
 * SegmentGeometryConstraint object, i.e. they have the same geometry.
 * 
 * @author The LSST CCS Team
 */
public final class CCDGeometryConstants {
    
    private final double parallelEdge, serialEdge;
    private final SegmentGeometryConstants segmentGeometryConstants;
    public static final int SEGMENTS_ALONG_SERIAL = 8;
    public static final int SEGMENTS_ALONG_PARALLEL = 2;
    public final int parallelSegments;
    public final int serialSegments;
    
    /**
     * Build a CCDGemetryConstants object describing a CCD geometry constants.
     * A CCD contains 16 segments whose geometry constants are defined by the
     * provided SegmentGeometryConstants. These Segments are all side by side, 
     * 2 in the parallel direction and 8 in the serial direction.
     * The Segments don't cover all the CCD area; there is an edge along the CCD
     * sides. The parallel edges have the same dimension as do the edges along
     * the serial direction.
     * 
     * @param segmentGeometryConstants The SegmentGeometryConstants object
                                   describing each Segment geometry constants.
     * @param parallelEdge   The CCD edge along each parallel side in pixels
     * @param serialEdge     The CCD edge along each serial side in pixels
     * 
     */
    public CCDGeometryConstants(SegmentGeometryConstants segmentGeometryConstants, double parallelEdge, double serialEdge) {        
        this(segmentGeometryConstants, parallelEdge, serialEdge, SEGMENTS_ALONG_PARALLEL, SEGMENTS_ALONG_SERIAL);
    }    
    public CCDGeometryConstants(SegmentGeometryConstants segmentGeometryConstants, double parallelEdge, double serialEdge, int parallelSegments, int serialSegments) {        
        this.segmentGeometryConstants = segmentGeometryConstants;
        this.parallelEdge = parallelEdge;
        this.serialEdge = serialEdge;
        this.parallelSegments = parallelSegments;
        this.serialSegments = serialSegments;
    }    
    
    SegmentGeometryConstants getSegmentGeometryConstraint() {
        return segmentGeometryConstants;
    }

    /**
     * Get the Segment parallel active size in pixels.
     * This corresponds to Designator variable "dimv" in Table 8 of LCA-13501
     * 
     * @return The Segment parallel active size.
     */
    public int getSegmentParallelActiveSize() {
        return segmentGeometryConstants.getSegmentParallelActiveSize();
    }
    
    /**
     * Get the Segment serial active size in pixels.
     * This corresponds to Designator variable "dimh" in Table 8 of LCA-13501
     * 
     * @return The Segment serial active size.
     */
    public int getSegmentSerialActiveSize() {
        return segmentGeometryConstants.getSegmentSerialActiveSize();
    }
    
    /**
     * Get the Segment serial prescan size in pixels.
     * This corresponds to Designator variable "preh" in Table 12 of LCA-13501
     * 
     * @return The Segment serial active size.
     */
    public int getSegmentSerialPrescanSize() {
        return segmentGeometryConstants.getSegmentSerialPrescanSize();
    }
    
    
    /**
     * The CCD dimension, in pixels, of the active area along the parallel direction.
     * This is the segments parallel size multiplied by the number of segments
     * laid out along the parallel direction.
     * This corresponds to Designator variable "ccdax" in Table 8 of LCA-13501
     * 
     * @return The CCD dimension, in pixels, of the active area along the parallel direction.
     */
    public int getActiveAreaParallelSize() {
        return getSegmentParallelActiveSize()*parallelSegments;
    }
    
    /**
     * The CCD dimension, in pixels, of the active area along the serial direction.
     * This is the segments serial size multiplied by the number of segments
     * laid out along the serial direction.
     * This corresponds to Designator variable "ccday" in Table 8 of LCA-13501
     * 
     * @return The CCD dimension, in pixels, of the active area along the serial direction.
     */
    public int getActiveAreaSerialSize() {
        return getSegmentSerialActiveSize()*serialSegments;
    }
        
    /**
     * The CCD dimension, in pixels, of the physical area along the parallel direction.
     * This is the sum of the segments parallel size multiplied by the number of segments
     * laid out along the parallel direction and twice the CCD edge on the parallel direction.
     * This corresponds to Designator variable "ccdpx" in Table 8 of LCA-13501
     * 
     * @return The CCD dimension, in pixels, of the physical area along the parallel direction.
     */
    public int getPhysicalAreaParallelSize() {
        return getActiveAreaParallelSize() + (int)(2*parallelEdge);
    }
    
    /**
     * The CCD dimension, in pixels, of the physical area along the serial direction.
     * This is the sum of the segments serial size multiplied by the number of segments
     * laid out along the serial direction and twice the CCD edge on the serial direction.
     * This corresponds to Designator variable "ccdpy" in Table 8 of LCA-13501
     * 
     * @return The CCD dimension, in pixels, of the physical area along the serial direction.
     */
    public int getPhysicalAreaSerialSize() {
        return getActiveAreaSerialSize() + (int)(2*serialEdge);
    }
    
    double getParallelEdge() {
        return parallelEdge;
    }

    double getSerialEdge() {
        return serialEdge;
    }
    
}
