package org.lsst.ccs.utilities.readout;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import jline.internal.Log;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.FocalPlane;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Segment;
import org.lsst.ccs.utilities.image.FitsHeaderMetadataProvider;
import org.lsst.ccs.utilities.image.MetaDataSet;

/**
 * An implementation of FitsHeaderMetadataProvider based on a Geometry object.
 * It internally creates CCD level FitsHeaderMetadataProvider objects that will
 * be used when fits files are written.
 *
 * This is achieve by holding a reference to the Geometry object for which the
 * fits files are being written. The CCD information is extracted using the
 * ImageSet.getImageSetId() method that should return the unique CCD id within
 * the provided geometry.
 *
 * @author The LSST CCS Team
 */
public class WcsCoordinatesFitsHeaderMetadataProvider implements FitsHeaderMetadataProvider {

    private static final Map<CCD, Map<String,Map<String,Object>>> wcsCoord = new ConcurrentHashMap<>();
    private static final Logger LOG = Logger.getLogger(WcsCoordinatesFitsHeaderMetadataProvider.class.getName());

    private static final Pattern rangePattern = Pattern.compile(".*(\\[\\d+:\\d+(\\s*,\\s*)\\d+:\\d+\\])+.*");

    
    private final Map<String,Map<String,Object>> props;
    private final CCD ccd;
    
    public WcsCoordinatesFitsHeaderMetadataProvider(CCD ccd) {
        this.ccd = ccd;
        props = getWcsCoordinatesFor(ccd);
        if ( props == null ) {
            throw new RuntimeException("Could not find WCS coordinates for "+ccd.getUniqueId());
        }

    }

    @Override
    public MetaDataSet getPrimaryHeaderMetadata() {
        MetaDataSet primaryHeaderMetadata = new MetaDataSet();
        primaryHeaderMetadata.addMetaDataMap("imageSet", props.values().iterator().next());
        return primaryHeaderMetadata;
    }

    @Override
    public MetaDataSet getDataExtendedHeaderMetadata(String imageExtensionName) {
        Segment s = ccd.getSegmentByName(imageExtensionName);
        String segmentName = s != null ? s.getName() : imageExtensionName;
        MetaDataSet extendedHeaderMetadata = new MetaDataSet();
        extendedHeaderMetadata.addMetaDataMap("channel", props.get(segmentName));
        return extendedHeaderMetadata;
    }

    @Override
    public MetaDataSet getAdditionalExtendedHeaderMetadata(String extendedKeyword) {
        return null;
    }
            
    public static Map<String,Map<String,Object>> getWcsCoordinatesFor(CCD ccd) {
        synchronized (wcsCoord) {
            
            if (!wcsCoord.containsKey(ccd)) {

                //Load WCS Coordinates for this CCD.
                Raft r = (Raft) ccd.getParent();
                if (r == null) {
                    throw new RuntimeException("Cannot get WCS coordinates without the Raft information");
                }

                String ccdType = ccd.getType().getManufacturer().toLowerCase();
                String ccdFullName = r.getName() + "/" + ccd.getName();
                String fileName = "wcs/keywords_" + ccdType + "_" + r.getName() + ".wcs";
                LOG.log(Level.FINEST, "Opening wcs file {0} for CCD {0}.", new Object[]{fileName, ccdFullName});

                InputStream in = BootstrapResourceUtils.getBootstrapResource(fileName);
                if (in == null) {
                    throw new RuntimeException("Could not find wcs file " + fileName);
                }

                try {
                    Properties wcsFile = new Properties();
                    wcsFile.load(in);

                    String legend = wcsFile.getProperty("legend");
                    String[] wcsKeys = legend.split(",");

                    //To load the information for one CCD we have to load the information
                    //for the full raft.
                    for (CCD c : r.getChildrenList()) {
                        //We now extract the Segment's WCS coordinates
                        Map<String, Map<String,Object>> ccdWCSProviders = new HashMap<>();
                        for (Segment s : c.getSegments()) {
                            String fullName = r.getName() + "/" + c.getName();
                            String segmentKey = fullName + "/" + s.getName().replace("Seg", "");
                            String segmentValue = wcsFile.getProperty(segmentKey);

                            Map<String, Object> segmentWcsCoords = new HashMap<>();

                            if(segmentValue == null) {
                                throw new RuntimeException("Null value for keyword: "+segmentKey+" in file "+fileName);
                            }
                            
                            while (true) {
                                Matcher m = rangePattern.matcher(segmentValue);
                                if (m.matches()) {
                                    for (int i = 1; i < m.groupCount(); i++) {
                                        String group = m.group(i);
                                        String newGroup = group.replace(",", "!");
                                        segmentValue = segmentValue.replace(group, newGroup);
                                    }
                                } else {
                                    break;
                                }
                            }
                            String[] wcsValues = segmentValue.split(", ");

                            if (wcsValues.length != wcsKeys.length) {
                                throw new RuntimeException("Inconsistent number of WCS values and keys in the legend: " + wcsValues.length + " " + wcsKeys.length);
                            }

                            for (int i = 0; i < wcsValues.length; i++) {
                                String wcsKey = wcsKeys[i].trim();
                                String wcsValue = wcsValues[i].trim();
                                wcsValue = wcsValue.replace("!", ",");
                                segmentWcsCoords.put(wcsKey, wcsValue);
                            }
                            ccdWCSProviders.put(s.getName(),segmentWcsCoords);
                        }
                        wcsCoord.put(c, ccdWCSProviders);
                    }

                } catch (IOException ioe) {
                    throw new RuntimeException("Could not load " + fileName, ioe);
                }
            }
            return wcsCoord.get(ccd);
        }
    }

    

    public static void main(String[] argv) throws Exception {

        FocalPlane fp = FocalPlane.createFocalPlane();

        for (Raft r : fp.getRafts()) {
            for (CCD ccd : r.getChildrenList()) {
                WcsCoordinatesFitsHeaderMetadataProvider c = new WcsCoordinatesFitsHeaderMetadataProvider(ccd);
            }
        }

    }

}
