package org.lsst.ccs.utilities.ccd;

import java.awt.Dimension;
import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.utilities.ccd.CornerRaft.CornerRaftPosition;
import org.lsst.ccs.utilities.conv.InputConversionEngine;

/**
 * A Geometry class defining the Camera Focal Plane.
 * It contains a 5x5 array of Rafts. Not all of them have to be present as
 * there could be a partially filled focal plane.
 * 
 * This class needs a RaftGeometryConstants object to get the Raft Geometry
 * dimensions, both to determine the size of the FocalPlane and to place the
 * Raft children in the array.
 * 
 * Since the E2V and the ITL Rafts height and width are the same, we use 
 * chose to internally use the Raft geometry constants for an E2V CCD type.
 * We might want to revisit this choice if the above condition changes.
 * 
 * The constants are as defined in LCA-13501
 * 
 * @author The LSST CCS Team
 */
public class FocalPlane extends Geometry<Raft> {

    private static final int GAP_BETWEEN_RAFTS = 50;
    private static final int S_BAYS = 5, P_BAYS = 5;    
    private static final RaftGeometryConstants RAFT_GEOMETRY_CONSTANTS = RaftGeometryConstants.createRaftGeometryConstants(new E2VCCDType());

    /**
     * Build a Camera FocalPlane object.
     * This object has no Rafts. They need to be added.
     * 
     */
    public FocalPlane() {        
        super("FocalPlane", new Dimension(P_BAYS*RAFT_GEOMETRY_CONSTANTS.getRaftx()+(P_BAYS-1)*GAP_BETWEEN_RAFTS,S_BAYS*RAFT_GEOMETRY_CONSTANTS.getRafty()+(S_BAYS-1)*GAP_BETWEEN_RAFTS), P_BAYS , S_BAYS);
    }

    @Override
    public String getName() {
        return name;
    }
    
    public FocalPlane(String definition) {
        this();
        //an empty definition means a fully populated focal plane
        if (definition.isEmpty()) {
            for ( int p = 0; p < 5; p++) {
                for (int s = 0; s < 5; s++) {
                    Raft raft = isCornerRaft(p, s) ? CornerRaft.createCornerRaft(getCornerRaftPosition(p, s)) : Raft.createRaft("");
                    addChildGeometry(raft, p, s);
                }
            }
        } else {
            List<String> rafts = (List) InputConversionEngine.convertArgToType(definition, List.class);

            for (String raftStr : rafts) {
                raftStr = raftStr.trim();
                if (!raftStr.startsWith("R")) {
                    throw new IllegalArgumentException("Illegal raft definition: " + raftStr + "; it must start with the raft name in the form Rij");
                }
                try {
                    int parallelPosition = Integer.parseInt(raftStr.substring(1, 2));
                    int serialPosition = Integer.parseInt(raftStr.substring(2, 3));
                    String raftDefinition = raftStr.length() > 4 ? raftStr.substring(4) : "";
                    Raft raft = isCornerRaft(parallelPosition, serialPosition) ? CornerRaft.createCornerRaft(getCornerRaftPosition(parallelPosition, serialPosition)) : Raft.createRaft(raftDefinition.trim());
                    addChildGeometry(raft, parallelPosition, serialPosition);
                } catch (Exception e) {
                    throw new IllegalArgumentException("Could not create a Raft for definition " + raftStr, e);
                }
            }
        }
    }
    
    @Override
    protected void addGeometryToGrid(Raft child, int p, int s) {
        if ( isCornerRaft(p,s) ) {
            if ( ! (child instanceof CornerRaft) ) {
                throw new RuntimeException("Only CornerRafts can be added to the corners of the focal plane.");
            } else {
                CornerRaft cr = (CornerRaft)child;
                if ( cr.getCornerRaftPosition().getParallelPosition() != p || cr.getCornerRaftPosition().getSerialPosition() != s ) {
                    throw new RuntimeException("CornerRaft for position "+cr.getCornerRaftPosition()+" cannot be added to focal plane in bay ("+p+","+s+")");
                }
            }
        }
        int yCoord = s * (child.getHeight()+GAP_BETWEEN_RAFTS) ;
        int xCoord = p*  (child.getWidth()+ GAP_BETWEEN_RAFTS) ;
        addGeometry(child, xCoord, yCoord);
    }

    public List<Raft> getRafts() {
        return getChildrenList();
    }
    
    public List<Reb> getRebs() {
        List<Reb> rebs = new ArrayList<>();
        for ( Raft r : getRafts() ) {
            rebs.addAll(r.getRebs());
        }
        return rebs;
    }
    
    public Reb getReb(String fullName) {
        for ( Reb reb : getRebs() ) {
            if ( reb.getFullName().equals(fullName) ) {
                return reb;
            }
        }
        throw new RuntimeException("Could not find reb "+fullName+" among ("+getRebs()+")");
    }
    
    /**
     * Utility static method to create a FocalPlane for the
     * provided input String.
     * 
     * @param definition  The FocalPlane String representation.
     *                    Example of focal plane definition: "[R33:[ccdType:e2v,rebs:[Reb0:[ccdType:itl],Reb1,Reb2]]]"
     * @return      A FocalPlane.
     */
    public static FocalPlane createFocalPlane(String definition) {
        FocalPlane fp = new FocalPlane(definition);
        return fp;
    }
    public static FocalPlane createFocalPlane() {
        return createFocalPlane("");
    }
    
    private static boolean isCornerRaft(int p, int s) {
        return (p == 0 && (s == 0 || s == 4)) || (p == 4 && (s == 0 || s == 4));        
    }
    
    private static CornerRaftPosition getCornerRaftPosition(int p, int s) {
        for ( CornerRaftPosition position : CornerRaftPosition.values() ) {
            if (position.getParallelPosition() == p && position.getSerialPosition() == s) {
                return position;
            }
        }
        throw new IllegalArgumentException("Could not find a CornerRaftPosition corresponding to ("+p+","+s+")");
    }
    
    static boolean isCornerRaft(String raftStr) {
        int parallelPosition = Integer.parseInt(raftStr.substring(1,2));
        int serialPosition = Integer.parseInt(raftStr.substring(2,3));
        return isCornerRaft(parallelPosition, serialPosition);        
    }    
}
