package org.lsst.ccs.utilities.ccd;

/**
 * A class describing a CCD.
 * Its default constructor is designed to contain 2 Segments along the parallel
 * position and 8 in the serial direction.
 * 
 * The CCD dimensions and the Segments readout parameters are described in the
 * CCDType provided at construction time.
 * 
 * @author The LSST CCS Team.
 */
public class WavefrontCCD extends CCD {

    public static enum WavefrontType {
        WF0("SW0"), WF1("SW1");
        
        private final String name;
        
        WavefrontType(String name) {
            this.name = name;
        }
        
        public String getName() {
            return name;
        }
    }
    
    private final WavefrontType sensorType;

    /**
     * Create an empty CCD object. It takes the CCDType describing
     * all the geometry parameter for this CCD and the readout parameters for
     * the embedded Segments.
     * 
     * @param ccdType The CCDType for this CCD.
     */
    WavefrontCCD(CCDType ccdType, WavefrontType guiderType) {
        super(1,8,ccdType);
        this.sensorType = guiderType;
    }

    @Override
    public String getName() {
        return sensorType.getName();        
    }

    @Override
    public void addChildGeometry(Segment child, int p, int s) {
        super.addChildGeometry(child, p, s); //To change body of generated methods, choose Tools | Templates.
        child.setParallelPosition(p+1);
    }

    
    
    /**
     * Utility method to create a WavefrontCCD for a given CCDType filled with Segments.
     * 
     * @param ccdType The CCDType for this CCD
     * @param wavefrontType The WavefrontCCD for this CCD
     * @return        A fully Segment populated WavefrontCCD for the given CCDType.
     */
    public static WavefrontCCD createWavefrontCCD(CCDType ccdType, WavefrontCCD.WavefrontType wavefrontType) {

        WavefrontCCD ccd = new WavefrontCCD(ccdType, wavefrontType);
        int p = 0;
        for (int s = 0; s < ccd.getSerialChildrenCount(); s++) {
            int channel = s + 1;
            Segment segment = Segment.createCCDSegment(ccdType, channel);
            ccd.addChildGeometry(segment, p, s);
        }
        return ccd;
    }
    
    public WavefrontType getWavefrontType() {
        return sensorType;
    }
}
