package org.lsst.ccs.utilities.ccd;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.utilities.location.Location;

/**
 * A Reb object. A Science Reb contains 3 CCDs of the same type.
 *
 * @author The LSST CCS Team.
 */
public class Reb  {

    public enum RebType { 
        WAVEFRONT(Location.LocationType.WAVEFRONT), 
        GUIDER(Location.LocationType.GUIDER), 
        SCIENCE(Location.LocationType.SCIENCE);  
        
        private final Location.LocationType locationType;
        
        RebType(Location.LocationType locationType) {
            this.locationType = locationType;
        }
        
        public Location.LocationType getLocationType() {
            return locationType;
        }
    }

    private final String name;
    private final List<CCD> ccds = new ArrayList<>();
    private final Raft parentRaft;
    private final RebType rebType;
    private CCDType ccdType;
    private boolean isAuxtelREB;
    
    /** 
     * Create an instance of a Reb.
     * 
     */
    Reb(RebType rebType, String name, Raft raft) {
        this.rebType = rebType;
        this.name = name;
        this.parentRaft = raft;
    }

    void addCCD(CCD ccd) {
        if ( ccdType == null ) {
            ccdType = ccd.getType();
        } else {
            if ( ! ccdType.equals(ccd.getType()) ) {
                throw new IllegalArgumentException("Cannot add CCD "+ccd.getName()+" due to inconsistent type. Expected type: "+ccdType.getName()+" while CCD type is "+ccd.getType().getName());
            }            
        }
        ccds.add(ccd);
        ccd.setParentReb(this);
    }

    
    public String getUniqueId() {
        return parentRaft != null ? parentRaft.getUniqueId()+"."+name : name;
    }

    public List<CCD> getCCDs() {
        return ccds;
    }
    
    public Raft getRaft() {
        return parentRaft;
    }
    
    public String getName() {
        return name;
    }

    public RebType getRebType() {
        return rebType;
    }
    
    public CCDType getCCDType() {
        return ccdType;
    }
    
    public String getFullName() {
        return parentRaft != null ? parentRaft.getName()+"/"+name : name;
    }

    /**
     * Get the Location of the REB in the focal-plane. 
     * @return The location.
     */
    public Location getLocation() {
        return Location.of(getFullName());
    }
    
    @Override
    public String toString() {
        return getFullName();
    }
    
    public void setCCDType(CCDType type) {
        if (rebType != RebType.SCIENCE) {
            throw new RuntimeException("Reb type can only be changed for SCIENCE rebs "+rebType);
        }
        ccdType = type;
        for (CCD ccd: ccds) {
            ccd.setCcdType(type);
        }
    }
    
    // Only for AuxTel
    void setIsAuxtelREB(boolean b) {
        isAuxtelREB = b;
    }

    public boolean isAuxtelREB() {
        return isAuxtelREB;
    }
}
