package org.lsst.ccs.imagenaming;

import java.io.Serializable;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A class representing an image name in the standard format.
 * @author tonyj
 */
public class ImageName implements Serializable, Comparable<ImageName> {

    private final static Pattern NAME_PATTERN = Pattern.compile("(\\w\\w)(\\d*)_(\\w)_(\\d{8})_(\\d+)");
    static final DateTimeFormatter FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd");
    private static final long serialVersionUID = 8767898207051188380L;
    private final Source source;
    private final Controller controller;
    private final LocalDate date;
    private final int number;
    private final int sourceIndex;

    /**
     * Create an image name from a string in the standard image name format.
     *
     * @param name The string to parse
     * @throws IllegalArgumentException if the string is not a valid image name
     */
    public ImageName(String name) {
        if (name == null) {
            throw new IllegalArgumentException("Invalid null image name ");
        }
        Matcher matcher = NAME_PATTERN.matcher(name);
        if (!matcher.matches()) {
            throw new IllegalArgumentException("Invalid image name " + name);
        }
        source = Source.fromCode(matcher.group(1));
        controller = Controller.fromCode(matcher.group(3));
        number = Integer.parseInt(matcher.group(5));
        if (number <= 0) {
            throw new IllegalArgumentException("Image number must be > 0");
        } 
        sourceIndex = matcher.group(2).length() > 0 ? Integer.parseInt(matcher.group(2)) : 0;
        checkSourceIndex(sourceIndex, source);
        try {
            date = LocalDate.parse(matcher.group(4), FORMATTER);
        } catch (DateTimeParseException x) {
            throw new IllegalArgumentException("Invalid date field", x);
        }
    }

    ImageName(Source source, Controller controller, LocalDate date, int imageNumber) {
        this(source, 0, controller, date, imageNumber);
    }

    ImageName(Source source, int sourceIndex, Controller controller, LocalDate date, int imageNumber) {
        checkSourceIndex(sourceIndex, source);
        this.source = source;
        this.controller = controller;
        this.date = date;
        this.number = imageNumber;
        this.sourceIndex = sourceIndex;
    }

    private void checkSourceIndex(int sourceIndex, Source source) throws IllegalArgumentException {
        if (sourceIndex > 0 && !source.isIndexed()) {
            throw new IllegalArgumentException("Source index not allowed with source="+source);
        } else if (sourceIndex < 0 || sourceIndex > 9999) {
            throw new IllegalArgumentException("Invalid source index: " + sourceIndex);
        }
    }
    
    /**
     * Get the source of the image
     * @return The source
     */
    public Source getSource() {
        return source;
    }

    /**
     * Get the source index (or 0 if no source index)
     * @return The source index (or 0 if no source index)
     */
    public int getSourceIndex() {
        return sourceIndex;
    }
    
    public String getSourceWithIndex() {
        return getSourceWithIndex(source, sourceIndex);
    }

    static String getSourceWithIndex(Source source, int sourceIndex) {
        return sourceIndex == 0 ? source.getCode() : source.getCode()+sourceIndex;
    }
    
    /**
     * Get the controller that initiated the image
     * @return The controller
     */
    public Controller getController() {
        return controller;
    }

    /**
     * Get the data of the image. Note that this is the date allocated
     * using LSST naming conventions, and not necessarily the actual date
     * on which the image was taken.
     * @return The date
     */
    public LocalDate getDate() {
        return date;
    }
    
    /**
     * Get the formatted date of the image
     * @return The date formatted in the standard way
     */
    public String getDateString() {
        return FORMATTER.format(date);
    }

    /**
     * Get the image sequence number
     * @return The number
     */
    public int getNumber() {
        return number;
    }

    /**
     * Get the formatted image number
     * @return The sequence number formatted in the standard way
     */
    public String getNumberString() {
        return String.format("%06d", number);
    }
    
    @Override
    public String toString() {
        return String.format("%s_%s_%s_%06d", getSourceWithIndex(), controller.getCode(), FORMATTER.format(date), number);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 97 * hash + Objects.hashCode(this.source);
        hash = 97 * hash + Objects.hashCode(this.controller);
        hash = 97 * hash + Objects.hashCode(this.date);
        hash = 97 * hash + this.number;
        hash = 97 * hash + this.sourceIndex;
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImageName other = (ImageName) obj;
        if (this.number != other.number) {
            return false;
        }
        if (this.source != other.source) {
            return false;
        }
        if (this.sourceIndex != other.sourceIndex) {
            return false;
        }
        if (this.controller != other.controller) {
            return false;
        }
        return Objects.equals(this.date, other.date);
    }

    @Override
    public int compareTo(ImageName other) {
        // Make the date and sequence number the most significant in terms of sort image names
        int c = this.date.compareTo(other.date);
        if (c != 0) return c;
        c = Integer.compare(this.number, other.number);
        if (c != 0) return c;
        c = this.source.compareTo(other.source);
        if (c != 0) return c;
        c = Integer.compare(this.sourceIndex, other.sourceIndex);
        if (c != 0) return c;        
        return this.controller.compareTo(other.controller);
    }
}
