package org.lsst.ccs.utilities.image.samp;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.astrogrid.samp.Client;
import org.astrogrid.samp.Message;
import org.astrogrid.samp.Metadata;
import org.astrogrid.samp.Response;
import org.astrogrid.samp.client.ClientProfile;
import org.astrogrid.samp.client.DefaultClientProfile;
import org.astrogrid.samp.client.HubConnector;
import org.astrogrid.samp.client.ResultHandler;
import org.astrogrid.samp.client.SampException;
import org.astrogrid.samp.hub.Hub;
import org.astrogrid.samp.hub.HubServiceMode;

/**
 * Utilities for communicating with DS9. This code is ds9 specific, it is not
 * easy to do the same thing perfectly generically, although support for other
 * clients could presumably be added.
 *
 * @author tonyj
 */
public class SampUtils implements Closeable {

    private final HubConnector conn;
    private final static Logger log = Logger.getLogger(SampUtils.class.getName());
    private static final int DEFAULT_TIMEOUT = 10000;

    /**
     * Create an instance of SampUtils and attempt to connect to a SAMP hub.
     * 
     * @param name The name of the subsystem (to be displayed in SAMP hub).
     * @param autoStartHub If <code>true</code> attempt to start a hub if one is
     * not already running.
     */
    public SampUtils(String name, boolean autoStartHub) {
        // Construct a connector
        ClientProfile profile = DefaultClientProfile.getProfile();
        conn = new HubConnector(profile);
        // Configure it with metadata about this application
        Metadata meta = new Metadata();
        meta.setName(name == null ? "CCS" : name);
        meta.setDescriptionText("CCS interface to SAMP for subsystem " + name);
        conn.declareMetadata(meta);
        conn.declareSubscriptions(conn.computeSubscriptions());
        if (!conn.isConnected()) {
            if (autoStartHub) {
                try {
                    Hub.checkExternalHubAvailability();
                    Hub.runExternalHub(HubServiceMode.NO_GUI);                                    
                    log.info("SAMP hub spawned.");
                    // Wait to see if we are able to connect to hub 
                    long start = System.currentTimeMillis();
                    long timeout = 10000; 
                    while (conn.getConnection() == null) {
                        if (System.currentTimeMillis()-start>timeout) throw new IOException("timeout while waiting to connect to recently started SAMP hub");
                        log.log(Level.INFO,"Waiting for samp hub to start");
                        Thread.sleep(100);
                    }
                } catch (InterruptedException | IOException ex) {
                    log.log(Level.WARNING,"Unable to create SAMP hub",ex);
                }
            } else {
                log.warning("Unable to connect to SAMP hub.");
            }

        }
        conn.setAutoconnect(10);
    }

    /**
     * Ask ds9 to display a fits file, assuming that the fits file is directly
     * readable by ds9 and is mosaic iraf format . This code is ds9 specific, it
     * is not easy to do the same thing perfectly generically.
     *
     * @param file The file to be displayed.
     * @throws org.astrogrid.samp.client.SampException If an error occurs
     */
    public void display(File file) throws SampException {
        URI uri = file.toURI();
        ds9Set("mosaicimage iraf", uri, DEFAULT_TIMEOUT);
        // TODO: Probably this should be configurable
        ds9Set("scale scope global", null, DEFAULT_TIMEOUT);
        ds9Set("zoom to fit", null, DEFAULT_TIMEOUT);
    }

    /**
     * Get a list of versions of DS9 clients connected to hub.
     *
     * @return A List of the version numbers of DS9 clients connected to the
     * hub, or an empty list if there are none.
     * @throws SampException If an errors occurs
     */
    public List<String> getDS9Version() throws SampException {
        List<Object> versions = ds9Get("version", DEFAULT_TIMEOUT);
        return versions.stream().map(Object::toString)
                .filter(item -> item.startsWith("ds9 "))
                .map(item -> item.substring(4))
                .collect(Collectors.toList());
    }

    /**
     * Send a ds9.set command over SAMP. Currently this method just broadcasts
     * the DS9 set command, and does not wait for a response, or check for
     * errors (other than logging them).
     *
     * @param command The command to be sent to DS9.
     * @param uri The optional URI to be sent, or <code>null</code>
     * @param timeout The timeout (in milliseconds?)
     * @throws SampException If an error occurs sending the message (for example
     * no hub running).
     */
    public void ds9Set(String command, URI uri, int timeout) throws SampException {
        Message message = new Message("ds9.set");
        message.addParam("cmd", command);
        if (uri != null) {
            message.addParam("url", uri.toString());
        }
        conn.callAll(message, new ResultHandler() {
            @Override
            public void result(Client client, Response rspns) {
                if (rspns.isOK()) {
                    log.log(Level.FINE, "SAMP result: {0}: {1}", new Object[]{client, rspns});
                } else {
                    log.log(Level.WARNING, "SAMP Error: {0}", rspns.getErrInfo().getErrortxt());
                }
            }

            @Override
            public void done() {
                log.log(Level.FINE, "SAMP done");
            }
        }, timeout);
    }

    /**
     * Send a ds9.get command over SAMP. This command waits for responses and
     * throws an exception if an error is received.
     *
     * @param command The command to send.
     * @param timeout The timeout (in milliseconds?)
     * @return The list of responses, one for each client that responds. This
     * may be an empty list if no clients respond.
     * @throws SampException If an error occurs.
     */
    public List<Object> ds9Get(String command, int timeout) throws SampException {
        CompletableFuture<List<Object>> future = new CompletableFuture<>();
        Message message = new Message("ds9.get");
        message.addParam("cmd", command);
        conn.callAll(message, new ResultHandler() {
            List<Object> result = new ArrayList<>();

            @Override
            public void result(Client client, Response rspns) {
                log.log(Level.FINE, "SAMP result: {0}: {1}", new Object[]{client, rspns});
                if (rspns.isOK()) {
                    result.add(rspns.getResult().get("value").toString());
                } else {
                    SampException x = new SampException("SAMP Error:" + rspns.getErrInfo().getErrortxt());
                    future.completeExceptionally(x);
                }
            }

            @Override
            public void done() {
                log.log(Level.FINE, "SAMP done");
                future.complete(result);
            }
        }, timeout);

        try {
            return future.get();
        } catch (InterruptedException ex) {
            throw new SampException("Exception during execution of SAMP command", ex);
        } catch (ExecutionException ex) {
            Throwable x = ex.getCause();
            if (x instanceof SampException) {
                throw (SampException) x;
            } else {
                throw new SampException("Unexpected error", x);
            }
        }
    }

    /**
     * Just for testing
     *
     * @param args
     * @throws SampException
     */
    public static void main(String[] args) throws SampException {
        SampUtils utils = new SampUtils("test",true);
        System.out.printf("version=%s\n", utils.getDS9Version());
        File file = new File("/tmp/image4829142949481327900fits");
        utils.display(file);
    }

    @Override
    public void close() throws IOException {
        conn.setActive(false);
    }
}
