package org.lsst.ccs.utilities.image;

import java.time.Instant;
import java.util.Date;
import nom.tam.fits.FitsDate;
import nom.tam.fits.FitsException;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Utilities for dealing with Dates/Times.
 *
 * @author tonyj
 */
public class DateUtils {

    private DateUtils() {
    }

    /***
     * Note we use Instance as the return type here, to preserve precision 
     * @param temporalObject
     * @return 
     */
    private static Instant convertTemporalToInstant(Object temporalObject) {
        if (temporalObject instanceof Date date) {
            return Instant.ofEpochMilli(date.getTime());
        } else if (temporalObject instanceof Instant instant) {
            return instant;
        } else if (temporalObject instanceof CCSTimeStamp ccsTimeStamp) {
            return ccsTimeStamp.getTAIInstant();
        } else {
            throw new IllegalArgumentException("Unsupported temporal class: " + temporalObject);
        }
    }

    public static double convertTemporalToMJD(Object temporalObject) {
        return convertInstantToMJD(convertTemporalToInstant(temporalObject));
    }

    public static String convertTemporalToString(Object temporalObject) {
        return convertInstantToString(convertTemporalToInstant(temporalObject));
    }

    /**
     * Convert an Instant to a modified Julian date (MJD). This method preserves
     * the full precision in the instant
     *
     * @param instant The instant to convert
     * @return The date formatted as a modified Julian date
     */
    public static double convertInstantToMJD(Instant instant) {
        long epochSecond = instant.getEpochSecond();
        int nano = instant.getNano();
        return  (epochSecond / 86400.0 + nano / 86400.0 / 1000000000.0) + 2440587.5 - 2_400_000.5;
    }

    /**
     * Convert an instant to a string following FITS standards. FITS dates
     * are limited to millisecond precision.
     * @param instant
     * @return 
     */
    static String convertInstantToString(Instant instant) {
        // Don't use Date.from due to truncation issue (LSSTCCS-2576)
        long epochSeconds = instant.getEpochSecond();
        int nanos = instant.getNano();
        long epochMillisRounded = epochSeconds * 1000 + Math.round(nanos/1_000_000.0);
        return convertDateToString(new Date(epochMillisRounded));
    }
    
    /**
     * Convert date to a string following FITS standards.
     *
     * @param date The input date
     * @return The formatted date/time
     */
    static String convertDateToString(Date date) {
        return FitsDate.getFitsDateString(date);
    }

    /**
     * Convert string to a date following FITS standards.
     *
     * @param string The string to convert
     * @return The converted Date
     * @throws FitsException If the string is not in the correct format.
     */
    public static Date convertStringToDate(String string) throws FitsException {
        return new FitsDate(string).toDate();
    }
}
