package org.lsst.ccs.utilities.ccd;

import java.awt.Dimension;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.utilities.ccd.Reb.RebType;
import org.lsst.ccs.utilities.conv.InputConversionEngine;

/**
 * A Raft Geometry object.
 * In accordance to LCA-13501 a Raft geometry is composed of three Reb
 * geometries. A Raft object is built from the information contained in the
 * provided RaftGeometryConstants object. These constants vary depending on
 * which CCD type they correspond to.
 * 
 * We assume here that a Raft contains only the same type of CCD.
 * If we will support heterogeneous CCD types in a raft we will have to
 * figure out how to place them in the addGeometryToGrid method.
 * 
 * @author The LSST CCS Team
 */
public class Raft extends Geometry<CCD> {

    private final RaftGeometryConstants raftConstants;
    private CCDType ccdType;
        
    protected final List<Reb> rebList = new ArrayList<>();

    /**
     * Create a Raft object for a given type of CCD.
     * The resulting Raft is empty, having no Rebs installed.
     * 
     * @param type The type of CCD in this Raft.
     */
    public Raft(CCDType type) {
        this(RaftGeometryConstants.createRaftGeometryConstants(type), type);
    }
    
    @SuppressWarnings("OverridableMethodCallInConstructor")
    Raft(RaftGeometryConstants raftConstants, CCDType type) {
        super("R", new Dimension(raftConstants.getRaftx(), raftConstants.getRafty()), 3, 3);
        this.raftConstants = raftConstants;
        this.ccdType = type;
        populateRebList();
    }
    
    protected void populateRebList() {
        rebList.add(new Reb(RebType.SCIENCE, "Reb0", this));
        rebList.add(new Reb(RebType.SCIENCE, "Reb1", this));
        rebList.add(new Reb(RebType.SCIENCE, "Reb2", this));
    }
    
    public List<Reb> getRebs() {
        return rebList;
    }
    
    public void setCCDType(CCDType type) {
        for (Reb reb : rebList) {
            reb.setCCDType(type);
        }
    }
    

    @Override
    protected void addGeometryToGrid(CCD child, int p, int s) {
        int xCoord = (2-p) * child.getWidth() + (2-p)*raftConstants.getGap_inx() + (int)raftConstants.getGap_outx();
        int yCoord = (2 - s) * (child.getHeight() + raftConstants.getGap_iny())+(int)raftConstants.getGap_outy();
        addGeometry(child, xCoord, yCoord);
        addCCDtoReb(child, p, s);
    }

    
    protected void addCCDtoReb(CCD child, int p, int s) {
        rebList.get(p).addCCD(child);
    }

    /**
     * Get the RaftGeometryConstants object for this Raft.
     * 
     * @return The Raft geometry constants.
     */
    public RaftGeometryConstants getRaftGeometryConstants() {
        return raftConstants;
    }
    
    /**
     * Utility static method to create a fully Reb/CCD populated Raft for the
     * provided CCD type.
     * 
     * @param type  The type of the CCDs in the Raft.
     * @return      A fully populated Raft.
     */
    public static Raft createRaft(CCDType type) {
        Raft raft = new Raft(type);
        //Add three Rebs, right to left.
        for (int i = 2; i >= 0; i--) {
            //Add three CCDs left to right
            for (int j = 0; j < 3; j++) {
                CCD ccd = CCD.createCCD(type);
                raft.addChildGeometry(ccd, i, j);
            }
        }
        return raft;
    }
    
    /**
     * Utility static method to create a fully Reb/CCD populated Raft for the
     * provided input String.
     * 
     * @param definition  The Raft String representation.
     *                    Example of rafts definition: "[ccdType:e2v,rebs:[Reb0:[ccdType:itl], Reb1, Reb2]]"
     * @return      A fully populated Raft.
     */
    public static Raft createRaft(String definition) {
        Map<String,String> props = (Map) InputConversionEngine.convertArgToType(definition, Map.class);
        String ccdTypeStr = props.getOrDefault("ccdType", "e2v").trim();
        CCDType ccdType = CCDType.getCCDType(ccdTypeStr);
        return createRaft(ccdType);
    }

    /**
     * Get the CCDType for this Raft.
     * 
     * @return The CCDType for this Raft.
     */
    public CCDType getType() {
        return ccdType;
    }
    
    
}
