package org.lsst.ccs.utilities.location;

import java.io.Serializable;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Corresponds to a REB location in the focal plane e.g.&nbsp;R22/Reb0.
 *
 * @author tonyj
 */
public class Location implements Comparable<Location>, Serializable {

    private static final long serialVersionUID = 9188620871451731827L;

    // Note: This is chosen for compatibility with DAQ::LocationSet
    private static final int BAY_MULTIPLIER = 4;
    private final static int INVALID_INDEX = -1;
    private final static int[] BAY_TO_INDEX = new int[]{
        0, 1, 2, 3, 4, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX,
        5, 6, 7, 8, 9, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX,
        10, 11, 12, 13, 14, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX,
        15, 16, 17, 18, 19, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX,
        20, 21, 22, 23, 24, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX, INVALID_INDEX};
    private final static int[] INDEX_TO_BAY = new int[]{
        0, 1, 2, 3, 4, 10, 11, 12, 13, 14, 20, 21, 22, 23, 24, 30, 31, 32, 33, 34, 40, 41, 42, 43, 44};

    private final static Pattern PATTERN = Pattern.compile("R(\\d\\d)/Reb(.)");

    public enum LocationType {
        /**
         * A WREB source.
         */
        WAVEFRONT(1),
        /**
         * A GREB source
         */
        GUIDER(2),
        /**
         * A Science REB source
         */
        SCIENCE(3);
        private final int CCDCount;

        LocationType(int CCDCount) {
            this.CCDCount = CCDCount;
        }

        /**
         * The number of CCDs associated with the source type.
         *
         * @return
         */
        public int getCCDCount() {
            return CCDCount;
        }

        String getBoardName(int board) {
            switch (this) {
                case SCIENCE:
                    return String.format("Reb%d", board);
                case WAVEFRONT:
                    return String.format("RebW");
                case GUIDER:
                    return String.format("RebG");
                default:
                    throw new RuntimeException("Unknown source type: " + this);
            }
        }

        String getSensorName(int board, int sensor) {
            switch (this) {
                case SCIENCE:
                    return String.format("S%d%d", board, sensor);
                case WAVEFRONT:
                    return String.format("SW%d", sensor);
                case GUIDER:
                    return String.format("SG%d", sensor);
                default:
                    throw new RuntimeException("Unknown source type: " + this);
            }
        }
    }

    private final byte bay;
    private final byte board;

    /**
     * Create a location, based on the DAQ index.
     *
     * @param index The DAQ location index.
     */
    public Location(int index) {
        this((byte) INDEX_TO_BAY[index / BAY_MULTIPLIER], (byte) (index % BAY_MULTIPLIER));
    }

    /**
     * Create location from bay number and board number
     *
     * @param bay Bay (e.g.\ 22, 10, 04 etc)
     * @param board Board (0-2)
     */
    public Location(int bay, int board) {
        if (bay < 0 || bay >= BAY_TO_INDEX.length || BAY_TO_INDEX[bay] == INVALID_INDEX) {
            throw new IllegalArgumentException("Invalid bay: " + bay);
        }
        if (board < 0 || board > (isCornerRaft(bay) ? 1 : 2)) {
            throw new IllegalArgumentException("Invalid board: " + board + " for bay " + bay);
        }
        this.bay = (byte) bay;
        this.board = (byte) board;
    }

    /**
     * Create location from string
     *
     * @param location String representation of location, of form Rnn/Rebm
     * @return The corresponding location
     */
    public static Location of(String location) {
        Matcher matcher = PATTERN.matcher(location);
        if (!matcher.matches()) {
            throw new IllegalArgumentException("Illegal location: " + location);
        }
        int bay = Integer.parseInt(matcher.group(1));
        int board;
        String boardName = matcher.group(2);
        switch (boardName) {
            case "W":
                board = 0;
                break;
            case "G":
                board = 1;
                break;
            default:
                board = Integer.parseInt(boardName);
                break;
        }
        return new Location(bay, board);
    }

    /**
     * The source type for this location
     *
     * @return The source type
     */
    public LocationType type() {
        if (isCornerRaft(bay)) {
            if (board == 0) {
                return LocationType.WAVEFRONT;
            } else {
                return LocationType.GUIDER;
            }
        } else {
            return LocationType.SCIENCE;
        }
    }

    private static boolean isCornerRaft(int bay) {
        return bay == 0 || bay == 40 || bay == 44 || bay == 04;
    }

    /**
     * Get the index for this location, as used by the DAQ
     *
     * @return The location index.
     */
    public int index() {
        return BAY_MULTIPLIER * BAY_TO_INDEX[bay] + board;
    }

    public byte getBay() {
        return bay;
    }

    public byte getBoard() {
        return board;
    }

    @Override
    public String toString() {
        return String.format("%s/%s", getRaftName(), getBoardName());
    }

    public String getRaftName() {
        return String.format("R%02d", bay);
    }

    public String getBoardName() {
        return type().getBoardName(board);
    }

    public String getSensorName(int index) {
        return type().getSensorName(board, index);
    }

    public SensorLocationSet getSensors() {
        SensorLocationSet result = new SensorLocationSet();
        Location.LocationType type = type();
        int count = type == Location.LocationType.WAVEFRONT ? 2 : type.getCCDCount();
        for (int index = 0; index < count; index++) {
            result.add(SensorLocation.of(getRaftName() + getSensorName(index)));
        }
        return result;
    }

    @Override
    public int compareTo(Location o) {
        return this.index() - o.index();
    }

    @Override
    public int hashCode() {
        return index();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        return this.index() == ((Location) obj).index();
    }
}
