package org.lsst.ccs.utilities.image;

/**
 * An interface to provide MetaDataSets to fill the content of Fits Headers.
 * According to the syntax used in header specification (spec) files the meta-data
 * is in the form of a MetaDataSet, that is a Map of Maps: the key names for 
 * the maps are defined as "mapName" and are used to determine at runtime the 
 * value of a fits keyword defined in the spec file as ${"mapName"."mapKey"}. 
 * The fits keyword is assigned the value corresponding to key "mapKey" in the 
 * map stored with key "mapName".
 * 
 * If "mapName" is not specified (i.e. ${"mapKey"}) then all the maps are searched
 * and the first one found containing key "mapKey" will yield the value.
 * 
 * This interface is designed to be able to provide all the meta-data needed to 
 * write a single fits file. 
 * 
 * Since a single fit file is meant to be written for
 * each CCD, the primary header meta-data should contain information for the CCD
 * itself, the Reb used to read it, the Raft in which it is embedded etc.
 * The extended data header should be written for each of the CCD segments, containing
 * information for the specific segment.
 * 
 * @author The LSST CCS Team
 */
public interface FitsHeaderMetadataProvider {
    
    /**
     * Get the primary Header MetaDataSet for a given ImageSet
     *      * 
     * @return The primary header meta-data for provided ImageSet.
     */
     MetaDataSet getPrimaryHeaderMetadata();
    
    /**
     * Get the extended header meta-data by index.
     * These are the headers containing the actual images. The index starts from 0.
     * 
     * @param imageExtensionName The image extension name to which this data is to be written.
     * 
     * @return The extended header meta-data by index
     */
    MetaDataSet getDataExtendedHeaderMetadata(String imageExtensionName);
    
    /**
     * Get the metadata for an additional extended Headers by key. The information
     * in these extended headers is stored as BinaryTables. These are all the
     * headers defined in the spec files with keywords other than "primary" and
     * "extended"; this keyword will be used in this method to identify which
     * metadata map to return.
     * 
     * @param extendedKeyword The keyword identifying the extended header.
     * 
     * @return The metadata for the requested extended header
     */
    MetaDataSet getAdditionalExtendedHeaderMetadata(String extendedKeyword);
}
