package org.lsst.ccs.utilities.readout;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.ccd.E2VCCDType;
import org.lsst.ccs.utilities.ccd.ITLCCDType;

/**
 *
 * @author tonyj
 */
public class ReadOutParametersOld extends ReadOutParameters {

    private static final Logger LOG = Logger.getLogger(ReadOutParametersOld.class.getName());
    private static final String 
            PARM_READ_ROWS = "ReadRows",
            PARM_READ_COLS = "ReadCols",
            PARM_PRE_ROWS = "PreRows",
            PARM_PRE_COLS = "PreCols",
            PARM_POST_ROWS = "PostRows",
            PARM_POST_COLS = "PostCols",
            PARM_READ_COLS2 = "ReadCols2",
            PARM_OVER_ROWS = "OverRows",
            PARM_OVER_COLS = "OverCols";

    public static final String[] DEFAULT_NAMES = {
        PARM_READ_ROWS, PARM_READ_COLS, PARM_PRE_ROWS, PARM_PRE_COLS,
        PARM_POST_ROWS, PARM_POST_COLS, PARM_READ_COLS2, PARM_OVER_ROWS,
        PARM_OVER_COLS};

    private final int preRows;
    private final int readRows;
    private final int postRows;
    private final int overRows;
    private final int preCols;
    private final int readCols;
    private final int postCols;
    private final int overCols;
    private final int readCols2;

    ReadOutParametersOld(CCDType ccdType, int[] registers, ReadOutRegisterMap registerMap) {
        super(ccdType, registers, registerMap);
        this.preRows = getValue(PARM_PRE_ROWS);
        this.readRows = getValue(PARM_READ_ROWS);
        this.postRows = getValue(PARM_POST_ROWS);
        this.overRows = getValue(PARM_OVER_ROWS);
        this.preCols = getValue(PARM_PRE_COLS);
        this.readCols = getValue(PARM_READ_COLS);
        this.postCols = getValue(PARM_POST_COLS);
        this.overCols = getValue(PARM_OVER_COLS);
        this.readCols2 = getValue(PARM_READ_COLS2);
        validate();
    }

    ReadOutParametersOld(CCDType ccdType) {
        this(ccdType, fabricateValues(ccdType), fabricateNames());
    }

    private static ReadOutRegisterMap fabricateNames() {
        return new ReadOutRegisterMap(DEFAULT_NAMES);
    }

    private static int[] fabricateValues(CCDType type) {
        return new int[]{0, type.getCCDGeometryConstants().getSegmentParallelActiveSize(), 0, getParallelOverscan(type),
            0, type.getCCDGeometryConstants().getSegmentSerialPrescanSize() + type.getCCDGeometryConstants().getSegmentSerialActiveSize(), 0, 0, getSerialOverscan(type)};
    }

    private static int getParallelOverscan(CCDType type) {
        if (type instanceof E2VCCDType) {
            return 46;
        } else if (type instanceof ITLCCDType) {
            return 48;
        } else {
            return 0;
        }
    }

    private static int getSerialOverscan(CCDType type) {
        if (type instanceof E2VCCDType) {
            return 22;
        } else if (type instanceof ITLCCDType) {
            return 32;
        } else {
            return 0;
        }
    }

    @Override
    public int getSerialReadPixels() {
        return readCols + readCols2 + overCols;
    }

    @Override
    public int getParallelReadPixels() {
        return readRows + overRows;
    }

    @Override
    final void validate() {
        boolean isValid = true;
        CCDType ccdType = getCCDType();

        int totalParallelPixels = ccdType.getCCDGeometryConstants().getSegmentParallelActiveSize();
        if (preRows + readRows + postRows != totalParallelPixels) {
            LOG.log(Level.WARNING, "Parallel parameters are invalid: preRows({0}) + readRows({1}) + postRows({2}) != segRows({3})", new Object[]{preRows, readRows, postRows, totalParallelPixels});
            isValid = false;
        }
        int totalSerialPixels = ccdType.getCCDGeometryConstants().getSegmentSerialActiveSize() + ccdType.getCCDGeometryConstants().getSegmentSerialPrescanSize();
        if (preCols + readCols + postCols + readCols2 != totalSerialPixels) {
            LOG.log(Level.WARNING, "Serial parameters are invalid: preCols({0}) + readCols({1}) + readCols2({2}) + postCols({3}) != serCols({4})", new Object[]{preCols, readCols, readCols2, postCols, totalSerialPixels});
            isValid = false;
        }
        if (!isValid) {
            LOG.warning("The geometry values in the Fits Headers will be incorrect.");
        }
    }

    @Override
    public int getTotalParallelSize() {
        return getCCDType().getCCDGeometryConstants().getSegmentParallelActiveSize()+overRows;
    }

    @Override
    public int getSerialPrescan() {
        return getCCDType().getCCDGeometryConstants().getSegmentSerialPrescanSize();
    }

    @Override
    public int getTotalSerialSize() {
        return preCols+readCols+postCols+readCols2+overCols;
    }

}
