package org.lsst.ccs.utilities.location;

import java.io.Serializable;
import java.util.AbstractSet;
import java.util.BitSet;
import java.util.Iterator;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.utilities.location.Location.LocationType;

/**
 * Default implementation of Set&lt;Location&gt;. This implementation uses a
 * bitmap to represent the selected locations.
 *
 * @author tonyj
 */
public class LocationSet extends AbstractSet<Location> implements Serializable {

    private static final long serialVersionUID = -4332892191028786738L;
    private static final long[] ALL_BITS = {0x7777777777737773L, 0x377737777L};
    private final BitSet locations;

    private final static Pattern RAFT_PATTERN = Pattern.compile("(-?)R(\\d\\d)");

    /**
     * Creates an empty location set
     */
    public LocationSet() {
        locations = new BitSet();
    }

    /**
     * Creates a LocationSet from a comma delimited String.
     *
     * @see Location#of(java.lang.String)
     * @param locations The locations
     */
    public LocationSet(String locations) {
        this();
        if (locations != null && !locations.trim().isEmpty()) {
            locationsFromCommaDelimitedString(locations, this.locations);
        }
    }

    /**
     * Creates a location set which is a copy of the provided set
     *
     * @param input The location set to copy
     */
    public LocationSet(Set<Location> input) {
        this();
        if (input instanceof LocationSet) {
            this.locations.or(((LocationSet) input).locations);
        } else if (input != null) {
            input.forEach((l) -> {
                locations.set(l.index());
            });
        }
    }

    /**
     * LocationSet for all locations.
     *
     * @return A LocationSet containing all valid locations
     */
    public static LocationSet all() {
        return new LocationSet(BitSet.valueOf(ALL_BITS));
    }

    /**
     * Creates a LocationSet from a list of strings. Also works for a single
     * comma delimited String.
     *
     * @see Location#of(java.lang.String)
     * @param location The location(s)
     * @return The LocationSet
     */
    public static LocationSet of(String... location) {
        BitSet locations = new BitSet();
        for (String l : location) {
            locationsFromCommaDelimitedString(l, locations);
        }
        return new LocationSet(locations);
    }

    private static void locationsFromCommaDelimitedString(String l, BitSet bitSet) throws NumberFormatException {
        if (l.trim().isEmpty()) return;
        String[] ll = l.split(",");
        for (String lll : ll) {
            lll = lll.trim();
            Matcher matcher = RAFT_PATTERN.matcher(lll);
            if (matcher.matches()) {
                boolean negative = "-".matches(matcher.group(1));
                int raft = Integer.parseInt(matcher.group(2));
                Location start = new Location(raft, 0);
                if (negative) {
                    if (bitSet.isEmpty()) {
                        bitSet.or(BitSet.valueOf(ALL_BITS));
                    }
                    for (int i = 0; i < (start.type() == LocationType.SCIENCE ? 3 : 2); i++) {
                        bitSet.clear(start.index() + i);
                    }
                } else {
                    for (int i = 0; i < (start.type() == LocationType.SCIENCE ? 3 : 2); i++) {
                        bitSet.set(start.index() + i);
                    }
                }
            } else { // Pass along to Location to see if it can parse it
                boolean negative = lll.startsWith("-");
                if (negative) {
                    if (bitSet.isEmpty()) {
                        bitSet.or(BitSet.valueOf(ALL_BITS));
                    }
                    bitSet.clear(Location.of(lll.substring(1)).index());
                } else {
                    bitSet.set(Location.of(lll).index());
                }
            }
        }
    }

    /**
     * Create a location set based on DAQ location index based BitSet.
     *
     * @param bitset
     */
    public LocationSet(BitSet bitset) {
        this();
        locations.or(bitset);
    }

    /**
     * Return the DAQ location index based BitSet
     *
     * @return
     */
    public BitSet getBitSet() {
        BitSet result = new BitSet();
        result.or(locations);
        return result;
    }

    @Override
    public boolean add(Location location) {
        final int index = location.index();
        boolean wasSet = locations.get(index);
        locations.set(index);
        return wasSet;
    }

    @Override
    public void clear() {
        locations.clear();
    }

    @Override
    public boolean remove(Object location) {
        if (location instanceof Location) {
            final int index = ((Location) location).index();
            boolean wasSet = locations.get(index);
            locations.clear(index);
            return wasSet;
        } else {
            return false;
        }
    }

    boolean isSet(int i) {
        return locations.get(i);
    }

    @Override
    public Iterator<Location> iterator() {
        return new Iterator<Location>() {
            private int current = -1;

            @Override
            public boolean hasNext() {
                return locations.nextSetBit(current + 1) >= 0;
            }

            @Override
            public Location next() {
                current = locations.nextSetBit(current + 1);
                return new Location(current);
            }

            @Override
            public void remove() {
                locations.clear(current);
            }
        };
    }

    @Override
    public int size() {
        return locations.cardinality();
    }

    /**
     * Returns location set as list of integers using Owen's numbering scheme.
     *
     * @return An int array
     */
    public int[] asIntArray() {
        int[] result = new int[locations.cardinality()];
        int i = 0;
        for (Location l : this) {
            result[i++] = l.getBay() * 4 + l.getBoard();
        }
        return result;
    }
}
