package org.lsst.ccs.utilities.image;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.lang.reflect.Array;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.AsynchronousFileChannel;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import nom.tam.util.ArrayDataOutput;
import nom.tam.util.AsciiFuncs;
import nom.tam.util.RandomAccess;

/**
 * An asynchronous implementation of ArrayDataOuput and Random access, as a replacement
 * for BufferedFile provided by no.tam.fits. The implementation only supports writing
 * files, not reading. This implementation is not thread-safe, since it maintains a 
 * current position within the file.
 * @author tonyj
 */
class AsyncBufferedFile implements ArrayDataOutput, RandomAccess {

    private static final int BUFFER_SIZE = 32768;
    private final AsynchronousFileChannel channel;
    private ByteBuffer currentBuffer;
    private long currentBufferPosition;
    private final List<BufferRecord> records = new ArrayList<>();
    
    private static class BufferRecord {
        private final long position;
        private final long size;
        private final Future<Integer> future;

        public BufferRecord(long position, long size, Future<Integer> future) {
            this.position = position;
            this.size = size;
            this.future = future;
        }

        private void waitForCompletion() throws IOException {
            try {
                future.get();
            } catch (InterruptedException ex) {
                throw new InterruptedIOException("Interrupt during IO");
            } catch (ExecutionException ex) {
                Throwable cause = ex.getCause();
                if (cause instanceof IOException) {
                    throw (IOException) cause;
                } else {
                    throw new IOException("Exception during FITS asynch IO", cause);
                }
            } 
        }

        private boolean overlaps(long position, int limit) {
            return position+limit >= this.position && this.position+this.size < position;
        }
    }

    AsyncBufferedFile(AsynchronousFileChannel channel) {
        this.channel = channel;
        currentBufferPosition = 0;
        currentBuffer = ByteBuffer.allocateDirect(BUFFER_SIZE);
        currentBuffer.order(ByteOrder.BIG_ENDIAN);
    }

    private void commitRecord(long currentBufferPosition, ByteBuffer currentBuffer) throws IOException {
       for (BufferRecord r: records) {
          if (r.overlaps(currentBufferPosition, currentBuffer.limit())) {
              r.waitForCompletion();
          }
       }
       records.add(new BufferRecord(currentBufferPosition, currentBuffer.limit(), channel.write(currentBuffer, currentBufferPosition)));
    }

    @Override
    public void flush() throws IOException {
        flush(BUFFER_SIZE);
    }
    
    private void flush(int newBufferSize) throws IOException {
        if (currentBuffer.position() > 0) {
            long newBufferPosition = currentBufferPosition + currentBuffer.position();
            currentBuffer.flip();
            commitRecord(currentBufferPosition, currentBuffer);
            if (newBufferSize > 0) {
                currentBufferPosition = newBufferPosition;
                currentBuffer = ByteBuffer.allocateDirect(newBufferSize);
                currentBuffer.order(ByteOrder.BIG_ENDIAN);
            }
        } else if (newBufferSize > currentBuffer.capacity()) {
            currentBuffer = ByteBuffer.allocateDirect(newBufferSize);
            currentBuffer.order(ByteOrder.BIG_ENDIAN);            
        }       
    }
    
    private void needBytes(int bytes) throws IOException {
        if (currentBuffer.remaining() < bytes) {
            flush(Math.max(BUFFER_SIZE, bytes));
        }
    }

    @Override
    public void write(boolean[] data) throws IOException {
        write(data, 0, data.length);
    }

    @Override
    public void write(boolean[] data, int offset, int length) throws IOException {
        for (int i = offset; i<offset+length; i++) {
            writeBoolean(data[i]);
        }
    }

    @Override
    public void write(char[] data) throws IOException {
        write(data, 0, data.length);
    }

    @Override
    public void write(char[] data, int offset, int length) throws IOException {
        for (int i = offset; i<offset+length; i++) {
            writeChar(data[i]);
        }
    }

    @Override
    public void write(double[] data) throws IOException {
        write(data, 0, data.length);
    }

    @Override
    public void write(double[] data, int offset, int length) throws IOException {
        for (int i = offset; i<offset+length; i++) {
            writeDouble(data[i]);
        }
    }

    @Override
    public void write(float[] data) throws IOException {
        write(data, 0, data.length);
    }

    @Override
    public void write(float[] data, int offset, int length) throws IOException {
        for (int i = offset; i<offset+length; i++) {
            writeFloat(data[i]);
        }
    }

    @Override
    public void write(int[] data) throws IOException {
        write(data, 0, data.length);
    }

    @Override
    public void write(int[] data, int offset, int length) throws IOException {
        for (int i = offset; i<offset+length; i++) {
            writeInt(data[i]);
        }
    }

    @Override
    public void write(long[] data) throws IOException {
        write(data, 0, data.length);
    }

    @Override
    public void write(long[] data, int offset, int length) throws IOException {
        for (int i = offset; i<offset+length; i++) {
            writeLong(data[i]);
        }
    }

    @Override
    public void write(short[] data) throws IOException {
        write(data, 0, data.length);
    }

    @Override
    public void write(short[] data, int offset, int length) throws IOException {
        for (int i = offset; i<offset+length; i++) {
            writeShort(data[i]);
        }
    }

    @Override
    public void write(String[] data) throws IOException {
        write(data, 0, data.length);
    }

    @Override
    public void write(String[] data, int offset, int length) throws IOException {
        for (int i = offset; i < offset + length; i++) {
            write(AsciiFuncs.getBytes(data[i]), 0, data[i].length());
        }
    }

    @Override
    public void writeArray(Object data) throws IOException {
        if (!data.getClass().isArray()) {
            throw new IOException("Invalid object passed to BufferedDataOutputStream.write" + data.getClass().getName());
        }
        int length = Array.getLength(data);
        // Is this a multidimensional array? If so process recursiv
        if (data.getClass().getComponentType().isArray()) {
            for (int i = 0; i < length; i++) {
                writeArray(Array.get(data, i));
            }
        } else {
            if (data instanceof boolean[]) {
                write((boolean[]) data, 0, length);
            } else if (data instanceof byte[]) {
                write((byte[]) data, 0, length);
            } else if (data instanceof char[]) {
                write((char[]) data, 0, length);
            } else if (data instanceof short[]) {
                write((short[]) data, 0, length);
            } else if (data instanceof int[]) {
                write((int[]) data, 0, length);
            } else if (data instanceof long[]) {
                write((long[]) data, 0, length);
            } else if (data instanceof float[]) {
                write((float[]) data, 0, length);
            } else if (data instanceof double[]) {
                write((double[]) data, 0, length);
            } else if (data instanceof String[]) {
                write((String[]) data, 0, length);
            } else {
                for (int i = 0; i < length; i++) {
                    writeArray(Array.get(data, i));
                }
            }
        }
    }

    @Override
    public void write(int data) throws IOException {
        needBytes(4);
        currentBuffer.putInt(data);
    }

    @Override
    public void write(byte[] data) throws IOException {
        write(data, 0, data.length);
    }

    @Override
    public void write(byte[] data, int offset, int length) throws IOException {
        needBytes(length);
        currentBuffer.put(data, offset, length);
    }

    @Override
    public void writeBoolean(boolean data) throws IOException {
        needBytes(1);
        currentBuffer.put(data ? (byte) 1 : (byte) 0);
    }

    @Override
    public void writeByte(int data) throws IOException {
        needBytes(1);
        currentBuffer.put((byte) data);
    }

    @Override
    public void writeShort(int data) throws IOException {
        needBytes(2);
        currentBuffer.putShort((short)data);
    }

    @Override
    public void writeChar(int data) throws IOException {
        needBytes(2);
        currentBuffer.put((byte) (data >>> 8));
        currentBuffer.put((byte) data);
    }

    @Override
    public void writeInt(int data) throws IOException {
        needBytes(4);
        currentBuffer.putInt(data);
    }

    @Override
    public void writeLong(long data) throws IOException {
        needBytes(8);
        currentBuffer.putLong(data);
    }

    @Override
    public void writeFloat(float data) throws IOException {
        needBytes(4);
        currentBuffer.putFloat(data);
    }

    @Override
    public void writeDouble(double data) throws IOException {
        needBytes(8);
        currentBuffer.putDouble(data);
    }

    @Override
    public void writeBytes(String data) throws IOException {
        write(AsciiFuncs.getBytes(data), 0, data.length());
    }

    @Override
    public void writeChars(String data) throws IOException {
        int len = data.length();
        for (int i = 0; i < len; i++) {
            writeChar(data.charAt(i));
        }
    }

    @Override
    public void writeUTF(String data) throws IOException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void close() throws IOException {
        flush(0);
        // Wait for all of the futures to complete
        for (BufferRecord r : records) {
            r.waitForCompletion();
        }
    }

    @Override
    public long getFilePointer() {
        return currentBufferPosition + currentBuffer.position();
    }

    @Override
    public void seek(long position) throws IOException {
        if (position > currentBufferPosition && position - currentBufferPosition < currentBuffer.capacity()) {
            currentBuffer.position((int) (position - currentBufferPosition));
        } else {
            flush(0);
            currentBufferPosition = position;
            currentBuffer = ByteBuffer.allocateDirect(BUFFER_SIZE);
            currentBuffer.order(ByteOrder.BIG_ENDIAN);
        }
    }

    @Override
    public void mark(int arg0) throws IOException {
        throw new UnsupportedOperationException("Not supported yet.");
    }
    
    @Override
    public void reset() throws IOException {
        throw new UnsupportedOperationException("Not supported yet."); 
    }

    @Override
    public long skip(long bytes) throws IOException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.        
    }

    @Override
    public void skipAllBytes(long arg0) throws IOException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void skipAllBytes(int arg0) throws IOException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }


    @Override
    public int read(byte[] arg0) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(byte[] arg0, int arg1, int arg2) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(boolean[] arg0) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(boolean[] arg0, int arg1, int arg2) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(char[] arg0) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(char[] arg0, int arg1, int arg2) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(double[] arg0) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(double[] arg0, int arg1, int arg2) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(float[] arg0) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(float[] arg0, int arg1, int arg2) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(int[] arg0) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(int[] arg0, int arg1, int arg2) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(long[] arg0) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(long[] arg0, int arg1, int arg2) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(short[] arg0) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int read(short[] arg0, int arg1, int arg2) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int readArray(Object arg0) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public long readLArray(Object arg0) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public void readFully(byte[] arg0, int arg1, int arg2) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public void readFully(byte[] arg0) throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int skipBytes(int arg0) throws IOException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public boolean readBoolean() throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public byte readByte() throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int readUnsignedByte() throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public short readShort() throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int readUnsignedShort() throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public char readChar() throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public int readInt() throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public long readLong() throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public float readFloat() throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public double readDouble() throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public String readLine() throws IOException {
        throw new UnsupportedOperationException("write only file");
    }

    @Override
    public String readUTF() throws IOException {
        throw new UnsupportedOperationException("write only file");
    }
}
