package org.lsst.ccs.utilities.ccd;

/**
 * A class describing a CCD.
 * Its default constructor is designed to contain 2 Segments along the parallel
 * position and 8 in the serial direction.
 * 
 * The CCD dimensions and the Segments readout parameters are described in the
 * CCDType provided at construction time.
 * 
 * @author The LSST CCS Team.
 */
public class GuiderCCD extends CCD {

    public static enum GuiderType {
        GUIDER0 ("SG0"), GUIDER1("SG1");
        
        private final String name;
        
        GuiderType(String name) {
            this.name = name;
        }
        
        public String getName() {
            return name;
        }
    }
    
    private final GuiderType guiderType;

    /**
     * Create an empty CCD object. It takes the CCDType describing
     * all the geometry parameter for this CCD and the readout parameters for
     * the embedded Segments.
     * 
     * @param ccdType The CCDType for this CCD.
     */
    GuiderCCD(CCDType ccdType, GuiderType guiderType) {
        super(ccdType);
        this.guiderType = guiderType;
    }

    @Override
    public String getName() {
        return guiderType.getName();        
    }

    /**
     * Utility method to create a GuiderCCD for a given CCDType filled with Segments.
     * 
     * @param ccdType The CCDType for this CCD
     * @param guiderType The GuiderType for this CCD
     * @return        A fully Segment populated CCD for the given CCDType.
     */
    public static GuiderCCD createGuiderCCD(CCDType ccdType, GuiderType guiderType) {

        GuiderCCD ccd = new GuiderCCD(ccdType, guiderType);
        for (int s = 0; s < ccd.getSerialChildrenCount(); s++) {
            for (int p = 0; p < ccd.getParallelChildrenCount(); p++) {
                int channel = p == 0 ? 16 - s : 1 + s;
                Segment segment = Segment.createCCDSegment(ccdType, channel);
                ccd.addChildGeometry(segment, p, s);
            }
        }
        return ccd;
    }
    
    public GuiderType getGuiderType() {
        return guiderType;
    }

}
