package org.lsst.ccs.utilities.image;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.channels.CompletionHandler;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;
import nom.tam.fits.BasicHDU;
import nom.tam.fits.FitsException;
import nom.tam.fits.FitsFactory;
import nom.tam.fits.FitsUtil;
import nom.tam.fits.Header;
import nom.tam.fits.HeaderCard;
import nom.tam.fits.HeaderCardException;
import org.lsst.ccs.utilities.ccd.image.data.RawImageData.BitsPerPixel;
import static org.lsst.ccs.utilities.image.HeaderSpecification.DataType.Float;
import org.lsst.ccs.utilities.image.HeaderSpecification.HeaderLine;

/**
 * A utility for writing FITS files following LSST conventions.
 *
 * @author tonyj
 */
public class FitsFileWriter implements Closeable {

    private class ImageExtension implements Closeable {

        private final long startHeader;
        private final long endPosition;
        private long currentPosition;
        private final FitsCheckSum.Checksum checkSum;
        private final Header header;

        private ImageExtension(long startHeader, long startData, long imageSize, Header header) throws IOException {
            this.startHeader = startHeader;
            this.currentPosition = startData;
            this.endPosition = startData + imageSize;
            this.header = header;
            this.checkSum = new FitsCheckSum.Checksum();
        }

        private synchronized Future<Integer> write(ByteBuffer src) throws IOException {
            int length = src.remaining();
            if (length + currentPosition > endPosition) {
                throw new FitsIOException(file, "Error writing file: %s, data received (%,d bytes) > expected length (%,d bytes)", length + currentPosition, endPosition);
            }
            // TODO: Also update the checksum asynchronously?
            FitsCheckSum.updateChecksum(src, checkSum);
            Future<Integer> future = asynchChannel.write(src, currentPosition);
            currentPosition += length;
            return future;
        }

        private synchronized <A> void write(ByteBuffer src, A attachment, CompletionHandler<Integer, ? super A> handler) throws IOException {
            int length = src.remaining();
            if (length + currentPosition > endPosition) {
                throw new FitsIOException(file, "Error writing file: %s, data received (%,d bytes) > expected length (%,d bytes)", length + currentPosition, endPosition);
            }
            // TODO: Also update the checksum asynchronously?
            FitsCheckSum.updateChecksum(src, checkSum);
            asynchChannel.write(src, currentPosition, attachment, handler);
            currentPosition += length;
        }

        //Invoked before closing the file.
        private synchronized void updateDataSum() throws IOException {
            try {
                FitsCheckSum.updateDataSum(header, checkSum.getCheckSum());
                synchronized(abf) {
                    abf.seek(startHeader);
                    header.write(abf);
                }
            } catch (FitsException ex) {
                throw new FitsIOException(file, ex, "Unable to add datasum to header");
            }
        }

        @Override
        public void close() throws IOException {
            updateDataSum();
        }
    }
    
    private final AsyncBufferedFile abf;
    private final AsynchronousFileChannel asynchChannel;
    private final Map<String, ImageExtension> imageExtensions;
    private final File file;
    private static final Logger LOG = Logger.getLogger(FitsFileWriter.class.getName());

    private void initializeFitsFileWriter(ImageSet images, MetaDataSet metaDataSet, Map<String, HeaderSpecification> config, List<FitsHeaderMetadataProvider> providers, BitsPerPixel bits) throws IOException, FitsException {

        List<FitsHeaderMetadataProvider> inner_providers = new ArrayList<>();
        if (providers != null) {
            inner_providers.addAll(providers);
        }

        FitsHeaderMetadataProvider inner_provider = new CompositeFitsHeaderMetadataProvider(inner_providers);

        int[][] intDummyData = new int[1][1];
        short[][] shortDummyData = new short[1][1];
        Object[] tableDummyData = new Object[0];
        Object dummyData = bits == BitsPerPixel.BIT16 ? shortDummyData : intDummyData;

        FitsFactory.setUseHierarch(true);

        MetaDataSet fullMetaData = inner_provider.getPrimaryHeaderMetadata();
        if (metaDataSet != null) {
            fullMetaData.addMetaDataSet(metaDataSet);
        }

        // Create primary header
        BasicHDU primary = BasicHDU.getDummyHDU();
        addMetaDataToHeader(primary, "primary", fullMetaData, config);
        FitsCheckSum.setChecksum(primary);
        primary.getHeader().write(abf);

        int serialPixels = images.getSerialPixels();
        int parallelPixels = images.getParallelPixels();

        //Create extension image headers for each image, and reserve space for data
        for (String imageExtensionName : images.getImageExtensionNames()) {
            BasicHDU hdu = FitsFactory.hduFactory(dummyData);

            MetaDataSet extendedMetadata = inner_provider.getDataExtendedHeaderMetadata(imageExtensionName);
            if (metaDataSet != null) {
                extendedMetadata.addMetaDataSet(metaDataSet);
            }

            extendedMetadata.addMetaData("extension", "NAXIS", 2);
            extendedMetadata.addMetaData("extension", "NAXIS1", images.getSerialPixels());
            extendedMetadata.addMetaData("extension", "NAXIS2", images.getParallelPixels());

            addMetaDataToHeader(hdu, "extended", extendedMetadata, config);
            if (bits == BitsPerPixel.BIT16) {
                // To store as unsigned 16 bit values, we have to set BZERO to 32768, and 
                // subtract 32768 from each value. 
                // See: http://heasarc.gsfc.nasa.gov/docs/software/fitsio/c/c_user/node23.html
                hdu.addValue("BSCALE", 1.0, "Unsigned 16 bit data");
                hdu.addValue("BZERO", 32768, "Unsigned 16 bit data");
            } else {
                hdu.addValue("BSCALE", 1.0, "");
                hdu.addValue("BZERO", 0.0, "");
            }
            Header header = hdu.getHeader();
            header.setXtension("IMAGE");
            FitsCheckSum.setChecksum(hdu);
            long startHeader = abf.getFilePointer();
            header.write(abf);
            long startData = abf.getFilePointer();
            long imageSize = bits.bytes() * serialPixels * parallelPixels;
            abf.seek(abf.getFilePointer() + imageSize);
            FitsUtil.pad(abf, imageSize);
            imageExtensions.put(imageExtensionName, new ImageExtension(startHeader, startData, imageSize, header));
        }

        // If necessary, create any additional extended HDU's here.
        // Create any extra BinTables from the specification
        FitsFactory.setUseAsciiTables(false);
        for (String key : config.keySet()) {
            if (!"primary".equals(key) && !"extended".equals(key)) {
                BasicHDU binary = FitsFactory.hduFactory(tableDummyData);

                MetaDataSet additionalMetaData = inner_provider.getAdditionalExtendedHeaderMetadata(key);
                if (metaDataSet != null) {
                    additionalMetaData.addMetaDataSet(metaDataSet);
                }
                addMetaDataToHeader(binary, key, additionalMetaData, config);
                Header header = binary.getHeader();
                header.setXtension("BINTABLE");
                FitsCheckSum.setChecksum(binary);
                header.write(abf);
            }
        }
    }

    /**
     * Construct a FitsFileWriter object for the provided ImageSet. By default
     * this will write 32 bit images.
     *
     * @param file The name of the output fits file
     * @param images The ImageSet of images to write in the Data extended
     * headers. Note that this specifies the images to write, not the actual
     * data for the images.
     * @throws IOException
     * @throws FitsException
     */
    @Deprecated
    public FitsFileWriter(File file, ImageSet images) throws IOException, FitsException {
        this(file, images, null, null, BitsPerPixel.BIT32);
    }

    public FitsFileWriter(File file, ImageSet images, Map<String, HeaderSpecification> specs) throws IOException, FitsException {
        this(file, images, null, null, BitsPerPixel.BIT32, specs);
    }

    @Deprecated
    public FitsFileWriter(File file, ImageSet images, MetaDataSet metaDataSet) throws IOException, FitsException {
        this(file, images, metaDataSet, null, BitsPerPixel.BIT32);
    }

    public FitsFileWriter(File file, ImageSet images, Map<String, HeaderSpecification> specs, MetaDataSet metaDataSet) throws IOException, FitsException {
        this(file, images, metaDataSet, null, BitsPerPixel.BIT32, specs);
    }

    public FitsFileWriter(File file, ImageSet images, MetaDataSet metaDataSet, List<FitsHeaderMetadataProvider> providers) throws IOException, FitsException {
        this(file, images, metaDataSet, providers, BitsPerPixel.BIT32);
    }

    @Deprecated
    public FitsFileWriter(File file, ImageSet images, List<FitsHeaderMetadataProvider> providers) throws IOException, FitsException {
        this(file, images, null, providers, BitsPerPixel.BIT32);
    }

    public FitsFileWriter(File file, ImageSet images, Map<String, HeaderSpecification> specs, List<FitsHeaderMetadataProvider> providers) throws IOException, FitsException {
        this(file, images, null, providers, BitsPerPixel.BIT32, specs);
    }

    @Deprecated
    public FitsFileWriter(File file, ImageSet images, MetaDataSet metaDataSet, List<FitsHeaderMetadataProvider> providers, BitsPerPixel bits) throws IOException, FitsException {
        this(file, images, metaDataSet, providers, bits,  FitsHeadersSpecifications.getHeaderSpecifications());
    }

    /**
     * Construct a FitsFileWriter object for the provided ImageSet.
     *
     * @param file The name of the output fits file
     * @param images The ImageSet of images to write in the Data extended
     * headers. Note that this specifies the images to write, not the actual
     * data for the images.
     * @param metaDataSet The meta-data maps to use to extract header info from
     * @param providers
     * @param bits The number of bits per pixel for images
     * @param specs The spec files used to create FITS file headers
     * @throws IOException
     * @throws FitsException
     */
    public FitsFileWriter(File file, ImageSet images, MetaDataSet metaDataSet, List<FitsHeaderMetadataProvider> providers, BitsPerPixel bits, Map<String, HeaderSpecification> specs) throws IOException, FitsException {
        Logger log = Logger.getLogger("nom.tam.fits");
        log.setLevel(Level.WARNING);
        this.file = file;
        asynchChannel = AsynchronousFileChannel.open(file.toPath(), StandardOpenOption.WRITE, StandardOpenOption.CREATE);
        abf = new AsyncBufferedFile(asynchChannel);
        imageExtensions = new HashMap<>();
        synchronized (abf) {
            initializeFitsFileWriter(images, metaDataSet, specs, providers, bits);
        }
    }
    
    /**
     * Construct a FitsFileWriter but do not create the HDUs. You must call
     * createHDUs before attempting to write to the file.
     * @param file The file to write
     * @throws java.io.IOException
     */
    public FitsFileWriter(File file) throws IOException {
        this.file = file;
        long start = System.currentTimeMillis();
        asynchChannel = AsynchronousFileChannel.open(file.toPath(), StandardOpenOption.WRITE, StandardOpenOption.CREATE);
        LOG.log(Level.FINE, () -> String.format("Open of %s took %,dms", file, System.currentTimeMillis() - start));
        abf = new AsyncBufferedFile(asynchChannel);
        imageExtensions = new HashMap<>();
    }
    
    /**
     * Complete initialization of a file created with the single argument constructor
     * @param images The ImageSet of images to write in the Data extended
     * headers. Note that this specifies the images to write, not the actual
     * data for the images.
     * @param metaDataSet The meta-data maps to use to extract header info from
     * @param providers
     * @param bits The number of bits per pixel for images
     * @param specs The spec files used to create FITS file headers
     * @throws IOException
     * @throws FitsException 
     */
    public void createHDUs(ImageSet images, MetaDataSet metaDataSet, List<FitsHeaderMetadataProvider> providers, BitsPerPixel bits, Map<String, HeaderSpecification> specs) throws IOException, FitsException {
        synchronized (abf) {
            initializeFitsFileWriter(images, metaDataSet, specs, providers, bits);
        }        
    }
    

    /**
     * Write the actual image data to the file. It is not necessary that all of
     * the data for the image be available at once, this method will write
     * whatever data is currently available in the byte buffer to the specified
     * image, and will keep track of how much has been written to each image to
     * allow more data to be written later. This method assumes the data is
     * given in the order it is to be written to the file. If any data
     * reordering is needed it needs to be done before calling this method.
     *
     * @param imageExtensionName The image extension name to which this data is
     * to be written
     * @param src The image
     * @return The number of bytes written
     * @throws IOException If an IOException is generated, or if more data is
     * sent than was expected for a particular image.
     */
    public int write(String imageExtensionName, ByteBuffer src) throws IOException {
        Future<Integer> write = getImageExtensionForName(imageExtensionName).write(src);
        try {
            return write.get();
        } catch (ExecutionException x) {
            Throwable cause = x.getCause();
            if (cause instanceof IOException) {
                throw (IOException) cause;
            } else {
                throw new FitsIOException(file, x, "Exception while writing FITS file");
            }
        } catch (InterruptedException x) {
            throw new InterruptedIOException("Interupt while writing FITS file");
        }
    }

    /**
     * Write a buffer to an image extension asynchronously
     *
     * @param imageExtensionName The image extension name to which this data is
     * to be written
     * @param src The buffer to write
     * @return A future which will eventually return the number of bytes written
     * @throws IOException If the operation cannot be queued.
     */
    public Future<Integer> asyncWrite(String imageExtensionName, ByteBuffer src) throws IOException {
        return getImageExtensionForName(imageExtensionName).write(src);
    }

    /**
     * Write a buffer to an image extension asynchronously
     *
     * @param <A> The type of the attachment
     * @param imageExtensionName The image extension name to which this data is
     * to be written
     * @param src The buffer to write
     * @param attachment The object to attach to the I/O operation; can be null
     * @param handler The handler for consuming the result
     * @throws IOException If the operation cannot be queued.
     */
    public <A> void asyncWrite(String imageExtensionName, ByteBuffer src, A attachment, CompletionHandler<Integer, ? super A> handler) throws IOException {
        getImageExtensionForName(imageExtensionName).write(src, attachment, handler);
    }

    private ImageExtension getImageExtensionForName(String imageExtensionName) {
        ImageExtension result = imageExtensions.get(imageExtensionName);
        if (result == null) {
            throw new IllegalArgumentException("Invalid image xtension name: " + imageExtensionName);
        }
        return result;
    }

    @Override
    public void close() throws IOException {
        // Update the datasum keywords in place
        for (ImageExtension imageExtension : imageExtensions.values()) {
            imageExtension.close();
        }
        abf.close();
        asynchChannel.close();
    }

    private static double formatDouble(double d) {
        String dStr = String.format("%5g", d);
        return Double.valueOf(dStr);
    }
    
    private void addMetaDataToHeader(BasicHDU hdu, String specName, MetaDataSet metaDataSet, Map<String, HeaderSpecification> config) throws HeaderCardException, IOException {
        LOG.log(Level.FINE, "Writing out fits header {0} from header list: {1}",new Object[]{specName,config.keySet()});
        HeaderSpecification spec = config.get(specName);
        if (spec == null) {
            throw new FitsIOException(file, "Missing specification for header: %s", specName);
        }
        for (HeaderLine header : spec.getHeaders()) {
            Object value = header.getValue(metaDataSet);
            if (header.isRequired() && value == null) {
                throw new FitsIOException(file, "Could not find value for required header keyword: %s defined as: \n%s", header.getKeyword(), header.getHeaderDefinition());
            }
            try {
                if (value != null) {
                    if (value.equals(HeaderSpecification.NULL)) {
                        hdu.getHeader().addLine( new HeaderCard(header.getKeyword(), null, header.getComment(), true));
                    } else {
                        switch (header.getDataType()) {
                            case Integer:
                                hdu.addValue(header.getKeyword(), ((Number) value).intValue(), header.getComment());
                                break;
                            case Float:
                                double data = ((Number) value).doubleValue();
                                if (!Double.isFinite(data)) {
                                    throw new IllegalArgumentException("Can not store non-finite floating point in FITS file");
                                }
                                //The following code is to reduce the number of significant digits to 5
                                //See https://jira.slac.stanford.edu/browse/LSSTCCS-2168
                                hdu.addValue(header.getKeyword(), formatDouble(data), header.getComment());
                                break;
                            case Boolean:
                                hdu.addValue(header.getKeyword(), (Boolean) value, header.getComment());
                                break;
                            case Date:
                                hdu.addValue(header.getKeyword(), DateUtils.convertDateToString(value), header.getComment());
                                break;
                            case MJD:
                                hdu.addValue(header.getKeyword(), DateUtils.convertDateToMJD(value), header.getComment());
                                break;
                            default:
                                hdu.addValue(header.getKeyword(), String.valueOf(value), header.getComment());
                        }
                    }
                }
            } catch (ClassCastException x) {
                throw new FitsIOException(file, x, "Meta-data header %s with value %s(%s) cannot be converted to type %s", header.getKeyword(), value, value.getClass(), header.getDataType());
            } catch (IllegalArgumentException | HeaderCardException x) {
                throw new FitsIOException(file, x, "Meta-data header %s with value %s(%s) cannot be written", header.getKeyword(), value, value.getClass());
            }
        }
    }

    public static class FitsIOException extends IOException {

        FitsIOException(File file, Throwable cause, String message, Object... args) {
            super("Error writing " + file + " : " + String.format(message, args), cause);
        }

        FitsIOException(File file, String message, Object... args) {
            this(file, null, message, args);
        }
    }

}
