package org.lsst.ccs.utilities.image;

import java.time.Instant;
import java.util.Date;
import nom.tam.fits.FitsDate;
import nom.tam.fits.FitsException;

/**
 * Utilities for dealing with Dates/Times.
 *
 * @author tonyj
 */
public class DateUtils {

    private DateUtils() {
    }

    private static Date convertTemporalToDate(Object temporalObject) {
        if (temporalObject instanceof Date) {
            return (Date) temporalObject;
        } else if (temporalObject instanceof Instant) {
            return Date.from((Instant) temporalObject);
        } else {
            throw new IllegalArgumentException("Unsupported temporal class: " + temporalObject);
        }
    }

    static double convertDateToMJD(Object temporalObject) {
        return convertDateToMJD(convertTemporalToDate(temporalObject));
    }

    static String convertDateToString(Object temporalObject) {
        return convertDateToString(convertTemporalToDate(temporalObject));
    }

    /**
     * Convert a java.util.Date to a modified Julian date (MJD).
     *
     * @param date The date to convert
     * @return The date formatted as a modified Julian date
     */
    static double convertDateToMJD(Date date) {
        return date.getTime() / 1000.0 / 86400.0 + 2440587.5 - 2_400_000.5;
    }

    /**
     * Convert date to a string following FITS standards.
     *
     * @param date The input date
     * @return The formatted date/time
     */
    static String convertDateToString(Date date) {
        return FitsDate.getFitsDateString(date);
    }

    /**
     * Convert string to a date following FITS standards.
     *
     * @param string The string to convert
     * @return The converted Date
     * @throws FitsException If the string is not in the correct format.
     */
    static Date convertStringToDate(String string) throws FitsException {
        return new FitsDate(string).toDate();
    }
}
